﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using Curse.Extensions;
using Curse.AddOns;
using Curse.Logging;
using ICSharpCode.SharpZipLib.BZip2;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Json;
using Curse.AddOnService.Extensions;

namespace Curse.AddOnService
{
    public class GameCache
    {
        public const int MinimumGameCount = 2;

        private static readonly GameCache _instance = new GameCache();

        private List<Game> _games = new List<Game>();
        private readonly int _updateThreadInterval;
        private Thread _updateThread = null;
        private readonly string _databaseConnectionString = null;
        private readonly string _staticFileDestination = null;
        private readonly bool _createFeedFiles = false;
        
        public static GameCache Instance
        {
            get
            {
                return _instance;
            }
        }

        private readonly HashSet<int> _semiNormalGames = new HashSet<int>();

        private GameCache()
        {
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString;
            _staticFileDestination = ConfigurationManager.AppSettings["FeedPath"];
            _createFeedFiles = (System.Environment.MachineName.ToLower() == ConfigurationManager.AppSettings["JobMachineName"].ToLower());
            
#if DEBUG
            _createFeedFiles = true;
#endif
            try
            {
                UpdateCache();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to build initial game cache!");
            }

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();

            _semiNormalGames.Add(432);
            _semiNormalGames.Add(4401);
        }

        public bool AllowSemiNormal(int gameID)
        {
            return _semiNormalGames.Contains(gameID);
        }

        public DateTime FileDate
        {
            get;
            set;
        }

        public List<Game> Games
        {
            get
            {
                return _games;
            }
        }

        private HashSet<int> _gameIDs = new HashSet<int>();

        public bool IsValidGame(int gameID)
        {
            return _gameIDs.Contains(gameID);
        }

        public void Initialize() { }

        private void CacheThread()
        {
            while (true)
            {

                try
                {                    
                    try
                    {
                        UpdateCache();
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Unable to update game cache!");                        
                    }

                    Thread.Sleep(_updateThreadInterval);
                }
                catch (ThreadAbortException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Update Thread Exception!");
                }              
            }
        }

        private void UpdateCache()
        {

            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to establish connection to database!", _databaseConnectionString);
                    return;
                }

                var games = new List<Game>();
                
                using (var command = new SqlCommand("spGetGames", conn))
                {
                    command.CommandType = CommandType.StoredProcedure;

                    // Populate the cache with recently modified addons:
                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            var game = new Game();
                            game.SetFromDataReader(conn, reader);
                            games.Add(game);                            
                        }
                    }
                }

                if (games.Count < MinimumGameCount)
                {
                    Logger.Warn("Game list is incomplete. Skipping cache update, for this iteration.");
                    return;
                }

                lock (_games)
                {
                    _games = games;
                    _gameIDs = new HashSet<int>(games.Select(p => p.ID));
                }
            }

            SaveToDisk(_games);
        }

        private void SaveToDisk(List<Game> games)
        {
            if (_createFeedFiles)
            {

                if (games.Count == 0)
                {
                    Logger.Warn("Games feed will not be created. It would be empty!");
                    return;
                }

                var feed = new GameFeed() { Data = games.ToArray() };
                feed.SaveToDisk(_staticFileDestination);

                var mobileFeed = new GameLookupFeed() { Data = games.Select(p => new GameLookup { ID = p.ID, Name = p.Name }).ToArray() };
                mobileFeed.SaveToDisk(_staticFileDestination);

                var uncompressedFeed = new UncompressedGameFeed {Data = feed.Data};
                uncompressedFeed.SaveToDisk(_staticFileDestination);
            }

        }
    }
}
