﻿using System;
using System.Collections.Concurrent;
using System.Configuration;
using System.Linq;
using System.Threading;
using Curse.AddOnService.Extensions;
using Curse.Logging;
using Curse.Minecraft.Models;

namespace Curse.AddOnService
{
    public class MinecraftCache
    {
        private static int _updateIntervalMillis;
        private static readonly MinecraftCache _instance = new MinecraftCache();
        private static readonly LogCategory Logger = new LogCategory("MinecraftCache");
        

        public static MinecraftCache Instance
        {
            get { return _instance; }
        }

        private readonly Thread _updateThread;

        public const int MinecraftGameID = 432;
        private readonly ConcurrentDictionary<string, MinecraftGameVersion> _gameVersions = new ConcurrentDictionary<string, MinecraftGameVersion>(StringComparer.InvariantCultureIgnoreCase);
        private readonly ConcurrentDictionary<string, MinecraftModLoaderVersion> _modLoaderVersions = new ConcurrentDictionary<string, MinecraftModLoaderVersion>(StringComparer.InvariantCultureIgnoreCase);
        private readonly bool _creatFeedFiles;

        private MinecraftCache()
        {
            try
            {
                _creatFeedFiles = string.IsNullOrEmpty(ConfigurationManager.AppSettings["MinecraftFeedServer"]) || ConfigurationManager.AppSettings["MinecraftFeedServer"].Equals(Environment.MachineName, StringComparison.InvariantCultureIgnoreCase);           
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            
            _updateThread = new Thread(UpdateCacheThread) { Priority = ThreadPriority.Normal };
        }

        public void Initialize()
        {
            _updateIntervalMillis = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _updateThread.Start();
            Logger.Info("Feed files " + (_creatFeedFiles ? "will" : "will not") + " be created from this node.");
        }

        private void UpdateCacheThread()
        {
            while (true)
            {
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Error while updating the Minecraft Cache.");
                }
                finally
                {
                    Thread.Sleep(_updateIntervalMillis);
                }

            }
        }

        private void UpdateCache()
        {
            var gameVersions = MinecraftGameVersionExtensions.GetAllNewAndNormal();
            var modLoaderVersions = MinecraftModLoaderVersionExtensions.GetAllNewAndNormal();

            // Remove any that are no longer there
            foreach (var modLoader in modLoaderVersions)
            {              
                _modLoaderVersions.AddOrUpdate(modLoader.Name, s => modLoader, (s, version) => modLoader);
            }

            // Remove any that are no longer there
            foreach (var gameVersion in gameVersions)
            {
                _gameVersions.AddOrUpdate(gameVersion.VersionString, s => gameVersion, (s, version) => gameVersion);
            }

            if (_creatFeedFiles)
            {
                try
                {
                    CreateFeedFiles();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to create feed fails.");
                }
            }

        }

        public MinecraftGameVersion GetMinecraftVersionByKey(string key)
        {
            MinecraftGameVersion found;
            if (_gameVersions.TryGetValue(key, out found))
            {
                return found;
            }

            return null;
        }

        public MinecraftModLoaderVersion GetMinecraftModLoaderByKey(string key)
        {
            MinecraftModLoaderVersion found;
            if (_modLoaderVersions.TryGetValue(key, out found))
            {
                return found;
            }

            return null;
        }



        #region Feed Files
        

        private void CreateFeedFiles()
        {
            {
                // Create a simple feed file which includes the minecraft version string, forge version string, latest and recommended            
                var allVersions = _modLoaderVersions.Values.ToArray();
                var feedEntries = allVersions.Select(p => new MinecraftModLoaderIndex {GameVersion = p.MinecraftVersion, Latest = p.Latest, Recommended = p.Recommended, Name = p.Name}).ToArray();
                var feed = new MinecraftModLoaderFeed(MinecraftGameID) { Data = feedEntries };
                feed.SaveToDisk(ConfigurationManager.AppSettings["FeedPath"]);
            }

            {
                // Create a simple feed file which includes the minecraft version string, forge version string, latest and recommended                            
                var feed = new MinecraftGameVersionFeed { Data = _gameVersions.Values.ToArray() };
                feed.SaveToDisk(ConfigurationManager.AppSettings["FeedPath"]);
            }
        }


        #endregion
    }
}