﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;
using Curse.AddOns;
using Curse.Extensions;
using Curse.Logging;

namespace Curse.AddOnService.Extensions
{
    public static class AddOnExtension
    {
        private static readonly string sDownloadBaseUrl;
        private static readonly string sAuthorBaseUrl;
        private static readonly string sCategoryBaseUrl;
        private static readonly string sAvatarBaseUrl = null;
        private static readonly Regex sHtmlStripper = new Regex("<!--.*?-->|<[^>]*>", RegexOptions.Compiled);
        private static readonly Regex sHrefGrabber = new Regex("href.*?\"(?<href>.*?)\"", RegexOptions.Compiled);

        static AddOnExtension()
        {
            sDownloadBaseUrl = ConfigurationManager.AppSettings["DownloadBaseUrl"];
            sAuthorBaseUrl = ConfigurationManager.AppSettings["AuthorBaseUrl"];
            sCategoryBaseUrl = ConfigurationManager.AppSettings["CategoryBaseUrl"];
        }

        public static void SetFromDataReader(this AddOn addon,
            SqlDataReader pReader,            
            Dictionary<int, List<AddOnAuthor>> authorCache,
            Dictionary<int, List<AddOnCategory>> categoryCache,
            Dictionary<int, Dictionary<int, AddOnFile>> newFileCache,
            Dictionary<int, AddOnFile> existingFiles,
            Dictionary<int, List<AddOnAttachment>> attachmentCache)
        {
            addon.Id = pReader.GetInt32(pReader.GetOrdinal("addon_id"));
#if DEBUG
            if (addon.Id == 31312)
            {
                addon.Id = 31312;
            }
#endif
            addon.Name = pReader.GetString(pReader.GetOrdinal("addon_name"));
            addon.PrimaryCategoryId = pReader.GetInt32(pReader.GetOrdinal("addon_primary_category_id"));

            var primaryCategory = CategoryCache.Instance.GetByID(addon.PrimaryCategoryId);

            if (primaryCategory != null)
            {
                addon.PrimaryCategoryName = primaryCategory.Name;
                addon.PrimaryCategoryAvatarUrl = primaryCategory.AvatarUrl;
            }

            addon.PortalName = pReader.GetString(pReader.GetOrdinal("addon_portal"));
            addon.SectionName = pReader.GetString(pReader.GetOrdinal("addon_section"));
            addon.Slug = pReader.GetString(pReader.GetOrdinal("addon_download_name"));
            addon.WebSiteURL = String.Format(sDownloadBaseUrl, addon.PortalName, addon.SectionName, pReader["addon_game_slug"], addon.Slug);

            var sectionId = pReader.GetInt32(pReader.GetOrdinal("addon_section_id"));
            addon.CategorySection = CategorySectionCache.Instance.GetByID(sectionId);

            if (pReader["addon_external_url"] != System.DBNull.Value && (string)pReader["addon_external_url"] != string.Empty)
            {
                addon.ExternalUrl = pReader.GetString(pReader.GetOrdinal("addon_external_url"));
            }

            addon.GameName = pReader.GetString(pReader.GetOrdinal("addon_game"));
            addon.GameId = pReader.GetInt32(pReader.GetOrdinal("addon_game_id"));
            addon.DateModified = pReader.GetDateTime(pReader.GetOrdinal("addon_last_updated"));
            addon.DateCreated = pReader.GetDateTime(pReader.GetOrdinal("addon_date_created"));
            addon.DateReleased = pReader.GetDateTime(pReader.GetOrdinal("addon_date_released"));
            addon.CommentCount = pReader.GetInt32(pReader.GetOrdinal("addon_num_comments"));
            addon.DownloadCount = pReader.GetInt32(pReader.GetOrdinal("addon_num_downloads"));

            addon.Rating = 0;
            addon.Likes = pReader.GetInt32(pReader.GetOrdinal("addon_average_rating"));

            addon.IsAvailable = (pReader.GetInt32(pReader.GetOrdinal("addon_is_available")) == 1) ? true : false;
            addon.Status = (ProjectStatus)pReader.GetByte(pReader.GetOrdinal("addon_status"));
            addon.Stage = (ProjectStage)pReader.GetByte(pReader.GetOrdinal("addon_stage"));

            if ((int)addon.Status == 0)
            {
                addon.Status = ProjectStatus.Deleted;
            }

            if ((int)addon.Stage == 0)
            {
                addon.Stage = ProjectStage.Deleted;
            }

            addon.InstallCount = pReader.GetInt32(pReader.GetOrdinal("addon_install_count"));
            addon.FullDescription = pReader.GetString(pReader.GetOrdinal("addon_description"));
            addon.Summary = getSummary(pReader.GetString(pReader.GetOrdinal("addon_summary")));
            addon.PrimaryAuthorName = pReader.GetString(pReader.GetOrdinal("addon_primary_author"));

            var primaryAuthor = new AddOnAuthor();
            primaryAuthor.Name = addon.PrimaryAuthorName;
            primaryAuthor.Url = primaryAuthor.GetAuthorUrl(primaryAuthor.Name);

            // Authors
            if (authorCache.ContainsKey(addon.Id))
            {
                addon.Authors = authorCache[addon.Id];

                int primaryAuthorIndex = addon.Authors.IndexOf(addon.Authors.FirstOrDefault(p => p.Name == addon.PrimaryAuthorName));
                if (primaryAuthorIndex >= 0)
                {
                    addon.Authors.RemoveAt(primaryAuthorIndex);
                    addon.Authors.Insert(0, primaryAuthor);
                } //if the author is in the list move them to the top
                else
                {
                    addon.Authors.Insert(0, primaryAuthor);
                }
            }
            else
            {
                addon.Authors = new List<AddOnAuthor>();
                addon.Authors.Add(primaryAuthor);
            }

            //Attachments
            if (attachmentCache.ContainsKey(addon.Id))
            {
                addon.Attachments = attachmentCache[addon.Id];
            }


            // Files
            var game = GameCache.Instance.Games.FirstOrDefault(p => p.ID == addon.GameId);
            if (newFileCache.ContainsKey(addon.Id))
            {
                foreach (var kvp in newFileCache[addon.Id])
                {
                    AddOnFile bestFile = null;
                    AddOnFile existingFile = null;
                    var existingFileIsBetter = false;

                    var file = kvp.Value;

                    if (existingFiles.TryGetValue(kvp.Key, out existingFile))
                    {
                        if (existingFile.Modules.Count > 0 && file.Modules.Count == 0)
                        {
                            existingFileIsBetter = true;
                        }
                        if (existingFile.Fingerprints.Count > 0 && file.Fingerprints.Count == 0)
                        {
                            existingFileIsBetter = true;
                        }
                        if (existingFile.Dependencies.Count > 0 && file.Dependencies.Count == 0)
                        {
                            existingFileIsBetter = true;
                        }
                    }
                    if (existingFileIsBetter)
                    {
                        Logger.Debug(string.Format("Addon '{0}' ({1}) has an incomplete file '{2}' ({3}). Using previously cached version.", addon.Name, addon.Id, file.FileName, file.Id));
                        bestFile = existingFile;
                        bestFile.FileStatus = file.FileStatus;
                        bestFile.FileDate = file.FileDate;
                    }
                    else
                    {
                        bestFile = file;
                    }

                    if (game != null)
                    {
                        if (addon.PackageType == PackageTypes.Folder || addon.PackageType == PackageTypes.SingleFile)
                        {
                            if (bestFile.Modules.Count == 0)
                            {
                                continue;
                            }
                            if (!addon.Files.ContainsKey(file.Id))
                            {
                                addon.Files.Add(file.Id, bestFile);
                            }
                        }
                        else
                        {
                            if (!addon.Files.ContainsKey(file.Id))
                            {
                                addon.Files.Add(file.Id, bestFile);
                            }
                        }
                    }
                    else
                    {
                        if (bestFile.Modules.Count == 0)
                        {
                            continue;
                        }
                        if (!addon.Files.ContainsKey(file.Id))
                        {
                            addon.Files.Add(file.Id, bestFile);
                        }
                    }
                }
            }

            // Categories
            if (categoryCache.ContainsKey(addon.Id))
            {
                addon.Categories = categoryCache[addon.Id];
            }
            else
            {
                // TODO: The game name is not right here we need the game slug
                string categoryUrl = sCategoryBaseUrl.FormatWith(addon.SectionName, addon.GameName, "miscellaneous");
                addon.Categories = new List<AddOnCategory>();
                addon.Categories.Add(new AddOnCategory(17, "Miscellaneous", categoryUrl));
            }

            // Based on the files, re-flag IsAvailable:            
            var normalFileCount = GameCache.Instance.AllowSemiNormal(addon.GameId) ?
                addon.Files.Count(p => p.Value.FileStatus == FileStatus.Normal || p.Value.FileStatus == FileStatus.SemiNormal || (p.Value.FileStatus == FileStatus.ClientOnly && p.Value.IsAlternate))
                : addon.Files.Count(p => p.Value.FileStatus == FileStatus.Normal || (p.Value.FileStatus == FileStatus.ClientOnly && p.Value.IsAlternate));

            if (addon.IsAvailable && normalFileCount == 0)
            {
                addon.IsAvailable = false;
            }

            // Set Individual Fingerprints            
            addon.LatestFiles = addon.GetLatestFiles();
            addon.DefaultFileId = addon.GetDefaultFileId();
            addon.IconId = 1;
            addon.DonationUrl = getDonationUrl(pReader.GetString(pReader.GetOrdinal("addon_donation_link")));
            addon.GameVersionLatestFiles = addon.GetGameVersionLatestFiles();
            addon.IsFeatured = pReader.GetInt32(pReader.GetOrdinal("isFeatured"));
            addon.PopularityScore = pReader.GetDouble(pReader.GetOrdinal("addon_popularity_score"));
            addon.GamePopularityRank = pReader.GetInt32(pReader.GetOrdinal("addon_game_popularity_rank"));
        }

        private static string getSummary(string rawSummary)
        {
            if (rawSummary.Contains("<"))
            {
                return Utility.StripHTML(rawSummary).Replace("\r", "").Trim();
            }
            return rawSummary;
        }

        private static string getDonationUrl(string donationLink)
        {
            if (string.IsNullOrEmpty(donationLink))
            {
                return null;
            }
            var match = sHrefGrabber.Match(donationLink);
            if (!match.Success)
            {
                return null;
            }
            return match.Groups["href"].Value;
        }

        private static AddOnFile[] GetFilteredFiles(AddOn addon, IEnumerable<AddOnFile>  files, out bool allowSemiNormalFiles)
        {
            switch (addon.PackageType)
            {
                case PackageTypes.ModPack:
                case PackageTypes.SingleFile:
                case PackageTypes.Folder:
                    // Hack to allow Kerbal feeds to be created
                    allowSemiNormalFiles = addon.GameId == 4401;
                    return files.Where(p => !p.IsAlternate && p.IsAvailable).ToArray();                    
                case PackageTypes.Mod:
                case PackageTypes.Cmod2:
                case PackageTypes.Ctoc:
                    allowSemiNormalFiles = true;
                    return files.Where(p => p.FileStatus == FileStatus.Normal || p.FileStatus == FileStatus.SemiNormal).ToArray();                    
                default:
                    allowSemiNormalFiles = false;
                    return null;
            }
        }

        private static List<AddOnFile> GetLatestFiles(this AddOn addon)
        {
            var game = GameCache.Instance.Games.FirstOrDefault(g => g.ID == addon.GameId);
            if (game == null)
            {
                return null;
            }
            
            // all I need to do is change the source            
            var allowSemiNormalFiles = false;
            var sourceFiles = GetFilteredFiles(addon, addon.Files.Values, out allowSemiNormalFiles).ToList();
            var files = new List<AddOnFile>();

            foreach (var file in sourceFiles)
            {

                if (file.AlternateFileId > 0 && !addon.Files.ContainsKey(file.AlternateFileId))
                {
                    file.AlternateFileId = 0;
                }

                if (!allowSemiNormalFiles && file.FileStatus == FileStatus.SemiNormal && file.AlternateFileId == 0)
                {
                    continue;
                }

                var existingFile = files.FirstOrDefault(p => p.ReleaseType == file.ReleaseType && p.IsAlternate == false);

                if (existingFile == null || file.FileDate > existingFile.FileDate)
                {
                    if (existingFile != null)
                    {
                        files.Remove(existingFile);
                    }

                    files.Add(file);
                }
            }

            // Now the alternates:
            foreach (FileType releaseType in Enum.GetValues(typeof(FileType)))
            {
                var primaryFile = files.FirstOrDefault(p => p.ReleaseType == releaseType);

                if (primaryFile == null)
                {
                    continue;
                }

                if (primaryFile.AlternateFileId > 0)
                {
                    var alternateFile = addon.Files.FirstOrDefault(p => p.Value.Id == primaryFile.AlternateFileId && p.Value.IsAvailable).Value;
                    if (alternateFile == null)
                    {
                        primaryFile.AlternateFileId = 0;
                    }
                    else if (!files.Contains(alternateFile))
                    {
                        files.Add(alternateFile);
                    }

                    if (addon.PackageType == PackageTypes.Ctoc || addon.PackageType == PackageTypes.Cmod2)
                    {
                        files.Remove(primaryFile);
                    }
                }
            }

            if (!allowSemiNormalFiles)
            {
                files.RemoveAll(p => p.FileStatus == FileStatus.SemiNormal);
            }

            return files;
        }

        private static HashSet<string> _validVersions = new HashSet<string>();
        private static HashSet<string> _invalidVersions = new HashSet<string>();

        private static List<GameVersionLatestFile> GetGameVersionLatestFiles(this AddOn addon)
        {            
            var latestFiles = new List<GameVersionLatestFile>();

            var allowSemiNormalFiles = false;
            var filteredFiles = GetFilteredFiles(addon, addon.Files.Values, out allowSemiNormalFiles);
            var distinctVersions = filteredFiles.SelectMany(p => p.GameVersion).Distinct();

            foreach (var version in distinctVersions) // Look for the latest files for each game version
            {
                if (_invalidVersions.Contains(version))
                {
                    continue;
                }

                if (!_validVersions.Contains(version))
                {
                    Version tmp;
                    if (Version.TryParse(version, out tmp))
                    {
                        _validVersions.Add(version);
                    }
                    else
                    {
                        _invalidVersions.Add(version);
                        continue;
                    }
                }

                foreach (FileType releaseType in Enum.GetValues(typeof(FileType))) // And also by type (Alpha, Beta, Release, etc)
                {
                    var bestForVersionAndType = filteredFiles.Where(p => p.ReleaseType == releaseType && p.GameVersion.Contains(version) && p.IsAvailable).OrderByDescending(p => p.FileDate).FirstOrDefault();
                    if (bestForVersionAndType != null)
                    {
                        latestFiles.Add(new GameVersionLatestFile
                        {
                            FileType = releaseType,
                            GameVesion = version,
                            ProjectFileID = bestForVersionAndType.Id,
                            ProjectFileName = bestForVersionAndType.FileName
                        });
                    }
                }
            }
            return latestFiles;
        }

        private static int GetDefaultFileId(this AddOn addon)
        {
            if (addon.LatestFiles == null || !addon.LatestFiles.Any())
            {
                Logger.Debug("Addon " + addon.Name + " has no latest files.", new { addon.Id, addon.DefaultFileId });
                return 0;
            }

            // Release
            if (addon.LatestFiles.Exists(p => p.ReleaseType == FileType.Release && p.IsAlternate == false && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == FileType.Release && p.IsAlternate == false).Id;
            }
            if (addon.LatestFiles.Exists(p => p.ReleaseType == FileType.Release && p.IsAlternate && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == FileType.Release && p.IsAlternate).Id;
            }

            // Beta
            if (addon.LatestFiles.Exists(p => p.ReleaseType == FileType.Beta && p.IsAlternate == false && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == FileType.Beta && p.IsAlternate == false && p.IsAvailable).Id;
            }
            if (addon.LatestFiles.Exists(p => p.ReleaseType == FileType.Beta && p.IsAlternate && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == FileType.Beta && p.IsAlternate).Id;
            }

            // Alpha
            if (addon.LatestFiles.Exists(p => p.ReleaseType == FileType.Alpha && p.IsAlternate == false && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == FileType.Alpha && p.IsAlternate == false).Id;
            }
            if (addon.LatestFiles.Exists(p => p.ReleaseType == FileType.Alpha && p.IsAlternate && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == FileType.Alpha && p.IsAlternate).Id;
            }

            return 0;
        }

        public static void SetFileFingerprints(this AddOn addon, SqlConnection pConn)
        {
            var command = new SqlCommand("curseService_GetAddOnFileFingerprints", pConn);
            command.CommandType = CommandType.StoredProcedure;
            command.Connection = pConn;
            command.Parameters.Add("@intProjectId", SqlDbType.Int);
            command.Parameters["@intProjectId"].Value = addon.Id;

            using (var reader = command.ExecuteReader())
            {
                if (!reader.HasRows)
                {
                    return;
                }

                while (reader.Read())
                {
                    int fileId = reader.GetInt32(0);
                    long fingerprint = reader.GetInt64(1);
                    if (!addon.Files.ContainsKey(fileId))
                    {
                        continue;
                    }
                    addon.Files[fileId].Fingerprints.Add(fingerprint);
                    if (reader[2] != System.DBNull.Value)
                    {
                        var module = new AddOnModule();
                        module.Fingerprint = fingerprint;
                        module.Foldername = reader.GetString(2);
                        addon.Files[fileId].Modules.Add(module);
                    }
                }
            }
        }

        public static void SetIndividualFileFingerprints(this AddOn addon, Dictionary<int, List<IndividualFileFingerprint>> pIndividualFingerprints)
        {
            if (!pIndividualFingerprints.ContainsKey(addon.Id))
            {
                return;
            }

            foreach (var fingerprint in pIndividualFingerprints[addon.Id])
            {
                if (!addon.Files.ContainsKey(fingerprint.FileId))
                {
                    continue;
                }
                if (!addon.Files[fingerprint.FileId].FolderFingerprints.ContainsKey(fingerprint.Folder))
                {
                    addon.Files[fingerprint.FileId].FolderFingerprints.Add(fingerprint.Folder, new List<long>());
                }
                addon.Files[fingerprint.FileId].FolderFingerprints[fingerprint.Folder].Add(fingerprint.Fingerprint);
            }

            // Make sure the individual fingerprints are sorted:
            foreach (var fileEntry in addon.Files)
            {
                foreach (var folderFingerprints in fileEntry.Value.FolderFingerprints)
                {
                    folderFingerprints.Value.Sort();
                }
            }
        }

        public static string GetAuthorUrl(this AddOnAuthor value, string username)
        {
            return sAuthorBaseUrl.FormatWith(username);
        }

        private static GameVersionLatestFile CreateGameVersionLatestFileRecord(AddOnFile addonFile)
        {
            return new GameVersionLatestFile
            {
                GameVesion = addonFile.GameVersion.LastOrDefault(),
                ProjectFileID = addonFile.Id,
                ProjectFileName = addonFile.FileNameOnDisk,
                FileType = addonFile.ReleaseType,
            };
        }
    }
}