﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Data.SqlClient;
using System.Data;
using System.Text.RegularExpressions;
using System.Configuration;
using Curse.Extensions;
using Curse.AddOns;
using System.Diagnostics;

namespace Curse.ClientService.Extensions
{
    public static class CAddOnExtension
    {

        private readonly static string sDownloadBaseUrl = null;
        private readonly static string sAuthorBaseUrl = null;
        private readonly static string sCategoryBaseUrl = null;
        private readonly static Regex sHtmlStripper = new Regex("<!--.*?-->|<[^>]*>", RegexOptions.Compiled);
        private readonly static Regex sHrefGrabber = new Regex("href.*?\"(?<href>.*?)\"", RegexOptions.Compiled);

        static CAddOnExtension()
        {
            sDownloadBaseUrl = ConfigurationManager.AppSettings["DownloadBaseUrl"];
            sAuthorBaseUrl = ConfigurationManager.AppSettings["AuthorBaseUrl"];
            sCategoryBaseUrl = ConfigurationManager.AppSettings["CategoryBaseUrl"];
        }

        public static void SetFromDataReader(this CAddOn addon,
            SqlDataReader pReader,
            Dictionary<int, List<CIndividualFileFingerprint>> individualFingerprints,
            Dictionary<int, List<CAddOnAuthor>> authorCache,
            Dictionary<int, List<CAddOnCategory>> categoryCache,
            Dictionary<int, List<CAddOnFile>> newFileCache,
            Dictionary<int, CAddOnFile> existingFiles)
        {

            addon.Id = pReader.GetInt32(pReader.GetOrdinal("addon_id"));
#if DEBUG
            if (addon.Id == 38489)
            {
                addon.Id = 38489;
            }
#endif
            addon.Name = pReader.GetString(pReader.GetOrdinal("addon_name"));
            addon.PrimaryCategoryId = pReader.GetInt32(pReader.GetOrdinal("addon_primary_category_id"));

            addon.PortalName = pReader.GetString(pReader.GetOrdinal("addon_portal"));
            addon.SectionName = pReader.GetString(pReader.GetOrdinal("addon_section"));
            addon.Slug = pReader.GetString(pReader.GetOrdinal("addon_download_name"));
            addon.WebSiteURL = String.Format(sDownloadBaseUrl, addon.PortalName, addon.SectionName, pReader["addon_game_slug"], addon.Slug);

            var sectionId = pReader.GetInt32(pReader.GetOrdinal("addon_section_id"));
            addon.CategorySection = CCategorySectionCache.Instance.GetByID(sectionId);

            if (pReader["addon_external_url"] != System.DBNull.Value && (string)pReader["addon_external_url"] != string.Empty)
            {
                addon.ExternalUrl = pReader.GetString(pReader.GetOrdinal("addon_external_url"));
            }

            addon.GameName = pReader.GetString(pReader.GetOrdinal("addon_game"));
            addon.GameId = pReader.GetInt32(pReader.GetOrdinal("addon_game_id"));
            addon.DateModified = pReader.GetDateTime(pReader.GetOrdinal("addon_last_updated"));
            addon.DateCreated = pReader.GetDateTime(pReader.GetOrdinal("addon_date_created"));
            addon.DateReleased = pReader.GetDateTime(pReader.GetOrdinal("addon_date_released"));
            addon.CommentCount = pReader.GetInt32(pReader.GetOrdinal("addon_num_comments"));
            addon.DownloadCount = pReader.GetInt32(pReader.GetOrdinal("addon_num_downloads"));

            addon.Rating = 0;
            addon.Likes = pReader.GetInt32(pReader.GetOrdinal("addon_average_rating"));

            addon.IsAvailable = (pReader.GetInt32(pReader.GetOrdinal("addon_is_available")) == 1) ? true : false;
            addon.Status = (EProjectStatus)pReader.GetByte(pReader.GetOrdinal("addon_status"));
            addon.Stage = (EProjectStage)pReader.GetByte(pReader.GetOrdinal("addon_stage"));
            
            if ((int)addon.Status == 0)
            {
                addon.Status = EProjectStatus.Deleted;
            }

            if ((int)addon.Stage == 0)
            {
                addon.Stage = EProjectStage.Deleted;
            }

            addon.InstallCount = pReader.GetInt32(pReader.GetOrdinal("addon_install_count"));
            addon.FullDescription = pReader.GetString(pReader.GetOrdinal("addon_description"));
            addon.Summary = getSummary(pReader.GetString(pReader.GetOrdinal("addon_summary")));
            addon.PrimaryAuthorName = pReader.GetString(pReader.GetOrdinal("addon_primary_author"));

            CAddOnAuthor primaryAuthor = new CAddOnAuthor();
            primaryAuthor.Name = addon.PrimaryAuthorName;
            primaryAuthor.Url = primaryAuthor.GetAuthorUrl(primaryAuthor.Name);

            // Authors
            if (authorCache.ContainsKey(addon.Id))
            {
                addon.Authors = authorCache[addon.Id];

                int primaryAuthorIndex = addon.Authors.IndexOf(addon.Authors.FirstOrDefault(p => p.Name == addon.PrimaryAuthorName));
                if (primaryAuthorIndex >= 0)
                {
                    addon.Authors.RemoveAt(primaryAuthorIndex);
                    addon.Authors.Insert(0, primaryAuthor);
                }//if the author is in the list move them to the top
                else addon.Authors.Insert(0, primaryAuthor);
            }
            else
            {
                addon.Authors = new List<CAddOnAuthor>();
                addon.Authors.Add(primaryAuthor);
            }

            // Files
            var game = CGameCache.Instance.Games.FirstOrDefault(p => p.ID == addon.GameId);
            if (newFileCache.ContainsKey(addon.Id))
            {
                foreach (CAddOnFile file in newFileCache[addon.Id])
                {
                    CAddOnFile bestFile = null;
                    CAddOnFile existingFile = null;
                    bool existingFileIsBetter = false;

                    if (existingFiles.TryGetValue(file.Id, out existingFile))
                    {
                        if (existingFile.Modules.Count > 0 && file.Modules.Count == 0)
                        {
                            Logger.Log("Addon '{0}' ({1}) has an incomplete file due to no modules. '{2}' ({3}). Using previously cached version.", ELogLevel.Debug, addon.Name, addon.Id, file.FileName, file.Id);
                            existingFileIsBetter = true;
                        }
                        if (existingFile.Fingerprints.Count > 0 && file.Fingerprints.Count == 0)
                        {
                            Logger.Log("Addon '{0}' ({1}) has an incomplete file due to no fingerprints. '{2}' ({3}). Using previously cached version.", ELogLevel.Debug, addon.Name, addon.Id, file.FileName, file.Id);
                            existingFileIsBetter = true;
                        }
                        if (existingFile.Dependencies.Count > 0 && file.Dependencies.Count == 0)
                        {
                            Logger.Log("Addon '{0}' ({1}) has an incomplete file due to no dependencies. '{2}' ({3}). Using previously cached version.", ELogLevel.Debug, addon.Name, addon.Id, file.FileName, file.Id);
                            existingFileIsBetter = true;
                        }
                    }

                    if (existingFileIsBetter)
                    {
                       bestFile = existingFile;
                        bestFile.FileStatus = file.FileStatus;
                        bestFile.FileDate = file.FileDate;
                    }
                    else
                    {
                        bestFile = file;
                    }

                    if (game != null)
                    {
                        //if (game.PackageType == EPackageType.Folder || game.PackageType == EPackageType.SingleFile)
                        if(addon.PackageType == EPackageType.Folder || addon.PackageType == EPackageType.SingleFile)
                        {
                            if (bestFile.Modules.Count == 0)
                            {
                                continue;
                            }
                            addon.Files.Add(file.Id, bestFile);
                        }
                        else
                        {
                            addon.Files.Add(file.Id, bestFile);
                        }
                    }
                    else
                    {
                        if (bestFile.Modules.Count == 0)
                        {
                            continue;
                        }
                        addon.Files.Add(file.Id, bestFile);
                    }
                }
            }

            // Categories
            if (categoryCache.ContainsKey(addon.Id))
            {
                addon.Categories = categoryCache[addon.Id];
            }
            else
            {
                // TODO: The game name is not right here we need the game slug
                string categoryUrl = sCategoryBaseUrl.FormatWith(addon.SectionName, addon.GameName, "miscellaneous");
                addon.Categories = new List<CAddOnCategory>();
                addon.Categories.Add(new CAddOnCategory(17, "Miscellaneous", categoryUrl));
            }

            // Based on the files, re-flag IsAvailable:
            int normalFileCount = addon.Files.Count(p => p.Value.FileStatus == EFileStatus.Normal || (p.Value.FileStatus == EFileStatus.ClientOnly && p.Value.IsAlternate == true));
            if (addon.IsAvailable && normalFileCount == 0)
            {
                addon.IsAvailable = false;
            }

            // Set Individual Fingerprints
            addon.SetIndividualFileFingerprints(individualFingerprints);
            addon.LatestFiles = addon.GetLatestFiles();
            addon.DefaultFileId = addon.GetDefaultFileId();
            addon.IconId = 1;
            addon.DonationUrl = getDonationUrl(pReader.GetString(pReader.GetOrdinal("addon_donation_link")));
        }

        private static string getSummary(string rawSummary)
        {
            if (rawSummary.Contains("<"))
            {
                return Utility.StripHTML(rawSummary).Replace("\r", "").Trim();
            }
            else
            {
                return rawSummary;
            }
        }

        private static string getDonationUrl(string donationLink)
        {
            if (string.IsNullOrEmpty(donationLink))
            {
                return null;
            }
            Match match = sHrefGrabber.Match(donationLink);
            if (!match.Success)
            {
                return null;
            }
            return match.Groups["href"].Value;
        }

        private static List<CAddOnFile> GetLatestFiles(this CAddOn addon)
        {
            var game = CGameCache.Instance.Games.FirstOrDefault(g => g.ID == addon.GameId);
            if (game == null) return null;

            List<CAddOnFile> files = new List<CAddOnFile>();
            // all I need to do is change the source
            IEnumerable<KeyValuePair<int, CAddOnFile>> source;
            //switch (game.PackageType)
            switch(addon.PackageType)
            {
                case EPackageType.SingleFile:
                case EPackageType.Folder:
                    source = addon.Files.Where(p => !p.Value.IsAlternate && p.Value.IsAvailable);
                    break;
                case EPackageType.Cmod2:
                case EPackageType.Ctoc:
                    source = addon.Files.Where(p => p.Value.FileStatus == EFileStatus.Normal || p.Value.FileStatus == EFileStatus.SemiNormal);
                    break;
                default:
                    return null;
            }

            foreach (KeyValuePair<int, CAddOnFile> kvp in source)
            {
                CAddOnFile file = kvp.Value;
                if (file.AlternateFileId > 0 && !addon.Files.ContainsKey(file.AlternateFileId))
                {
                    file.AlternateFileId = 0;
                }

                if (file.FileStatus == EFileStatus.SemiNormal && file.AlternateFileId == 0)
                {
                    continue;
                }

                CAddOnFile existingFile = files.FirstOrDefault(p => p.ReleaseType == file.ReleaseType && p.IsAlternate == false);

                if (existingFile == null || file.FileDate > existingFile.FileDate)
                {
                    if (existingFile != null)
                    {
                        files.Remove(existingFile);
                        files.Find(p => p.Id == existingFile.AlternateFileId);
                    }

                    files.Add(file);
                }
            }

            // Now the alternates:
            foreach (EFileType releaseType in System.Enum.GetValues(typeof(EFileType)))
            {
                CAddOnFile primaryFile = files.FirstOrDefault(p => p.ReleaseType == releaseType);

                if (primaryFile == null)
                {
                    continue;
                }

                if (primaryFile.AlternateFileId > 0)
                {
                    CAddOnFile alternateFile = addon.Files.FirstOrDefault(p => p.Value.Id == primaryFile.AlternateFileId && p.Value.IsAvailable).Value;
                    if (alternateFile == null)
                    {
                        primaryFile.AlternateFileId = 0;
                    }
                    else if (!files.Contains(alternateFile))
                    {
                        files.Add(alternateFile);
                    }

                    //if (game.PackageType == EPackageType.Ctoc || game.PackageType == EPackageType.Cmod2)
                    if (addon.PackageType == EPackageType.Ctoc || addon.PackageType == EPackageType.Cmod2)
                    {
                        files.Remove(primaryFile);
                    }
                }
            }

            //remove the semi-normals
            for (var i = files.Count - 1; i > 0; i--)
            {
                if (files[i].FileStatus == EFileStatus.SemiNormal)
                    files.RemoveAt(i);
            }

            return files;
        }

        private static int GetDefaultFileId(this CAddOn addon)
        {
            if (addon.LatestFiles == null)
            {
                Logger.Log("Warning: Addon {0} has no latest files.", ELogLevel.Warning, addon.Id);
                return 0;
            }
            
            // Release
            if (addon.LatestFiles.Exists(p => p.ReleaseType == EFileType.Release && p.IsAlternate == false && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == EFileType.Release && p.IsAlternate == false).Id;
            }
            else if (addon.LatestFiles.Exists(p => p.ReleaseType == EFileType.Release && p.IsAlternate == true && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == EFileType.Release && p.IsAlternate == true).Id;
            }

            // Beta
            if (addon.LatestFiles.Exists(p => p.ReleaseType == EFileType.Beta && p.IsAlternate == false && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == EFileType.Beta && p.IsAlternate == false && p.IsAvailable).Id;
            }
            else if (addon.LatestFiles.Exists(p => p.ReleaseType == EFileType.Beta && p.IsAlternate == true && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == EFileType.Beta && p.IsAlternate == true).Id;
            }

            // Alpha
            if (addon.LatestFiles.Exists(p => p.ReleaseType == EFileType.Alpha && p.IsAlternate == false && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == EFileType.Alpha && p.IsAlternate == false).Id;
            }
            else if (addon.LatestFiles.Exists(p => p.ReleaseType == EFileType.Alpha && p.IsAlternate == true && p.IsAvailable))
            {
                return addon.LatestFiles.First(p => p.ReleaseType == EFileType.Alpha && p.IsAlternate == true).Id;
            }

            return 0;
        }

        public static void SetFileFingerprints(this CAddOn addon, SqlConnection pConn)
        {
            SqlCommand command = new SqlCommand("curseService_GetAddOnFileFingerprints", pConn);
            command.CommandType = CommandType.StoredProcedure;
            command.Connection = pConn;
            command.Parameters.Add("@intProjectId", SqlDbType.Int);
            command.Parameters["@intProjectId"].Value = addon.Id;

            using (SqlDataReader reader = command.ExecuteReader())
            {
                if (!reader.HasRows)
                {
                    return;
                }

                while (reader.Read())
                {
                    int fileId = reader.GetInt32(0);
                    long fingerprint = reader.GetInt64(1);
                    if (!addon.Files.ContainsKey(fileId))
                    {
                        continue;
                    }
                    addon.Files[fileId].Fingerprints.Add(fingerprint);
                    if (reader[2] != System.DBNull.Value)
                    {
                        CAddOnModule module = new CAddOnModule();
                        module.Fingerprint = fingerprint;
                        module.Foldername = reader.GetString(2);
                        addon.Files[fileId].Modules.Add(module);

                    }
                }
            }
        }

        public static void SetIndividualFileFingerprints(this CAddOn addon, Dictionary<int, List<CIndividualFileFingerprint>> pIndividualFingerprints)
        {

            if (!pIndividualFingerprints.ContainsKey(addon.Id))
            {
                return;
            }

            foreach (CIndividualFileFingerprint fingerprint in pIndividualFingerprints[addon.Id])
            {

                if (!addon.Files.ContainsKey(fingerprint.FileId))
                {
                    continue;
                }
                if (!addon.Files[fingerprint.FileId].FolderFingerprints.ContainsKey(fingerprint.Folder))
                {
                    addon.Files[fingerprint.FileId].FolderFingerprints.Add(fingerprint.Folder, new List<long>());
                }
                addon.Files[fingerprint.FileId].FolderFingerprints[fingerprint.Folder].Add(fingerprint.Fingerprint);
            }

            // Make sure the individual fingerprints are sorted:
            foreach (KeyValuePair<int, CAddOnFile> fileEntry in addon.Files)
            {
                foreach (KeyValuePair<string, List<long>> folderFingerprints in fileEntry.Value.FolderFingerprints)
                {
                    folderFingerprints.Value.Sort();
                }
            }
        }

        public static string GetAuthorUrl(this CAddOnAuthor value, string username)
        {
            return sAuthorBaseUrl.FormatWith(username);
        }
    }
}
