﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Linq;
using Curse.AddOns;
using Curse.AddOnService.Utilities;
using Curse.Extensions;
using Curse.Logging;
using Curse.MurmurHash;
using ICSharpCode.SharpZipLib.BZip2;
using Newtonsoft.Json;
using System.IO;

namespace Curse.AddOnService.Extensions
{
    public static class FeedFileExtensions
    {
        private static readonly Uri[] _cdnBaseUrls;

        static FeedFileExtensions()
        {
            _cdnBaseUrls = ConfigurationManager.AppSettings["AddOnFeedFileUrlFormats"].Split(';').Select(p => new Uri(p, UriKind.Absolute)).ToArray();
        }

        private static long GetFeedHash<T>(FeedFile<T> file)
        {

            var tempFile = Path.GetTempFileName();
            try
            {
                using (var fs = File.Open(tempFile, FileMode.Truncate))
                using (var sw = new StreamWriter(fs))
                using (var jw = new JsonTextWriter(sw))
                {
                    jw.Formatting = Formatting.Indented;
                    var serializer = new JsonSerializer();
                    serializer.Serialize(jw, file.Data);
                }

                return MurmurHash2.GetHashCode(File.ReadAllBytes(tempFile));
            }
            finally
            {
                if (File.Exists(tempFile))
                {
                    File.Delete(tempFile);
                }
            }        
        }

        public static void SaveToDisk<T>(this FeedFile<T> feedFile, string rootFolderPath)           
        {
            // Create a simple feed file which includes the minecraft version string, forge version string, latest and recommended            

            var tempFiles = new HashSet<string>();
            
            try
            {
                feedFile.Timestamp = DateTime.UtcNow.ToEpoch();
                var basePath = Path.Combine(rootFolderPath, feedFile.RelativeFolderPath);
                var hashFilePath = Path.Combine(basePath, feedFile.FileName + ".hash");
                var versionFilePath = Path.Combine(basePath, feedFile.FileName + ".txt");

                Directory.CreateDirectory(basePath);

                var tempfile = Path.Combine(basePath, Guid.NewGuid() + ".json");
                tempFiles.Add(tempfile);

                using (var fs = File.Open(tempfile, FileMode.CreateNew))
                using (var sw = new StreamWriter(fs))
                using (var jw = new JsonTextWriter(sw))
                {
                    jw.Formatting = Formatting.Indented;
                    var serializer = new JsonSerializer();
                    serializer.Serialize(jw, feedFile);
                }

                // Ensure the file actually changed
                var newFileHash = GetFeedHash(feedFile);
                var currentfile = Path.Combine(rootFolderPath, feedFile.RelativeFolderPath, feedFile.FileName);

                try
                {
                    if (File.Exists(versionFilePath) && File.Exists(currentfile) && File.Exists(hashFilePath) && long.Parse(File.ReadAllText(hashFilePath)) == newFileHash)
                    {
                        Logger.Debug("Getting rid of temp file at: " + tempfile);

                        try
                        {
                            File.Delete(tempfile);
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex);
                        }

                        return;
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to determine if feed file has changed!");
                }


                if (feedFile.FileName.EndsWith(".bz2"))
                {
                    var compressedFile = Path.Combine(basePath, Guid.NewGuid() + ".json.bz2");
                    tempFiles.Add(compressedFile);

                    using (var destination = File.Open(compressedFile, FileMode.CreateNew))
                    using (var source = File.Open(tempfile, FileMode.Open))
                    {
                        using (var bz2 = new BZip2OutputStream(destination))
                        {
                            source.CopyTo(bz2);
                        }
                    }
                    tempfile = compressedFile;
                }

                FileHelper.ReplaceFile(currentfile, tempfile);

                // Create the version json             
                var versionTempFile = Path.Combine(basePath, Guid.NewGuid() + ".json");
                File.WriteAllText(versionTempFile, feedFile.Timestamp.ToString());
                
                FileHelper.ReplaceFile(versionFilePath, versionTempFile);

                // Write out the hash
                File.WriteAllText(hashFilePath, newFileHash.ToString());

                // Invalidate CDN URls
                try
                {
                    foreach (var urlFormat in _cdnBaseUrls)
                    {
                        var cdnUrl = feedFile.GetUri(urlFormat);
                        CdnCacheInvalidator.InvalidateUrl(cdnUrl.ToString());
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex);
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to save feed file to disk!");
            }
            finally 
            {
                foreach (var tempFile in tempFiles)
                {
                    if (File.Exists(tempFile))
                    {
                        File.Delete(tempFile);
                    }
                }               

            }
        }
    }
}