﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.AddOns;
using System.Data.SqlClient;
using System.Data;
using Curse.Extensions;
using System.Configuration;
using System.Drawing;
using System.IO;
using Curse.Logging;

namespace Curse.AddOnService.Extensions
{
    public static class GameExtension
    {

        public static void SetFromDataReader(this Game game, SqlConnection pConn, SqlDataReader pReader)
        {
            game.ID = (int)pReader["ID"];
            game.Name = (string)pReader["Name"];
            game.Slug = (string)pReader["Slug"];

            game.SupportsAddons = (bool)pReader["SupportsAddons"];
            game.SupportsVoice = (bool)pReader["SupportsVoice"];

            game.Order = (int)pReader["Order"];

            game.SupportsNotifications = (bool)pReader["SupportsNotifications"];

            game.BundleAssets = (bool) pReader["BundleAssets"];

            #region Game Files

            using (var command = new SqlCommand("SELECT * FROM GameFile WHERE GameID = @ID", pConn))
            {
                command.Parameters.Add("@ID", SqlDbType.Int).Value = game.ID;
                using (var reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var gameFile = new GameFile
                        {
                            Id = (int) reader["ID"],
                            GameId = game.ID,
                            FileName = reader.GetNullableValue<string>("FileName"),
                            IsRequired = reader.GetBoolean(reader.GetOrdinal("IsRequired")),
                            FileType = (GameFileType) reader["FileType"],
                            PlatformType = (GamePlatformType) reader["PlatformType"]
                        };

                        game.GameFiles.Add(gameFile);
                    }
                }
            }

            #endregion

            #region Game Assets
            var gameAssetPath = ConfigurationManager.AppSettings["GameAssetsPath"];

            using (var command = new SqlCommand("SELECT * FROM GameAsset WHERE GameID = @ID OR GameID = 0 ORDER BY GameID ASC", pConn))
            {
                command.Parameters.Add("@ID", SqlDbType.Int).Value = game.ID;
                using (var reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var assetGameID = (int)reader["GameID"];
                        var gameAsset = new GameAsset
                        {
                            Id = (int) reader["ID"],
                            GameId = game.ID,
                            Name = reader.GetNullableValue<string>("Name"),
                        };

                        if (game.BundleAssets)
                        {
                            var assetPath = Path.Combine(gameAssetPath, game.ID.ToString());

                            assetPath = Path.Combine(assetPath, gameAsset.Name + ".png");
                            if (!File.Exists(assetPath))
                            {
                                if (assetGameID != 0)
                                {
                                    Logger.Warn("Unable to find game asset for " + game.Name + " at '" + assetPath + "'");
                                }
                                continue;
                            }

                            gameAsset.Resource = LoadImageNoLock(assetPath);
                        }

                        if (game.Assets.FirstOrDefault(p => p.Name == gameAsset.Name) != null)
                        {
                            var existingAsset = game.Assets.FirstOrDefault(p => p.Name == gameAsset.Name);
                            game.Assets.Remove(existingAsset);
                        }
                        game.Assets.Add(gameAsset);
                    }
                }
            }

            #endregion

            #region Sections

            // Category Section
            if (game.SupportsAddons)
            {
                try
                {
                    game.CategorySections = CategorySectionCache.Instance.GetAllSectionsByGameID(game.ID);
                }
                catch (Exception exc)
                {
                    Logger.Error(exc, "Failed to get category sections.");
                }
            }
            #endregion

            #region Game Detection Hints

            if (game.SupportsAddons)
            {
                using (var command = new SqlCommand("spGetGameDetectionHints", pConn))
                {
                    command.CommandType = CommandType.StoredProcedure;
                    command.Parameters.Add(new SqlParameter("@intGameId", SqlDbType.Int));
                    command.Parameters["@intGameId"].Value = game.ID;

                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            var hint = new GameDetectionHint();
                            hint.ID = (int) reader["ID"];
                            hint.HintType = (HintType) reader["HintType"];
                            hint.HintPath = (string) reader["HintPath"];
                            hint.HintKey = (string) reader["HintKey"];
                            game.GameDetectionHints.Add(hint);
                        }
                    }
                }
            }

            #endregion

            #region File Parsing Rules

            if (game.SupportsAddons)
            {
                using (var command = new SqlCommand("spGetFileParsingRules", pConn))
                {
                    command.CommandType = CommandType.StoredProcedure;
                    command.Parameters.Add(new SqlParameter("@intGameId", SqlDbType.Int));
                    command.Parameters["@intGameId"].Value = game.ID;

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            var rule = new GameFileParsingRule();

                            rule.FileExtension = (string) reader["FileExtension"];
                            rule.InclusionPattern = (string) reader["InclusionPattern"];
                            if (reader["CommentStripPattern"] != System.DBNull.Value)
                            {
                                rule.CommentStripPattern = (string) reader["CommentStripPattern"];
                            }
                            game.FileParsingRules.Add(rule);
                        }
                    }
                }
            }
            #endregion

            #region Addon Settings

            game.ProfilerAddOnId = pReader.GetNullableValue<int>("ProfilerAddOnId");
            game.AddOnSettingsStartingFolder = pReader.GetNullableValue<string>("AddOnSettingsStartingFolder");
            game.AddOnSettingsFolderFilter = pReader.GetNullableValue<string>("AddOnSettingsFolderFilter");
            game.AddOnSettingsFileFilter = pReader.GetNullableValue<string>("AddOnSettingsFileFilter");
            game.AddOnSettingsFileRemovalFilter = pReader.GetNullableValue<string>("AddOnSettingsFileRemovalFilter");
            #endregion
        }

        public static Bitmap LoadImageNoLock(string path)
        {
            var ms = new MemoryStream(File.ReadAllBytes(path)); // Don't use using!!            
            return (Bitmap)Bitmap.FromStream(ms);
        }
    }
}
