﻿using System;
using System.Configuration;
using System.ServiceModel;
using Curse.AddOnService.Extensions;
using Curse.AddOnService.GameManagement;
using Curse.AddOnService.Responses;
using Curse.Logging;

namespace Curse.AddOnService
{
    // NOTE: You can use the "Rename" command on the "Refactor" menu to change the class name "GameManagementService" in code, svc and config file together.
    // NOTE: In order to launch WCF Test Client for testing this service, please select GameManagementService.svc or GameManagementService.svc.cs at the Solution Explorer and start debugging.
    [ServiceBehavior]
    public class GameManagementService : IGameManagementService
    {
        public GetAllGamesResponse GetAllGames(string apiKey)
        {
            try
            {
                VerifyApiKey(apiKey);
                return new GetAllGamesResponse
                {
                    Status = BasicServiceResponseStatus.Successful,
                    Games = Game.GetAll()
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex,"[GetAllGames] Failed to retrieve games from the database.");
                return new GetAllGamesResponse {Status = BasicServiceResponseStatus.Error};
            }
        }

        public GetGameDetailsResponse GetGameDetails(int gameID, string apiKey)
        {
            try
            {
                VerifyApiKey(apiKey);
                if (gameID < 1)
                {
                    return new GetGameDetailsResponse {Status = BasicServiceResponseStatus.Invalid};
                }

                var game = Game.Get(gameID);
                if (game == null)
                {
                    return new GetGameDetailsResponse {Status = BasicServiceResponseStatus.NotFound};
                }

                return new GetGameDetailsResponse
                {
                    Status = BasicServiceResponseStatus.Successful,
                    Game = game,
                    GameFiles = GameFile.GetAllForGame(gameID),
                    GameDetectionHints = GameDetectionHint.GetAllForGame(gameID)
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "");
                return new GetGameDetailsResponse {Status = BasicServiceResponseStatus.Error};
            }
        }

        public BasicServiceResponse CreateGame(Game newGame, string apiKey)
        {
            try
            {
                VerifyApiKey(apiKey);
                if (newGame.ID < 1)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                newGame.SaveToDatabase();
                return new BasicServiceResponse
                {
                    Status = BasicServiceResponseStatus.Successful,
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[CreateGame] Unexpected exception");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public BasicServiceResponse UpdateGame(Game updatedGame, string apiKey)
        {
            try
            {
                VerifyApiKey(apiKey);
                if (updatedGame.ID < 1)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                updatedGame.SaveToDatabase();
                return new BasicServiceResponse {Status = BasicServiceResponseStatus.Successful};
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[CreateGame] Unexpected exception");
                return new CreateResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public CreateResponse CreateGameDetectionHint(GameDetectionHint newGameDetectionHint, string apiKey)
        {
            try
            {
                VerifyApiKey(apiKey);
                if (newGameDetectionHint.ID > 0 || newGameDetectionHint.HintType == HintType.Unknown)
                {
                    return new CreateResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                var game = Game.Get(newGameDetectionHint.GameID);
                if (game == null)
                {
                    return new CreateResponse {Status = BasicServiceResponseStatus.NotFound};
                }

                newGameDetectionHint.SaveToDatabase();
                return new CreateResponse
                {
                    Status = BasicServiceResponseStatus.Successful,
                    ID = newGameDetectionHint.ID
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[CreateGame] Unexpected exception");
                return new CreateResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public BasicServiceResponse UpdateGameDetectionHint(GameDetectionHint updatedGameDetectionHint, string apiKey)
        {
            try
            {
                VerifyApiKey(apiKey);
                if (updatedGameDetectionHint.ID < 1 || updatedGameDetectionHint.HintType == HintType.Unknown)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                var game = Game.Get(updatedGameDetectionHint.GameID);
                if (game == null)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };
                }

                updatedGameDetectionHint.SaveToDatabase();
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[CreateGame] Unexpected exception");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public BasicServiceResponse DeleteGameDetectionHint(int hintID, string apiKey)
        {
            try
            {
                VerifyApiKey(apiKey);
                if (hintID < 0)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                var hint = GameDetectionHint.Get(hintID);
                if (hint == null)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };
                }

                hint.Delete();
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[DeleteGameFile] Unexpected Exception.");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public CreateResponse CreateGameFile(GameFile newGameFile, string apiKey)
        {
            try
            {
                VerifyApiKey(apiKey);
                if (newGameFile.ID > 0 || newGameFile.FileType == GameFileType.Unknown || newGameFile.PlatformType == GamePlatformType.Unknown)
                {
                    return new CreateResponse {Status = BasicServiceResponseStatus.Invalid};
                }

                var game = Game.Get(newGameFile.GameID);
                if (game == null)
                {
                    return new CreateResponse { Status = BasicServiceResponseStatus.NotFound };
                }

                newGameFile.SaveToDatabase();
                return new CreateResponse
                {
                    Status = BasicServiceResponseStatus.Successful,
                    ID = newGameFile.ID
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[CreateGame] Unexpected exception");
                return new CreateResponse {Status = BasicServiceResponseStatus.Error};
            }
        }

        public BasicServiceResponse UpdateGameFile(GameFile updatedGameFile, string apiKey)
        {
            try
            {
                VerifyApiKey(apiKey);
                if (updatedGameFile.ID < 1 || updatedGameFile.FileType == GameFileType.Unknown || updatedGameFile.PlatformType == GamePlatformType.Unknown)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                var game = Game.Get(updatedGameFile.GameID);
                if (game == null)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };
                }

                updatedGameFile.SaveToDatabase();
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[CreateGame] Unexpected exception");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public BasicServiceResponse DeleteGameFile(int fileID, string apiKey)
        {
            try
            {
                VerifyApiKey(apiKey);
                if (fileID < 0)
                {
                    return new BasicServiceResponse {Status = BasicServiceResponseStatus.Invalid};
                }

                var file = GameFile.Get(fileID);
                if (file == null)
                {
                    return new BasicServiceResponse {Status = BasicServiceResponseStatus.NotFound};
                }

                file.Delete();
                return new BasicServiceResponse {Status = BasicServiceResponseStatus.Successful};
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[DeleteGameFile] Unexpected Exception.");
                return new BasicServiceResponse {Status = BasicServiceResponseStatus.Error};
            }
        }

        private void VerifyApiKey(string apiKey)
        {
            if (apiKey != ConfigurationManager.AppSettings["GameManagementApiKey"])
            {
                var ipAddress = OperationContext.Current.GetClientIPAddress();
                Logger.Error("API Key Restricted operation attempted without the correct API Key", new {IPAddress = ipAddress.ToString()});
                throw new InvalidOperationException("API Key Restricted operation attempted without the correct API Key");
            }
        }
    }
}
