﻿using System.Diagnostics;
using Curse.CloudServices;
using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Configuration;
using System.IO;
using Curse.Database.Helpers;
using Curse.Logging.Uploader;
using Curse.ServiceEncryption;
using System.Runtime.Caching;

namespace Curse.AddOnService
{
    public class Global : System.Web.HttpApplication
    {

        private static readonly ObjectCache TokenCache = MemoryCache.Default;

        private class CachedAuthTokenStorageProvider : IAuthTokenStorageProvider
        {
            private static string GetCacheKey(string token)
            {
                return "AuthToken:" + token;
            }

            public AuthenticationTokenData FindToken(string token)
            {
                var cacheKey = GetCacheKey(token);
                AuthenticationTokenData value = null;
                if (TokenCache.Contains(cacheKey))
                {
                    value = TokenCache.Get(cacheKey) as AuthenticationTokenData;
                }

                return value;
            }

            public void StoreToken(string token, AuthenticationTokenData data)
            {
                var cacheKey = GetCacheKey(token);
                TokenCache.Set(cacheKey, data, new DateTimeOffset(DateTime.UtcNow.AddMonths(1)));
            }
        }

        protected void Application_Start(object sender, EventArgs e)
        {
            
            var sw = Stopwatch.StartNew();
            
            Logger.Init(ConfigurationManager.AppSettings["LogPath"], "RadiumService");
            Logger.Info("AddOn Service Starting...");
            LogUploader.Initialize(12, ConfigurationManager.AppSettings["LogServiceUrl"], ConfigurationManager.AppSettings["LogServiceApiKey"]);

            try
            {
                AuthenticationConfiguration.ApiKey = ConfigurationManager.AppSettings["ApiKey"];
                AuthenticationConfiguration.TokenLifespan = TimeSpan.FromSeconds(int.Parse(ConfigurationManager.AppSettings["EncryptionTokenLifespan"]));                
                EncryptionToken.Initialize(ConfigurationManager.AppSettings["EncryptionTokenKey"], int.Parse(ConfigurationManager.AppSettings["EncryptionTokenIterations"]));
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to initialize authentication configuration!");
            }

            try
            {                
                //Initialize the directorys
                var feedDirectory = ConfigurationManager.AppSettings["FeedPath"];                    
                var userBackupDirectory = ConfigurationManager.AppSettings["UserBackupsPath"];


                Logger.Info("Checking feed directory at: " + feedDirectory);

                if (!Directory.Exists(feedDirectory))
                {
                    Directory.CreateDirectory(feedDirectory);
                    Logger.Info("Creating Feed Directory...");
                }

                Logger.Info("Checking user backup directory at: " + userBackupDirectory);
                if (!Directory.Exists(userBackupDirectory))
                {
                    Directory.CreateDirectory(userBackupDirectory);
                    Logger.Info("Creating User Backup Directory...");
                }
                    
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "File IO could not be established!");                 
            }

            try
            {
                CdnCacheInvalidator.Initialize();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to initialize cache invalidation APIs");
            }

            try
            {
                DatabaseConnectionHelper.Initialize(new Dictionary<DatabaseType, string>
                {
                    {DatabaseType.Elerium, ConfigurationManager.ConnectionStrings["Elerium"].ConnectionString},
                    {DatabaseType.RadiumService, ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString}
                });

                Logger.Info("Initializing Avatars...");
                AvatarCache.Instance.Initialize();

                Logger.Info("Initializing Category Sections...");
                CategorySectionCache.Instance.Initialize();

                Logger.Info("Initializing Games...");
                GameCache.Instance.Initialize();

                Logger.Info("Initializing Categories...");
                CategoryCache.Instance.Initialize();

                Logger.Info("Initializing Sync...");
                AddOnSyncCache.Instance.Initialize();

                Logger.Info("Initializing Addon Cache...");
                AddOnCache.Instance.Initialize();

                Logger.Info("Initializing Minecraft Cache...");
                MinecraftCache.Instance.Initialize();                                         
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Could not start addon Service!");
                throw;
            }
            
            Logger.Info("Services Started in " + sw.ElapsedMilliseconds + " milliseconds");
        }

        protected void Session_Start(object sender, EventArgs e)
        {

        }

        protected void Application_BeginRequest(object sender, EventArgs e)
        {

        }

        protected void Application_AuthenticateRequest(object sender, EventArgs e)
        {

        }

        protected void Application_Error(object sender, EventArgs e)
        {

        }

        protected void Session_End(object sender, EventArgs e)
        {

        }

        protected void Application_End(object sender, EventArgs e)
        {            
            Logger.Info("Service Stopping");
            LogUploader.Shutdown();            
        }
    }
}