﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Caching;
using System.Web;
using System.ServiceModel;

namespace Curse.Caching
{
    [CallbackBehavior(UseSynchronizationContext = false)]
    public static class CacheManager
    {
        private static readonly TimeSpan _defaultDuration = TimeSpan.FromHours(1);

        public static void Add(string key, object value, TimeSpan? cacheDuration = null, bool useAbsoluteDuration = false, IEnumerable<string> dependencyKeys = null, CacheItemPriority cacheItemPriority = CacheItemPriority.Normal)
        {
            if (value == null)
            {
                return;
            }
            CacheDependency dep = null;
            Cache cache = HttpRuntime.Cache;
            if (dependencyKeys != null)
            {
                string[] dependencies = dependencyKeys as string[] ?? dependencyKeys.ToArray();
                if (dependencies.Length > 0)
                {
                    AddDependencyObjects(dependencies);
                    dep = new CacheDependency(null, dependencies);
                }
            }

            TimeSpan duration = cacheDuration.HasValue ? cacheDuration.Value : _defaultDuration;
            DateTime absoluteExpiration = useAbsoluteDuration ? DateTime.UtcNow + duration : Cache.NoAbsoluteExpiration;
            TimeSpan slidingExpiration = useAbsoluteDuration ? Cache.NoSlidingExpiration : duration;

            cache.Insert(key, value, dep, absoluteExpiration, slidingExpiration, cacheItemPriority, null);
        }

        private static void AddDependencyObjects(string[] keys)
        {
            foreach (string key in keys)
            {
                AddDependencyObject(key);
            }
        }

        private static void AddDependencyObject(string key)
        {
            if (HttpRuntime.Cache.Get(key) == null)
            {
                return;
            }

            HttpRuntime.Cache.Add(key, new object(), null, Cache.NoAbsoluteExpiration, Cache.NoSlidingExpiration, CacheItemPriority.NotRemovable, null);
        }

        public static bool Exists(string key)
        {
            object result = HttpRuntime.Cache.Get(key);
            return result != null;
        }


        public static bool TryGet<T>(string key, out T value)
        {
            object result = HttpRuntime.Cache.Get(key);
            if (result == null)
            {
                value = default(T);
                return false;
            }
            else if (result is T)
            {
                value = (T)result;
                return true;
            }
            else
            {
                value = default(T);
                return false;
            }

        }

        public static T Get<T>(string key)
        {
            T result;
            if (TryGet<T>(key, out result))
            {
                return result;
            }
            else
            {
                return default(T);
            }
        }

        public static T GetOrAdd<T>(string key, Func<T> func, TimeSpan? cacheDuration = null, string[] dependencyKeys = null, CacheItemPriority priority = CacheItemPriority.Normal)
        {
            T value = default(T);
            if (!TryGet<T>(key, out value))
            {
                value = func();
                Add(key, value, cacheDuration ?? _defaultDuration, false, dependencyKeys, priority);
            }
            return value;
        }

        public static void Remove(string key)
        {
            HttpRuntime.Cache.Remove(key);
        }

        public static void Remove(string[] keys)
        {
            foreach (string key in keys)
            {
                Remove(key);
            }
        }

        public static void Expire(string key)
        {

            Remove(key);

            // Transmit removal to other caches in the cluster
            CacheCluster.Instance.TransmitInvalidationKey(key);

        }

        public static void Expire(string[] keys)
        {
            Remove(keys);

            // Transmit removal to other caches in the cluster
            CacheCluster.Instance.TransmitInvalidationKeys(keys);
        }

        public static void ExpireRemotely(string key)
        {
            // Transmit removal to other caches in the cluster
            CacheCluster.Instance.TransmitInvalidationKey(key);
        }

        public static void ExpireRemotely(string[] keys)
        {
            // Transmit removal to other caches in the cluster
            CacheCluster.Instance.TransmitInvalidationKeys(keys);
        }
    }
}
