﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.SqlClient;
using System.Net;
using System.ServiceModel;

namespace Curse.Caching
{

    public class ClusterManager
    {
        public static readonly ClusterManager Instance = new ClusterManager();
        public Dictionary<int, ClusterMember> Members { get; private set; }
        public ClusterMember Self { get; private set; }
        private string _connectionString = null;

        public ClusterManager()
        {
            Members = new Dictionary<int, ClusterMember>();
        }

        public void Initialize(string applicationName, SqlConnection conn, bool enableLogging = false)
        {            
            if (!System.Net.NetworkInformation.NetworkInterface.GetIsNetworkAvailable())
            {
                throw new InvalidOperationException("This host cannot participate in a cluster, as it has no network available.");
            }

            if (enableLogging)
            {
                CacheLog.Initialize();
            }

            _connectionString = conn.ConnectionString;
            
            IPHostEntry host = Dns.GetHostEntry(Dns.GetHostName());

            CacheLog.Info("Retrieved host name from DNS: '{0}'", host);

            SqlCommand cmd = conn.CreateCommand();
            cmd.CommandText = "select * from ClusterMember";
            using (SqlDataReader reader = cmd.ExecuteReader())
            {
                while (reader.Read())
                {
                    ClusterMember member = new ClusterMember(reader);
                    Members.Add(member.ID, member);
                }
            }

            int? maxPort = Members.Any() ? (int?)Members.Values.Max(p => p.Port) : null;

            Self = Members.Values.FirstOrDefault(p => p.ApplicationIdentity.Equals(applicationName, StringComparison.InvariantCultureIgnoreCase) && p.HostName == host.HostName); 
            if (Self == null)
            {
                ClusterMember member = new ClusterMember();
                member.ApplicationIdentity = applicationName;
                member.HostName = host.HostName;
                member.Port = maxPort.HasValue ? maxPort.Value + 1 : 8080;
                member.Save(conn);
                Members.Add(member.ID, member);
                Self = member;
            }

            CacheLog.Info("Registered with cluster on port '{0}' with application identity '{1}'", Self.Port, Self.ApplicationIdentity);
        }

        public void AddMember(int id)
        {
            if (Members.ContainsKey(id))
            {
                return;                
            }

            using (SqlConnection conn = new SqlConnection(_connectionString))
            {
                conn.Open();
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select * from ClusterMember where @ID = ID";
                cmd.Parameters.AddWithValue("@ID", id);

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        
                        ClusterMember member = new ClusterMember(reader);
                        Members.Add(member.ID, member);
                    }
                }
            }            
        }
    }
}
