﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web;
using Curse.Radon.DownloadTracker.Configuration;
using System.Data.SqlClient;
using System.Net;
using System.Net.Sockets;
using System.Data;
using System.Threading;
using System.Data.SqlTypes;
using NLog;
using System.Collections.Specialized;

namespace Curse.Radon.DownloadTracker
{
	public class DownloadTracker : IHttpModule
	{
		private HttpApplication _currentApplication;
		
        private static Dictionary<string, Tuple<IPAddress, int, DateTime, string, string, int?>> _downloadedFileIDs = 
            new Dictionary<string, Tuple<IPAddress, int, DateTime, string, string, int?>>();
		
        private static readonly object _syncRoot = new object();
		private static bool _initialized = false;
		private static string _tempTableName = "dbo.#ProjectFileDownloadTemp";
		private static Logger _logger = LogManager.GetCurrentClassLogger();
		private static bool _loggingEnabled = DownloadTrackerConfiguration.Database.LoggingEnabled;
		private static int _restTimeSeconds = DownloadTrackerConfiguration.Database.WriteRestTimeSeconds;

		void TrackFileDownload(object sender, EventArgs e)
		{
			try
			{
				HttpApplication app = (HttpApplication)sender;
				HttpContext context = app.Context;
				IEnumerable<string> headerKeys = context.Request.Headers.AllKeys;

				// Requester is likely a download manager.
				if (headerKeys.Contains("Range")
					|| headerKeys.Contains("Want-Digest")) return;

			    string userAgent = context.Request.UserAgent ?? String.Empty;

			    string referrer = context.Request.UrlReferrer.NullSafeToString();

				string filePath = context.Request.FilePath;

				if (!filePath.Contains("files")) return;

				// Get the FileID
				string[] parts = filePath.Split('/');
				if (parts.Length != 5) return;

				string highIDString = parts[2];
				string lowIDString = parts[3];
				int highID = -1;
				int lowID = -1;
				int fileID = -1;
				if (int.TryParse(highIDString, out highID) && int.TryParse(lowIDString, out lowID))
				{
					fileID = (highID * 1000) + lowID;
				}

				IPAddress address = GetClientIPAddress(context.Request);

			    int? sourceFileID = null;
			    if (context.Request.Headers["x-elerium-packfileid"] != null)
			    {
			        var packValue = context.Request.Headers["x-elerium-packfileid"];
                    int temp;
			        Int32.TryParse(packValue, out temp);
			        sourceFileID = temp;
			    }

			    if (fileID > 0)
				{
					string key = fileID + "-" + address;

					// We have a file ID, let's try to track the download.
					lock (_syncRoot)
					{
						if (!_downloadedFileIDs.ContainsKey(key))
						{
							_downloadedFileIDs.Add(key, new Tuple<IPAddress, int, DateTime, string, string, int?>(address, fileID, DateTime.UtcNow, referrer, userAgent, sourceFileID));
						}
					}
				}
			}
			catch (Exception ex)
			{
				if (_loggingEnabled)
				{
					_logger.LogException(LogLevel.Error, ex.StackTrace, ex);
				}
			}
		}

		private void InsertProjectFileDownloads()
		{
			while (true)
			{
			    try
			    {
			        SaveToDatabase();
			    }
			    catch (Exception ex)
			    {
                    if (_loggingEnabled)
                    {
                        _logger.LogException(LogLevel.Error, "General uncaught error: " + ex.StackTrace, ex);
                    }
			    }

				Thread.Sleep(TimeSpan.FromSeconds(_restTimeSeconds));
			}
		}


        void SaveToDatabase()
        {
            int threadID = (int)Thread.CurrentThread.ManagedThreadId;
            Dictionary<string, Tuple<IPAddress, int, DateTime, string, string, int?>> projectFileDownloads = new Dictionary<string, Tuple<IPAddress, int, DateTime, string, string, int?>>();
            int rowCount = -1;
            lock (_syncRoot)
            {
                projectFileDownloads = new Dictionary<string, Tuple<IPAddress, int, DateTime, string, string, int?>>(_downloadedFileIDs);
                _downloadedFileIDs.Clear();
            }

            if (projectFileDownloads.Count > 0)
            {
                using (SqlConnection conn = new SqlConnection(DownloadTrackerConfiguration.ConnectionString))
                {
                    try
                    {
                        conn.Open();

                        // Create the temp table.
                        CreateTempTable(conn, threadID);

                        // Load its schema.
                        string fullTableName = GetTempTableName(conn, threadID);
                        DataTable table = new DataTable();
                        using (SqlCommand command = new SqlCommand(string.Format("SELECT TOP 1 * FROM {0}", fullTableName), conn))
                        {
                            using (SqlDataReader reader = command.ExecuteReader(CommandBehavior.SchemaOnly))
                            {
                                table.Load(reader);
                            }
                        }

                        // Populate DataTable.
                        foreach (KeyValuePair<string, Tuple<IPAddress, int, DateTime, string, string, int?>> t in projectFileDownloads)
                        {
                            DataRow row = table.NewRow();
                            row["IPAddress"] = t.Value.Item1.GetAddressBytes();
                            row["ProjectFileID"] = t.Value.Item2;
                            row["DateDownloaded"] = t.Value.Item3;
                            row["Referrer"] = t.Value.Item4;
                            row["UserAgent"] = t.Value.Item5;
                            if (t.Value.Item6 != null)
                            {
                                row["SourceFileID"] = t.Value.Item6;    
                            }
                            table.Rows.Add(row);
                        }

                        using (SqlTransaction transaction = conn.BeginTransaction())
                        {
                            try
                            {
                                // Bulk Copy into temp table.
                                using (SqlBulkCopy bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.Default,
                                    transaction))
                                {
                                    try
                                    {
                                        bulk.BatchSize = 5000;
                                        bulk.BulkCopyTimeout = 5000;
                                        bulk.DestinationTableName = fullTableName;
                                        bulk.WriteToServer(table);
                                    }
                                    catch (Exception ex)
                                    {
                                        if (_loggingEnabled)
                                        {
                                            _logger.LogException(LogLevel.Error, "Error with bulk write: " + ex.StackTrace, ex);
                                        }
                                    }
                                    finally
                                    {
                                        bulk.Close();
                                    }
                                }

                                try
                                {
                                    // Insert into ProjectFileDownload from temp table joining on ProjectFile to ensure proper tracking.
                                    string tableName = string.Format("[{0}].[dbo].[FileDownload]", conn.Database);
                                    string commandText = "INSERT INTO " + tableName + "(IPAddress, ProjectFileID, DateDownloaded, Referrer, UserAgent, SourceFileID) "
                                                         + "SELECT pt.[IPAddress], pt.[ProjectFileID], pt.[DateDownloaded], pt.[Referrer], pt.[UserAgent], pt.[SourceFileID]"
                                                         + "FROM " + GetTempTableName(conn, threadID) + " pt ";
                                    using (SqlCommand command = new SqlCommand(commandText, conn, transaction))
                                    {
                                        rowCount = command.ExecuteNonQuery();
                                    }
                                }
                                catch (Exception ex)
                                {
                                    transaction.Rollback();
                                    if (_loggingEnabled)
                                    {
                                        _logger.LogException(LogLevel.Error, "Error with copy from temp table: " + ex.StackTrace, ex);

                                        SqlException sqlEx = ex as SqlException;
                                        if (sqlEx != null)
                                        {
                                            _logger.LogException(LogLevel.Error, "Sql Error: " + sqlEx.Message, sqlEx);
                                        }
                                    }
                                }

                                transaction.Commit();
                            }
                            catch (Exception ex)
                            {
                                transaction.Rollback();
                                if (_loggingEnabled)
                                {
                                    _logger.LogException(LogLevel.Error, "Error with transaction: " + ex.StackTrace, ex);

                                    SqlException sqlEx = ex as SqlException;
                                    if (sqlEx != null)
                                    {
                                        _logger.LogException(LogLevel.Error, "Sql Error: " + sqlEx.Message, sqlEx);
                                    }
                                }
                            }
                        }

                        // If we didn't insert the expected number, log the bad ones.
                        if (rowCount != projectFileDownloads.Count && _loggingEnabled)
                        {
                            string commandText = "SELECT * FROM " + GetTempTableName(conn, threadID) + " pt "
                                + "LEFT JOIN [File] pf ON pt.[ProjectFileID] = pf.[ID] WHERE pf.[ID] IS NULL";
                            using (SqlCommand command = new SqlCommand(commandText, conn))
                            {
                                StringBuilder sb = new StringBuilder();
                                using (SqlDataReader reader = command.ExecuteReader())
                                {
                                    if (reader.HasRows)
                                    {
                                        bool readRows = false;
                                        while (reader.Read())
                                        {
                                            byte[] ipAddressBytes = new byte[16];
                                            reader.GetBytes(reader.GetOrdinal("IPAddress"), 0, ipAddressBytes, 0, 16);
                                            IPAddress ipAddress = new IPAddress(ipAddressBytes);
                                            if (reader["ProjectFileID"] != null)
                                            {
                                                readRows = true;
                                            }
                                            sb.AppendLine("ProjectFileID: " + reader["ProjectFileID"] + ", IPAddress: " + ipAddress + ", DateDownloaded: " + reader["DateDownloaded"] + ", Referrer: " + reader["Referrer"] + ", UserAgent: " + reader["UserAgent"] + ", SourceFileID: " + reader["SourceFileID"]);
                                        }
                                        if (readRows)
                                        {
                                            StringBuilder error = new StringBuilder();
                                            error.AppendLine("Downloads were attempted for the following:");
                                            _logger.Log(LogLevel.Error, error.ToString() + sb);
                                        }
                                    }
                                }
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        if (_loggingEnabled)
                        {
                            _logger.LogException(LogLevel.Error, "General uncaught error: " + ex.StackTrace, ex);
                        }
                    }
                    finally
                    {
                        if (conn.State != ConnectionState.Closed)
                        {
                            try
                            {
                                // Try to drop the temp table. It might not have been created if we arrived here from an exception.
                                DropTempTable(conn, threadID);
                            }
                            catch { }
                            conn.Close();
                        }
                    }
                }
            }

        }

		private string GetTempTableName(SqlConnection conn, int threadID)
		{
			return _tempTableName + threadID;
		}

		private void CreateTempTable(SqlConnection conn, int threadID)
		{
            using (SqlCommand command = new SqlCommand("CREATE TABLE " + GetTempTableName(conn, threadID) + " (IPAddress varbinary(16) NOT NULL, ProjectFileID int NOT NULL, DateDownloaded datetime NOT NULL, Referrer varchar(4000), UserAgent varchar(4000), SourceFileID int NULL)", conn))
			{
				command.ExecuteNonQuery();
			}
		}

		private void DropTempTable(SqlConnection conn, int threadID)
		{
			using (SqlCommand command = new SqlCommand("DROP TABLE " + GetTempTableName(conn, threadID), conn))
			{
				command.ExecuteNonQuery();
			}
		}

		private IPAddress GetClientIPAddress(HttpRequest request)
		{
			string ipString = string.Empty;
			if (request.Headers != null)
			{
				ipString = request.Headers["X-Forwarded-For"];
			}
			if (string.IsNullOrEmpty(ipString))
			{
				ipString = request.UserHostAddress;
			}

			if (ipString.Contains(","))
			{
				string[] ipArray = ipString.Split(',');
				ipString = ipArray[ipArray.Length - 1].Trim();
			}

			IPAddress ipAddress;
			if (IPAddress.TryParse(ipString, out ipAddress))
			{
				AddressFamily family = ipAddress.AddressFamily;
				if (family == AddressFamily.InterNetwork || family == AddressFamily.InterNetworkV6)
				{
					byte[] bytes = ipAddress.GetAddressBytes();
					if (bytes != null && bytes.Length > 0)
					{
						return ipAddress;
					}
				}
			}
			return null;
		}

		#region IHttpModule Implementation

		public void Dispose()
		{
			_currentApplication = null;
		}

		public void Init(HttpApplication context)
		{
			if (!_initialized)
			{
				lock (_syncRoot)
				{
					if (!_initialized)
					{
						try
						{
                            new Thread(InsertProjectFileDownloads)
							{
							    IsBackground = true,
                                Name = "DownloadTracker"
							}.Start();
							_logger.Log(LogLevel.Info, "Download tracker started.");
							_initialized = true;
						}
						catch (Exception ex)
						{
							if (_loggingEnabled)
							{
								_logger.LogException(LogLevel.Error, ex.StackTrace, ex);
							}
						}
					}
				}
			}

			if (_initialized)
			{
				_currentApplication = context;
				_currentApplication.BeginRequest += new EventHandler(this.TrackFileDownload);
			}
		}

		#endregion
	}
}
