﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.Serialization;
using System.ComponentModel;

namespace Curse.AddOns
{
    [Serializable]
    [DataContract(Namespace = "Curse.AddOns")]
    public class AddOn : INotifyPropertyChanged
    {

        #region Events

        [field: NonSerialized]
        public event PropertyChangedEventHandler PropertyChanged;

        protected void OnPropertyChanged(string name)
        {
            PropertyChangedEventHandler handler = PropertyChanged;
            if (handler != null)
            {
                handler(this, new PropertyChangedEventArgs(name));
            }
        }

        #endregion

        #region Data Members

        [DataMember]
        public int Id
        {
            get;
            set;
        }

        [DataMember]
        public string Name
        {
            get;
            set;
        }

        [DataMember]
        public List<AddOnAuthor> Authors
        {
            get;
            set;
        }

        [DataMember]
        public List<AddOnAttachment> Attachments
        {
            get;
            set;
        }
        
        public AddOnAttachment DefaultAttachment
        {
            get
            {
                var defaultAttachment = Attachments != null ? Attachments.FirstOrDefault(p => p.IsDefault) : null;
                return defaultAttachment ?? null;
            }
        }



        [DataMember]
        public string WebSiteURL
        {
            get;
            set;
        }

        [DataMember]
        public int GameId
        {
            get;
            set;
        }

        [DataMember]
        public string Summary
        {
            get;
            set;
        }

        [DataMember]
        public int DefaultFileId
        {
            get;
            set;
        }

        [DataMember]
        public int CommentCount
        {
            get;
            set;
        }

        [DataMember]
        public double DownloadCount
        {
            get;
            set;
        }

        [DataMember]
        public int Rating
        {
            get;
            set;
        }

        [DataMember]
        public int InstallCount
        {
            get;
            set;
        }

        [DataMember]
        public int IconId
        {
            get;
            set;
        }

        [DataMember]
        public List<AddOnFile> LatestFiles
        {
            get;
            set;
        }

        [DataMember]
        public List<AddOnCategory> Categories
        {
            get;
            set;
        }

        [DataMember]
        public string PrimaryAuthorName
        {
            get;
            set;
        }

        [DataMember]
        public string ExternalUrl
        {
            get;
            set;
        }

        [DataMember]
        public ProjectStatus Status
        {
            get;
            set;
        }

        [DataMember]
        public ProjectStage Stage
        {
            get;
            set;
        }

        [DataMember]
        public string DonationUrl
        {
            get;
            set;
        }

        [DataMember]
        public int PrimaryCategoryId
        {
            get
            {
                if (mPrimaryCategoryId == 0)
                {
                    if (Categories != null)
                    {
                        var first = Categories.FirstOrDefault();
                        mPrimaryCategoryId = first != null ? first.Id : 0;
                    }
                }
                return mPrimaryCategoryId;
            }
            set
            {
                mPrimaryCategoryId = value;
            }
        }


        [DataMember]
        public string PrimaryCategoryName
        {
            get;
            set;
        }

        [DataMember]
        public string PrimaryCategoryAvatarUrl
        {
            get;
            set;
        }

        [DataMember]
        public int Likes { get; set; }

        [DataMember]
        public CategorySection CategorySection
        {
            get;
            set;
        }

        [DataMember]
        public PackageTypes PackageType
        {
            get
            {
                if (CategorySection != null)
                {
                    return CategorySection.PackageType;
                }
                else return PackageTypes.Folder;
            }
            set{
                if (CategorySection != null)
                {
                    CategorySection.PackageType = value;
                }
            }
        }

        [DataMember]
        public string AvatarUrl
        {
            get;
            set;
        }

        public string Slug
        {
            get
            {
                return mSlug;
            }
            set
            {
                mSlug = value;
            }
        }

        public string ClientUrl
        {
            get
            {
                if (ExternalUrl != null)
                {
                    return ExternalUrl;
                }
                return WebSiteURL;
            }
        }

        [DataMember]
        public List<GameVersionLatestFile> GameVersionLatestFiles
        {
            get;
            set;
        }

        [DataMember]
        public int IsFeatured { get; set; }

        [DataMember]
        public double PopularityScore { get; set; }

        [DataMember]
        public int GamePopularityRank { get; set; }

        #endregion

        #region Non Serialized Fields

        [NonSerialized]
        private string mFullDescription;

        [NonSerialized]
        private string mGameName;

        [NonSerialized]
        private string mPortalName;

        [NonSerialized]
        private string mSectionName;

        [NonSerialized]
        private string mSlug;

        [NonSerialized]
        private DateTime mDateModified;

        [NonSerialized]
        private DateTime mDateCreated;

        [NonSerialized]
        private DateTime mDateReleased;

        [NonSerialized]
        private bool mIsAvailable;                
        
        [NonSerialized]
        private Dictionary<int, AddOnFile> mFiles; 

        #endregion

        #region Serialized Fields

        private int mPrimaryCategoryId;

        private DateTime mCacheDate;

        #endregion

        #region Properties

        public string FullDescription
        {
            get
            {
                return mFullDescription;
            }
            set
            {
                mFullDescription = value;
            }
        }

        public string GameName
        {
            get
            {
                return mGameName;
            }
            set
            {
                mGameName = value;
            }
        }
        
        public string PortalName
        {
            get
            {
                return mPortalName;
            }
            set
            {
                mPortalName = value;
            }
        }        
       
        public string SectionName
        {
            get
            {
                return mSectionName;
            }
            set
            {
                mSectionName = value;
            }
        }        
                
        public DateTime DateModified
        {
            get
            {
                return mDateModified;
            }
            set
            {
                mDateModified = value;
            }
        }
        
        public DateTime DateCreated
        {
            get
            {
                return mDateCreated;
            }
            set
            {
                mDateCreated = value;
            }
        }
        
        public DateTime DateReleased
        {
            get
            {
                return mDateReleased;
            }
            set
            {
                mDateReleased = value;
            }
        }        
       
        public bool IsAvailable
        {
            get
            {
                return mIsAvailable;
            }
            set
            {
                mIsAvailable = value;
            }
        }                      

        public AddOnFile DefaultFile
        {
            get
            {
                if (LatestFiles == null)
                {
                    return null;
                }
                AddOnFile file = LatestFiles.FirstOrDefault(p => p.Id == DefaultFileId);
                if (file == null)
                {
                    file = LatestFiles.FirstOrDefault();
                }
                return file;
            }
        }

        public DateTime LatestFileDate
        {
            get
            {
                var files = Files;
                var latestFiles = LatestFiles;
                if (files != null && files.Any())
                {
                    return files.Max(f => f.Value.FileDate);
                }
                
                if (latestFiles != null && latestFiles.Any())
                {
                    return latestFiles.Max(f => f.FileDate);
                }
                return DefaultFileDate;
            }
             
        }
                      
        public Dictionary<int, AddOnFile> Files
        {
            get
            {
                return mFiles;
            }
            set
            {
                mFiles = value;
                OnPropertyChanged("Files");
                OnPropertyChanged("LastestFileDate");
            }
        }

        public bool IncludeInFingerprinting
        {
            get
            {
                return Status != ProjectStatus.Deleted;
            }
        }

        public string DefaultFileName
        {
            get
            {
                return DefaultFile.FileName;
            }
        }

        public DateTime DefaultFileDate
        {
            get
            {
                var defaultFile = DefaultFile;

                return (defaultFile != null) ? DefaultFile.FileDate : DateTime.UtcNow;
            }
        }

        public FileType DefaultFileType
        {
            get
            {
                return DefaultFile.ReleaseType;
            }
        }
        
        public string AuthorList
        {
            get
            {
                List<string> authorList = new List<string>();
                foreach (AddOnAuthor author in Authors)
                {
                    authorList.Add(author.Name);
                }
                return string.Join(", ", authorList.ToArray());
            }
        }

        public AddOnAuthor[] NonPrimaryAuthors
        {
            get { return Authors != null ? Authors.Skip(1).ToArray() : new AddOnAuthor[0]; }
        }

        public string PrimaryAuthor
        {
            get
            {
                if (Authors == null)
                {
                    return null;
                }
                var first = Authors.FirstOrDefault();
                return first != null ? first.Name : null;
            }
        }

        public string CategoryList
        {
            get
            {
                List<string> catList = new List<string>();
                foreach (AddOnCategory cat in Categories)
                {
                    catList.Add(cat.Name);
                }
                return string.Join(", ", catList.ToArray());
            }
        }

        #endregion

        #region Client Only Properties

        public DateTime CacheDate
        {
            get
            {
                return mCacheDate;
            }
            set
            {
                mCacheDate = value;
            }
        }
               
        public bool IsCacheExpired
        {
            get
            {
                return mCacheDate.AddDays(7) < DateTime.UtcNow; // Temporary slug check
            }
        }

        private DateTime _filesCacheDate;
        public DateTime FilesCacheDate
        {
            get
            {
                return _filesCacheDate;
            }
            set
            {
                _filesCacheDate = value;
            }
        }

        public bool IsFilesCacheExpired
        {
            get
            {
                return _filesCacheDate.AddMinutes(10) < DateTime.UtcNow; // Temporary slug check
            }

        }

        private DateTime _descriptionCacheDate;
        public DateTime DescriptionCacheDate
        {
            get
            {
                return _descriptionCacheDate;
            }
            set
            {
                _descriptionCacheDate = value;
            }
        }

        public bool IsDescriptionCacheExpired
        {
            get
            {
                return _descriptionCacheDate.AddMinutes(10) < DateTime.UtcNow; // Temporary slug check
            }

        }

        #endregion

        #region Constructor

        public AddOn()
        {
            IsAvailable = false;
            Status = ProjectStatus.Normal;
            mCacheDate = DateTime.UtcNow;
            LatestFiles = new List<AddOnFile>();
            Categories = new List<AddOnCategory>();
            Files = new Dictionary<int, AddOnFile>();
            Authors = new List<AddOnAuthor>();
        }

        #endregion

        public AddOn Clone()
        {
            return (AddOn)MemberwiseClone();
        }

        public static AddOn Empty = new AddOn
        {
            Id = 0,
            GameVersionLatestFiles = new List<GameVersionLatestFile>(),
            CacheDate = DateTime.UtcNow,
            FilesCacheDate = DateTime.UtcNow,
            Attachments = new List<AddOnAttachment>(),
            IsAvailable = false,
            Status = ProjectStatus.Normal,
            LatestFiles = new List<AddOnFile>(),
            Categories = new List<AddOnCategory>(),
            Files = new Dictionary<int, AddOnFile>(),
            Authors = new List<AddOnAuthor>(),
            CategorySection = new CategorySection(),
        };


    }

}
