﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using System.ComponentModel;

namespace Curse.AddOns
{
    [Serializable]
    [DataContract(Namespace = "Curse.AddOns")]
    public class CAddOn : INotifyPropertyChanged
    {

        #region Events

        [field: NonSerialized]
        public event PropertyChangedEventHandler PropertyChanged;

        protected void OnPropertyChanged(string name)
        {
            PropertyChangedEventHandler handler = PropertyChanged;
            if (handler != null)
            {
                handler(this, new PropertyChangedEventArgs(name));
            }
        }

        #endregion

        #region Data Members

        [DataMember]
        public int Id
        {
            get;
            set;
        }

        [DataMember]
        public string Name
        {
            get;
            set;
        }

        [DataMember]
        public List<CAddOnAuthor> Authors
        {
            get;
            set;
        }

        [DataMember]
        public string WebSiteURL
        {
            get;
            set;
        }

        [DataMember]
        public int GameId
        {
            get;
            set;
        }

        [DataMember]
        public string Summary
        {
            get;
            set;
        }

        [DataMember]
        public int DefaultFileId
        {
            get;
            set;
        }

        [DataMember]
        public int CommentCount
        {
            get;
            set;
        }

        [DataMember]
        public double DownloadCount
        {
            get;
            set;
        }

        [DataMember]
        public int Rating
        {
            get;
            set;
        }

        [DataMember]
        public int InstallCount
        {
            get;
            set;
        }

        [DataMember]
        public int IconId
        {
            get;
            set;
        }

        [DataMember]
        public List<CAddOnFile> LatestFiles
        {
            get;
            set;
        }

        [DataMember]
        public List<CAddOnCategory> Categories
        {
            get;
            set;
        }

        [DataMember]
        public string PrimaryAuthorName
        {
            get;
            set;
        }

        [DataMember]
        public string ExternalUrl
        {
            get;
            set;
        }

        [DataMember]
        public EProjectStatus Status
        {
            get;
            set;
        }

        [DataMember]
        public EProjectStage Stage
        {
            get;
            set;
        }

        [DataMember]
        public string DonationUrl
        {
            get;
            set;
        }

        [DataMember]
        public int PrimaryCategoryId
        {
            get
            {
                if (mPrimaryCategoryId == 0)
                {
                    mPrimaryCategoryId = Categories.FirstOrDefault().Id;
                }
                return mPrimaryCategoryId;
            }
            set
            {
                mPrimaryCategoryId = value;
            }
        }

        [DataMember]
        public int Likes { get; set; }

        [DataMember]
        public CCategorySection CategorySection
        {
            get;
            set;
        }

        [DataMember]
        public EPackageType PackageType
        {
            get
            {
                if (CategorySection != null)
                {
                    return CategorySection.PackageType;
                }
                else return EPackageType.Folder;
            }
            set{
                if (CategorySection != null)
                {
                    CategorySection.PackageType = value;
                }
            }
        }

        public string Slug
        {
            get
            {
                return mSlug;
            }
            set
            {
                mSlug = value;
            }
        }

        public string ClientUrl
        {
            get
            {
                if (ExternalUrl != null)
                {
                    return ExternalUrl;
                }
                return WebSiteURL;
            }
        }

        #endregion

        #region Non Serialized Fields

        [NonSerialized]
        private string mFullDescription;

        [NonSerialized]
        private string mGameName;

        [NonSerialized]
        private string mPortalName;

        [NonSerialized]
        private string mSectionName;

        [NonSerialized]
        private string mSlug;

        [NonSerialized]
        private DateTime mDateModified;

        [NonSerialized]
        private DateTime mDateCreated;

        [NonSerialized]
        private DateTime mDateReleased;

        [NonSerialized]
        private bool mIsAvailable;                
        
        [NonSerialized]
        private Dictionary<int, CAddOnFile> mFiles; 

        #endregion

        #region Serialized Fields

        private int mPrimaryCategoryId;

        private DateTime mCacheDate;

        #endregion

        #region Properties

        public string FullDescription
        {
            get
            {
                return mFullDescription;
            }
            set
            {
                mFullDescription = value;
            }
        }

        public string GameName
        {
            get
            {
                return mGameName;
            }
            set
            {
                mGameName = value;
            }
        }
        
        public string PortalName
        {
            get
            {
                return mPortalName;
            }
            set
            {
                mPortalName = value;
            }
        }        
       
        public string SectionName
        {
            get
            {
                return mSectionName;
            }
            set
            {
                mSectionName = value;
            }
        }        
                
        public DateTime DateModified
        {
            get
            {
                return mDateModified;
            }
            set
            {
                mDateModified = value;
            }
        }
        
        public DateTime DateCreated
        {
            get
            {
                return mDateCreated;
            }
            set
            {
                mDateCreated = value;
            }
        }
        
        public DateTime DateReleased
        {
            get
            {
                return mDateReleased;
            }
            set
            {
                mDateReleased = value;
            }
        }        
       
        public bool IsAvailable
        {
            get
            {
                return mIsAvailable;
            }
            set
            {
                mIsAvailable = value;
            }
        }                      

        public CAddOnFile DefaultFile
        {
            get
            {
                if (LatestFiles == null)
                {
                    return null;
                }
                CAddOnFile file = LatestFiles.FirstOrDefault(p => p.Id == DefaultFileId);
                if (file == null)
                {
                    file = LatestFiles.FirstOrDefault();
                }
                return file;
            }
        }
                      
        public Dictionary<int, CAddOnFile> Files
        {
            get
            {
                return mFiles;
            }
            set
            {
                mFiles = value;
            }
        }

        public bool IncludeInFingerprinting
        {
            get
            {
                return Status != EProjectStatus.Deleted;
            }
        }

        public string DefaultFileName
        {
            get
            {
                return DefaultFile.FileName;
            }
        }

        public DateTime DefaultFileDate
        {
            get
            {
                return DefaultFile.FileDate;
            }
        }

        public EFileType DefaultFileType
        {
            get
            {
                return DefaultFile.ReleaseType;
            }
        }
        
        public string AuthorList
        {
            get
            {
                List<string> authorList = new List<string>();
                foreach (CAddOnAuthor author in Authors)
                {
                    authorList.Add(author.Name);
                }
                return string.Join(", ", authorList.ToArray());
            }
        }

        public string CategoryList
        {
            get
            {
                List<string> catList = new List<string>();
                foreach (CAddOnCategory cat in Categories)
                {
                    catList.Add(cat.Name);
                }
                return string.Join(", ", catList.ToArray());
            }
        }

        #endregion

        #region Client Only Properties

        public DateTime CacheDate
        {
            get
            {
                return mCacheDate;
            }
            set
            {
                mCacheDate = value;
            }
        }
               
        public bool IsCacheExpired
        {
            get
            {
                return mCacheDate.AddDays(7) < DateTime.UtcNow; // Temporary slug check
            }

        }                          

        #endregion

        #region Constructor

        public CAddOn()
        {
            IsAvailable = false;
            Status = EProjectStatus.Normal;
            mCacheDate = DateTime.UtcNow;
            LatestFiles = new List<CAddOnFile>();
            Categories = new List<CAddOnCategory>();
            Files = new Dictionary<int, CAddOnFile>();
            Authors = new List<CAddOnAuthor>();
        }

        #endregion

        public CAddOn Clone()
        {
            return (CAddOn)MemberwiseClone();
        }

    }

}
