﻿using Curse;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Net;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Configuration;
using System.Web.Services;
using System.Web;

namespace Curse.Auth
{
    /**
     * Class used to handle user and session validation with the central authentication service
     * Internally handles session management
     * Intended to be shared among all services to provide a unified and redundant architecture for validation
     * AuthenticationSession expiration should be short, long enough to handle a sequence of requests based on needs of each service
     * The authentication service maintains long term session keys, and will expire them when passwords change
     *
     * @author Michael Comperda
     */
    public class Authentication
    {
        /**
         * Initialization constructor
         * Initializes the authentication object for use with the central authentication service
         * Asserts on bad parameters
         *
         * @param pServiceUrl        the fully qualified URL to the central authentication service
         * @param pSiteId            the site id to use for access to the central authentication service
         * @param pSiteKey           the site key as a hex string for the cipher with the central authentication service
         * @param pSessionExpiration the length in seconds to allow a session to be locally cached from it's initial validations
         */
        private static bool sFreePremium = false;
        public int SiteId
        {
            get
            {
                return mSiteId;
            }
        }

        public Authentication(String pServiceUrl,
                              Int32 pSiteId,
                              String pSiteKey,
                              Int32 pSessionExpiration)
        {
            Debug.Assert(pServiceUrl != null);
            Debug.Assert(pSiteId > 0);
            Debug.Assert(pSiteKey != null);
            Debug.Assert(pSessionExpiration >= 0);

            mServiceUrl = pServiceUrl;
            mSiteId = pSiteId;
            mCipher = new StringCipher(pSiteKey);
            mSessionExpiration = pSessionExpiration;
            if (ConfigurationManager.AppSettings["FreePremium"] != null)
            {
                sFreePremium = bool.Parse(ConfigurationManager.AppSettings["FreePremium"]);
            }

            //Thread thread = new Thread(SessionExpirationThread);
            //thread.Start();
        }

        public int GetSessionCount()
        {
            int count = 0;
            lock (mSessions)
            {
                count = mSessions.Count;
            }
            return count;
        }

        public SessionData GetSession(String pSession)
        {
            SessionData data;
            lock (mSessions)
            {                
                if (mSessions.TryGetValue(pSession, out data))
                {
                    return data;
                }
            }
            return null;
        }
        /**
         * Runs a loop indefinately waiting EXPIRATION_CHECK_DELAY between loops checking for expired sessions
         * Removes sessions which have expired
         * Sessions will be revalidated with the central authentication service as required
         */
        private void SessionExpirationThread()
        {
            try
            {
                List<String> expired = new List<String>();
                while (true)
                {
                    expired.Clear();
                    lock (mSessions)
                    {
                        foreach (KeyValuePair<String, SessionData> pair in mSessions)
                        {
                            if (DateTime.UtcNow.Subtract(pair.Value.Validated).TotalSeconds > mSessionExpiration)
                            {
                                expired.Add(pair.Key);
                            }
                        }
                        foreach (String session in expired)
                        {
                            mSessions.Remove(session);
                        }
                    }
                    Thread.Sleep(EXPIRATION_CHECK_DELAY);
                }
            }
            catch (ThreadAbortException)
            {
            }
        }

        public string EncryptPassword(string pPassword)
        {
            return mCipher.Encrypt(pPassword);
        }

        /**
         * Publically exposed method for handling a syncronous Login attempt to the central authentication service
         * Handles generation of a hex based hash string
         *
         * @param  pUsername the username
         * @param  pPassword the unencrypted password
         * @param  pSession  an out parameter set to the session key generated
         * @return           a StatusCode indicating the result of the attempt
         */
        public StatusCode Login(String pUsername,
                                String pPassword,
                                out Int32 pUserId,
                                out String pSession,
                                out Byte pPremium)
        {
            StatusCode status = StatusCode.Ok;
            pUserId = 0;
            pSession = null;
            pPremium = 0;
            string ip = HttpContext.Current.Request.GetClientIPAddress().ToString();

            Logger.Log("IP is: " + ip, ELogLevel.Info);

            bool secureLogin = Convert.ToBoolean(ConfigurationManager.AppSettings["InsecureLogin"]);        

            String url = null;
            if (secureLogin)
            {
                url = mServiceUrl + "/validateInsecureClientUser";
            }
            else
            {
                url = mServiceUrl + "/validateClientUser";
            }
                        
            
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(url);
            request.Method = "POST";
            request.ContentType = "application/x-www-form-urlencoded";
            request.Timeout = MAX_REQUEST_TIMEOUT;
            using (StreamWriter writer = new StreamWriter(request.GetRequestStream()))
            {
                pPassword = mCipher.Encrypt(pPassword);
                writer.Write("siteid={0}&", mSiteId);
                writer.Write("username={0}&", Uri.EscapeDataString(pUsername));
                writer.Write("password={0}&", Uri.EscapeDataString(pPassword));                    
            }

            HttpWebResponse response = null;
            for (byte attempt = 1; attempt <= 5; attempt++)
            {
                try
                {
                    response = (HttpWebResponse)request.GetResponse();
                    break;
                }
                catch (WebException exc)
                {
                    if (exc.Status == WebExceptionStatus.ConnectFailure)
                    {
                        status = StatusCode.AuthenticationUnavailable;
                    }
                    else if (exc.Status == WebExceptionStatus.ProtocolError)
                    {
                        status = StatusCode.AuthenticationFailed;
                        break;
                    }
                    else
                    {
                        status = StatusCode.Unknown;
                        break;
                    }
                }
            }

            if (response != null && response.StatusCode == HttpStatusCode.OK)
            {
                using (StreamReader reader = new StreamReader(response.GetResponseStream()))
                {
                    String[] buf = reader.ReadToEnd().Split(':');
                    if (buf.Length != 3)
                    {
                        status = StatusCode.AuthenticationFailed;
                    }
                    else
                    {
                        Int32.TryParse(buf[0], out pUserId);
                        pSession = buf[1];
                        if (IsPremiumPromotion())
                        {
                            pPremium = (byte)1;
                        }
                        else
                        {
                            pPremium = Byte.Parse(buf[2]);
                        }
                        lock (mSessions)
                        {
                            mSessions[pSession] = new SessionData(pUserId, pPremium, ip);
                        }
                    }
                }
            }
            else
            {
                status = StatusCode.AuthenticationFailed;
            }
                    
            return status;
        }

        private bool IsPremiumPromotion()
        {
            return sFreePremium;
        }


        public StatusCode ValidateSessionKeyToUserId(String pSession,
                                                     out Int32 pUserId)
        {
            StatusCode status = StatusCode.Ok;
            pUserId = 0;
            string ip = HttpContext.Current.Request.GetClientIPAddress().ToString();

            lock (mSessions)
            {
                SessionData session;
                if (mSessions.TryGetValue(pSession, out session))
                {
                    if (DateTime.UtcNow.Subtract(session.Validated).TotalSeconds > mSessionExpiration)
                    {
                        mSessions.Remove(pSession);
                    }
                    // For now, ignore IP.
                    //else if (session.IpAddress != ip)
                    //{
                    //    return StatusCode.AuthenticationInvalidSession;
                    //}
                    else if (session.Expired)
                    {
                        mSessions.Remove(pSession);
                        return StatusCode.AuthenticationInvalidSession;
                    }
                    else
                    {
                        pUserId = session.UserId;
                        session.LastAccessed = DateTime.UtcNow;
                    }
                }
            }

            if (pUserId > 0)
            {
                return StatusCode.Ok;
            }

           
            String url = null;

            if (Convert.ToBoolean(ConfigurationManager.AppSettings["InsecureLogin"]))
            {
                url = mServiceUrl + "/validateUserSession";
            }
            else
            {
                url = mServiceUrl + "/validateClientUserSession";
            }

            
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(url);
            request.Method = "POST";
            request.ContentType = "application/x-www-form-urlencoded";
            request.Timeout = MAX_REQUEST_TIMEOUT;
            using (StreamWriter writer = new StreamWriter(request.GetRequestStream()))
            {
                writer.Write("siteid={0}&", mSiteId);
                writer.Write("session={0}", pSession);
            }

            HttpWebResponse response = null;
            // 5 Attempts:
            for (byte attempt = 1; attempt <= 5; attempt++)
            {
                try
                {
                    response = (HttpWebResponse)request.GetResponse();
                    break;
                }
                catch (WebException exc)
                {
                    if (exc.Status == WebExceptionStatus.ConnectFailure)
                    {
                        status = StatusCode.AuthenticationUnavailable;
                    }
                    else if (exc.Status == WebExceptionStatus.ProtocolError)
                    {
                        status = StatusCode.AuthenticationInvalidSession;
                        break;
                    }
                    else
                    {
                        status = StatusCode.Unknown;
                        break;
                    }
                    return status;
                }
            }

            if (response != null && response.StatusCode == HttpStatusCode.OK)
            {
                using (StreamReader reader = new StreamReader(response.GetResponseStream()))
                {
                    String buf = reader.ReadToEnd();
                    Int32.TryParse(buf, out pUserId);
                }
                Byte premium = 0;
                status = GetPremiumLevel(pUserId, out premium);
                if (status != StatusCode.Ok)
                {
                    return status;
                }
                lock (mSessions)
                {
                    mSessions[pSession] = new SessionData(pUserId, premium, ip);
                }
            }
            else
            {
                status = StatusCode.AuthenticationInvalidSession;
            }  
            return status;
        }

        /**
         * Private helper method to get premium status
         *
         * @param  pUserId        the user id to get the premium status for
         * @param  pPremiumLevel  an out parameter set to the premium level if successful
         * @return                a StatusCode indicating the result of the attempt
         */
        private StatusCode GetPremiumLevel(Int32 pUserId,
                                           out Byte pPremiumLevel)
        {
            StatusCode status = StatusCode.Ok;
            pPremiumLevel = 0;
           
            String url = mServiceUrl + "/getPremiumLevel";
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(url);
            request.Method = "POST";
            request.ContentType = "application/x-www-form-urlencoded";
            request.Timeout = MAX_REQUEST_TIMEOUT;
            using (StreamWriter writer = new StreamWriter(request.GetRequestStream()))
            {
                writer.Write("siteid={0}&", mSiteId);
                writer.Write("uid={0}", pUserId);
            }

            HttpWebResponse response = null;

            for (byte attempt = 1; attempt <= 5; attempt++)
            {
                try
                {
                    response = (HttpWebResponse)request.GetResponse();
                    break;
                }
                catch (WebException exc)
                {
                    if (exc.Status == WebExceptionStatus.ConnectFailure)
                    {
                        status = StatusCode.AuthenticationUnavailable;                        
                    }
                    else if (exc.Status == WebExceptionStatus.ProtocolError)
                    {
                        status = StatusCode.AuthenticationInvalidUser;
                        break;
                    }
                    else
                    {
                        status = StatusCode.Unknown;
                        break;
                    }
                    pPremiumLevel = 0;
                }
            }

            if (response != null && response.StatusCode == HttpStatusCode.OK)
            {
                using (StreamReader reader = new StreamReader(response.GetResponseStream()))
                {
                    String buf = reader.ReadToEnd();
                    Byte.TryParse(buf, out pPremiumLevel);
                }
            }
            else
            {
                status = StatusCode.AuthenticationInvalidSession;
            }
            
            return status;
        }

        /**
         * Publically exposed method for handling premium validation
         * Handles checking local session cache for immediate validations
         * Does NOT handles validation against the central authentication service
         * Premium status is only valid so long as the session is cached locally
         *
         * @param  pSession the session key
         * @return          true if the user is cached and premium, false otherwise
         */
        public Byte GetPremiumLevel(String pSession)
        {
            Byte premium = 0;
            lock (mSessions)
            {
                SessionData data;
                if (mSessions.TryGetValue(pSession, out data))
                {
                    premium = data.Premium;
                }
            }
            return premium;
        }

        /**
         * Private constants
         */
        // 1 minute session recycling
        private const Int32 EXPIRATION_CHECK_DELAY = 60 * 1000;
        private const Int32 SIZEOF_JUNK = 32;
        private const Int32 MAX_REQUEST_TIMEOUT = 30000;

        /**
         * Private member data
         */
        private String mServiceUrl = null;
        private Int32 mSessionExpiration = 0;
        private Int32 mSiteId = 0;
        private StringCipher mCipher = null;
        private Random mRandom = new Random();
        private Dictionary<String, SessionData> mSessions =
            new Dictionary<String, SessionData>();        

    }
}
