﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Net;
using Curse;
using Curse.Contactology.net.emailcampaigns.soapserver;
using System.Threading;
using System.Web.Services.Protocols;
using System.Diagnostics;

namespace Curse.Contactology
{
    public static class Mailer
    {
        
        private static readonly string sBulkUsername;
        private static readonly string sBulkPassword;

        private static readonly string sTransactionalUsername;
        private static readonly string sTransactionalPassword;
        
        // Static constructor. Automatically called on first object use:
        static Mailer()
        {

            Logger.SetAutoGrowthMegabytes = 10;
            Logger.SetLogLevel = ELogLevel.Debug;
            Logger.SetLogPath = @"c:\Logs\Curse.Contactology\";

            sBulkUsername = ConfigurationManager.AppSettings["Contactology.BulkUsername"];
            sBulkPassword = ConfigurationManager.AppSettings["Contactology.BulkPassword"];

            sTransactionalUsername = ConfigurationManager.AppSettings["Contactology.TransactionalUsername"];
            sTransactionalPassword = ConfigurationManager.AppSettings["Contactology.TransactionalPassword"];
        }

        public static bool SendCustomEmail(bool sendThreaded, int campaignId, string emailAddress, Dictionary<int, string> customValues, ESendMode sendMode)
        {
            
            CustomEmail customEmail = new CustomEmail();
            customEmail.CustomValues = customValues;
            customEmail.EmailAddress = emailAddress;
            customEmail.CampaignId = (int)campaignId;
            customEmail.SendMode = sendMode;

            if (sendThreaded)
            {
                Thread t = new Thread(new ParameterizedThreadStart(DoSendCustomEmail));
                t.Start(customEmail);
            }
            else
            {
                DoSendCustomEmail(customEmail);
            }
            return true;
        }

        private class CustomEmail
        {
            public string EmailAddress;
            public Dictionary<int, string> CustomValues;
            public int CampaignId;
            public ESendMode SendMode;
        }

        public static void DoSendCustomEmail(Object o)
        {
            AddRecipientContact contact = new AddRecipientContact();

            try
            {
                CustomEmail customEmail = (CustomEmail)o;
                EmailCampaigns client = GetPreparedClient(customEmail.SendMode, 0);                                                               
                contact.emailAddress = customEmail.EmailAddress;
                contact.customFields = GetCustomFieldValues(customEmail.CustomValues);
                contact.source = "Auth Server";
                contact.status = 0;
                int success = client.AddRecipientToCampaign(contact, customEmail.CampaignId);
            }
            catch (WebException ex)
            {
                Logger.Log(ELogLevel.Error, null, Environment.NewLine + "   Web Exception: {0}"
                    + Environment.NewLine + "   Stack Trace: {1}"
                    + Environment.NewLine + "   Email: {2}", ex.Message, ex.StackTrace, contact.emailAddress);
            }
            catch (SoapException ex)
            {
                Logger.Log(ELogLevel.Error, null, Environment.NewLine + "   Soap Exception: {0}" + Environment.NewLine + "   Stack Trace: {1}", ex.Message, ex.StackTrace);
            }
            catch (Exception ex)
            {
                Logger.Log(ELogLevel.Error, null, Environment.NewLine + "   General Exception: {0}" + Environment.NewLine + "   Stack Trace: {1}", ex.Message, ex.StackTrace);
            }

        }

        public static bool DeleteList(ESendMode sendMode, int listId)
        {
            EmailCampaigns client = GetPreparedClient(sendMode, 0);
            return (client.DeleteList(listId) == 0);
        }

        public static List<int> GetCompletedCampaigns(ESendMode sendMode)
        {

            EmailCampaigns client = GetPreparedClient(sendMode, 0);
            Campaign[] completedCampaignArray = null;
            List<int> completedCampaigns = new List<int>();

            try
            {
                completedCampaignArray = client.GetCompletedCampaigns("U", "ID", 0, 0);
            }
            catch
            {
                return completedCampaigns;
            }
            
            foreach (Campaign campaign in completedCampaignArray)
            {
                completedCampaigns.Add(campaign.campaign_id);
            }
            return completedCampaigns;

        }

        public static DynamicCampaign SendDynamicCampaign(  string name,
                                                            string senderName,
                                                            string senderAddress,
                                                            string subject,
                                                            string htmlBody,
                                                            string textBody,
                                                            Dictionary<string, Dictionary<int, string>> contacts)
        {

            string firstKey = null;
            foreach (string key in contacts.Keys)
            {
                firstKey = key;
                break;
            }

            
            // Method scope variables
            List<string> combinedCSV = new List<string>();            
            DynamicCampaign dynamicCampaign = new DynamicCampaign();
            dynamicCampaign.RecipientCount = contacts.Count;
            dynamicCampaign.Name = name;            
            
            // Set the attributes for the CSV import
            CsvAttributes attributes = new CsvAttributes();
            attributes.emailIndex = contacts[firstKey].Count;
            attributes.saveFirstRow = 1;
            attributes.customFieldMapping = GetCustomFieldMapping(contacts[firstKey]);            
            

            // Get a contactology client:
            EmailCampaigns client = GetPreparedClient(ESendMode.Bulk, 240 * 1000);
            
            // Create a new list, used just for this dynamic campaign:
            dynamicCampaign.ListId = client.AddList(name, "", 2);
            int[] lists = { dynamicCampaign.ListId };
            int importedCounter = 0;
                        
            // Iterate over our contacts, and for each 900 import them into our new list.
            foreach (KeyValuePair<string, Dictionary<int, string>> kvp in contacts)
            {
                ++importedCounter;
                List<string> contactCsv = new List<string>();
                foreach (KeyValuePair<int, string> customValue in kvp.Value)
                {
                    contactCsv.Add(customValue.Value.Replace(",", "&#44;"));                    
                }
                contactCsv.Add(kvp.Key);                
                combinedCSV.Add(string.Join(",", contactCsv.ToArray()));

                if (combinedCSV.Count >= 900 || importedCounter == contacts.Count)
                {
                    string combinedCSVString = string.Join("\n", combinedCSV.ToArray());
                    int failureCount = 0;
                    // Keep trying to import contacts, up to 10 times.
                    while (failureCount <= 10 && !SafelyImportContacts(client, combinedCSVString, attributes, "Auth Server", lists))
                    {
                        ++failureCount;
                        Thread.Sleep(1000);
                    }
                    combinedCSV.Clear();
                }
            }
                        
            dynamicCampaign.CampaignId = client.CreateAndSendCampaign(name, "", subject, htmlBody, textBody, DateTime.UtcNow,
                senderAddress, senderName, 0, "", "1", "", 0, new CustomFieldMapping[0], lists, 1, "mcomperda@curse.com");


            return dynamicCampaign;
        }

        private static bool SafelyImportContacts(EmailCampaigns client, string csv, CsvAttributes attributes, string source, int[] lists )
        {
            try
            {
                ImportResult result = client.ImportCsv(csv, attributes, "Auth Server", 1, new CustomFieldValue[0], lists);
                return true;
            }
            catch
            {
                return false;
            }
            
        }


        private static bool UpdateExistingContact(EmailCampaigns client, string emailAddress, Dictionary<int, string> customValues)
        {

            CsvAttributes attributes = new CsvAttributes();
            attributes.emailIndex = customValues.Count;
            attributes.saveFirstRow = 1;
            
            attributes.customFieldMapping = GetCustomFieldMapping(customValues);
            int[] lists = {4};

            List<string> contactCsv = new List<string>();
            foreach (KeyValuePair<int, string> customValue in customValues)
            {
                contactCsv.Add(customValue.Value);
            }
            contactCsv.Add(emailAddress);

            ImportResult result = client.ImportCsv(string.Join(",", contactCsv.ToArray()), attributes, "Auth Server", 1, new CustomFieldValue[0], lists);

            return true;
        }

        private static CustomFieldMapping[] GetCustomFieldMapping(Dictionary<int, string> customValues)
        {            
            CustomFieldMapping[] customMapping = new CustomFieldMapping[ customValues.Count ];
            int index = 0;
            foreach (KeyValuePair<int, string> customValue in customValues)
            {
                CustomFieldMapping mapping = new CustomFieldMapping();
                mapping.columnNum = index;
                mapping.customFieldID = customValue.Key;
                customMapping[index] = mapping;
                ++index;
            }
            return customMapping;
        }

        private static CustomFieldValue[] GetCustomFieldValues(Dictionary<int, string> customValues)
        {
            if (customValues.Count == 0)
            {
                return null;
            }
            List<CustomFieldValue> customValuesList = new List<CustomFieldValue>();
            foreach (KeyValuePair<int, string> customValue in customValues)
            {
                CustomFieldValue customFieldValue = new CustomFieldValue();
                customFieldValue.ID = customValue.Key;
                customFieldValue.Value = customValue.Value;
                customValuesList.Add(customFieldValue);
            }
            return customValuesList.ToArray();
        }

        private static EmailCampaigns GetPreparedClient(ESendMode sendMode, int timeoutInSeconds)
        {
            EmailCampaigns client = new EmailCampaigns();
                                                               
            string username = null;
            string password = null;

            switch (sendMode)
            {
                case ESendMode.Bulk:
                    if (timeoutInSeconds == 0)
                        timeoutInSeconds = 60 * 1000;
                    client.Timeout = timeoutInSeconds;
                    username = sBulkUsername;
                    password = sBulkPassword;
                    break;

                case ESendMode.Transactional:
                    if (timeoutInSeconds == 0)
                        timeoutInSeconds = 5 * 1000;
                    client.Timeout = timeoutInSeconds; // 5 Seconds
                    username = sTransactionalUsername;
                    password = sTransactionalPassword;
                    break;
            }            

            ICredentials credentials = new NetworkCredential(username, password);
            System.Net.ServicePointManager.Expect100Continue = false;
            client.Credentials = credentials;            
            return client;
        }


    }

}
