﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Configuration;
using System.IO;
using System.Data.SqlClient;

using Curse.Extensions;
using System.Reflection;
using Microsoft.SqlServer.Management.Smo;
using Microsoft.SqlServer.Management.Common;


namespace Curse.Logging
{
    public class LogProvider
    {
        #region Singleton Implementation

        private static readonly LogProvider _instance = new LogProvider();

        public static LogProvider Instance
        {
            get
            {
                return _instance;
            }
        }

        #endregion

        // Initialization State
        private bool _isInitialized;
        private bool _supportsDatabaseLogging;

        // Target Paths
        private string _connectionString;
        private string _folderPath;

        private LogTarget _defaultLogMode = LogTarget.Database;
        
        private int _productID;


        public void Log(LogTarget logTarget, LogEntryType type, string className, string methodName, string body)
        {
            LogEntry entry = new LogEntry(type, _productID, className, methodName, body);
            Log(logTarget, entry);
        }

        public void Log(LogTarget logMode, LogEntry entry)
        {

            switch (logMode)
            {
                case LogTarget.FileSystem:
                    LogToFileSystem(entry);
                    break;
                case LogTarget.Database:
                    LogToDatabase(entry);
                    break;
            }

        }

        private void LogToDatabase(LogEntry entry)
        {
            if (!_isInitialized || !_supportsDatabaseLogging)
            {
                return;
            }


        }

        private void LogToFileSystem(LogEntry entry)
        {
            if (!_isInitialized)
            {
                return;
            }
        }
        

        public void Initialize(LogTarget defaultLogMode, int productID, ConnectionStringSettings connectionString, string folderPath)
        {
            _defaultLogMode = defaultLogMode;
            _productID = productID;

            // Required
            this.SetupFileSystemLogging(folderPath);

            // Optional
            this.SetupDatabaseLogging(connectionString);            

            _isInitialized = true;
        }

        private void SetupFileSystemLogging(string folderPath)
        {
            // Ensure that the specified folder path actually exists
            if (!Directory.Exists(folderPath))
            {
                throw new ArgumentException("The folder patch specified, '{0}', does not exist!".FormatWith(folderPath));
            }
            _folderPath = folderPath;
        }

        private void SetupDatabaseLogging(ConnectionStringSettings connectionString)
        {

            if (connectionString == null)
            {
                return;
            }

            using (SqlConnection conn = new SqlConnection(connectionString.ConnectionString))
            {

                // 1. Confirm the ability to connect:
                try
                {
                    conn.Open();
                }
                catch(Exception ex)
                {
                    throw new ArgumentException("The database in the provided connection string, '{0}', could not be accessed. The exception reported is: {1}".FormatWith(connectionString.ConnectionString, ex.Message)); 
                }


                // 2. Check for the table's existence
                SqlCommand command = conn.CreateCommand();
                command.CommandText = "SELECT COUNT(0) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_TYPE='BASE TABLE' AND TABLE_NAME='LogEntry'";
                
                if ((int)command.ExecuteScalar() == 0)
                {


                    // 3. Create the table  
                    string sqlContents = null; 
                    try
                    {
                        Assembly assembly = Assembly.GetExecutingAssembly();
                                                
                        using (StreamReader reader = new StreamReader(assembly.GetManifestResourceStream("Curse.Logging.LogEntry.sql")))
                        {
                            sqlContents = reader.ReadToEnd();
                        }
                    }
                    catch (Exception ex)
                    {
                        throw new ArgumentException("The LogEntry table SQL could not be loaded. The exception reported is: {0}".FormatWith(ex.Message));
                    }                    

                    try
                    {
                        Server server = new Server( new ServerConnection(conn));
                        server.ConnectionContext.ExecuteNonQuery(sqlContents);                                                
                    }
                    catch (Exception ex)
                    {
                        throw new ArgumentException("The LogEntry table SQL could not be executed. The exception reported is: {0}".FormatWith(ex.Message));
                    }
                }    
            }

            _connectionString = connectionString.ConnectionString;
            _supportsDatabaseLogging = true;

        }     
    }
}
