/*
 *
 * Copyright (c) 2006/2007 Sam Collett (http://www.texotela.co.uk)
 * Licensed under the MIT License:
 * http://www.opensource.org/licenses/mit-license.php
 *
 *  20 February 2007
 *    removeOption can now take a regular expression
 *    (useful if you want to remove multiple options in one go)
 *
 *  13 February 2007
 *    addOption can also replace options that already exist with the same value
 *    selectOptions can clear previously selected options
 *    new copyOptions now allows you to copy options from one select to another
 *
 *  2 February 2007
 *    Fix for Safari 2.0 - couldn't add option
 *
 *  5 December 2006
 *   Select option(s) by value with 'selectOptions'
 *   Based on code by Mathias Bank (http://www.mathias-bank.de)
 *
 */
 
(function($) {
 
/**
 * Adds (single/multiple) options to a select box (or series of select boxes)
 *
 * @name     addOption
 * @author   Sam Collett (http://www.texotela.co.uk)
 * @type     jQuery
 * @example  $("#myselect").addOption("Value", "Text"); // add single value (will be selected)
 * @example  $("#myselect").addOption("Value 2", "Text 2", false); // add single value (won't be selected)
 * @example  $("#myselect").addOption({"foo":"bar","bar":"baz"}, false); // add multiple values, but don't select
 *
 */
$.fn.addOption = function()
{
	var a = arguments;
	if(a.length == 0) return this;
	// select option when added? default is true
	var sO = true;
	// multiple items
	var m = false;
	if(typeof a[0] == "object")
	{
		m = true;
		var items = a[0];
	}
	if(a.length >= 2)
	{
		if(typeof a[1] == "boolean") sO = a[1];
		else if(typeof a[2] == "boolean") sO = a[2];
		if(!m)
		{
			var v = a[0];
			var t = a[1];
		}
	}
	this.each(
		function()
		{
			if(this.nodeName.toLowerCase() != "select") return;
			if(m)
			{
				
				for(var i in items)
				{
					$(this).addOption(i, items[i], sO);
				}
			}
			else
			{
				var option = document.createElement("option");
				option.value = v;
				option.text = t;
				var i;
				var r = false;
				// get options
				var o = this.options;
				// get number of options
				var oL = o.length;
				// loop through existing options
				for(i = 0; i < oL; i++)
				{
					// replace existing option
					if(o[i].value == option.value)
					{
						r = true;
						break;
					}
				}
				if(i < oL && !r) i = oL;
				this.options[i] = option;
				if(sO)
				{
					o[i].selected = true;
				}
			}
		}
	);
	return this;
};

/**
 * Add options via ajax
 *
 * @name     ajaxAddOption
 * @author   Sam Collett (http://www.texotela.co.uk)
 * @type     jQuery
 * @param    String url   Page to get options from (must be valid JSON)
 * @param    Object params   (optional) Any parameters to send with the request
 * @param    Boolean select   (optional) Select the added options, default true
 * @example  $("#myselect").ajaxAddOption("myoptions.php");
 *
 */
$.fn.ajaxAddOption = function(url, params, select)
{
	if(typeof url != "string") return this;
	if(typeof params != "object") params = {};
	if(typeof select != "boolean") select = true;
	this.each(
		function()
		{
			var self = this;
			$.getJSON(url,
				params,
				function(r)
				{
					$(self).addOption(r, select);
				}
			);
		}
	);
	return this;
};

/**
 * Removes an option (by value or index) from a select box (or series of select boxes)
 *
 * @name     removeOption
 * @author   Sam Collett (http://www.texotela.co.uk)
 * @type     jQuery
 * @param    String|RegExp|Number Option to remove
 * @example  $("#myselect").removeOption("Value"); // remove by value
 * @example  $("#myselect").removeOption(/^val/i); // remove options with a value starting with 'val'
 * @example  $("#myselect").removeOption(/./); // remove all options
 * @example  $("#myselect").removeOption(0); // remove by index
 *
 */
$.fn.removeOption = function()
{
	var a = arguments;
	if(a.length == 0) return this;
	var ta = typeof a[0];
	if(ta == "string") var v = a[0];
	else if(ta == "object" || ta == "function") var v = a[0]; /* regular expression */
	else if(ta == "number") var i = a[0];
	else return this;
	this.each(
		function()
		{
			if(this.nodeName.toLowerCase() != "select") return;
			if(v)
			{
				// get options
				var o = this.options;
				// get number of options
				var oL = o.length;
				for(var i=oL-1; i>=0; i--)
				{
					if(v.constructor == RegExp)
					{
						if (o[i].value.match(v))
						{
							o[i] = null;
						}
					}
					else if(o[i].value == v)
					{
						o[i] = null;
					}
				}
			}
			else
			{
				this.remove(i);
			}
		}
	);
	return this;
};

/**
 * Sort options (ascending or descending) in a select box (or series of select boxes)
 *
 * @name     sortOptions
 * @author   Sam Collett (http://www.texotela.co.uk)
 * @type     jQuery
 * @param    Boolean ascending   (optional) Sort ascending (true/undefined), or descending (false)
 * @example  // ascending
 * $("#myselect").sortOptions(); // or $("#myselect").sortOptions(true);
 * @example  // descending
 * $("#myselect").sortOptions(false);
 *
 */
$.fn.sortOptions = function(ascending)
{
	var a = typeof ascending == "undefined" ? true : ascending;
	this.each(
		function()
		{
			if(this.nodeName.toLowerCase() != "select") return;
			
			// get options
			var o = this.options;
			// get number of options
			var oL = o.length;
			// create an array for sorting
			var sA = [];
			// loop through options, adding to sort array
			for(var i = 0; i<oL; i++)
			{
				sA[i] =
				{
					v: o[i].value,
					t: o[i].text
				};
			}
			// sort items in array
			sA.sort(
				function(o1, o2)
				{
					// option text is made lowercase for case insensitive sorting
					o1t = o1.t.toLowerCase();
					o2t = o2.t.toLowerCase();
					// if options are the same, no sorting is needed
					if(o1t == o2t) return 0;
					if(a)
					{
						return o1t < o2t ? -1 : 1;
					}
					else
					{
						return o1t > o2t ? -1 : 1;
					}
				}
			);
			// change the options to match the sort array
			for(var i = 0; i<oL; i++)
			{
				o[i].text = sA[i].t;
				o[i].value = sA[i].v;
			}
		}
	);
	return this;
};
/**
 * Selects an option by value
 *
 * @name     selectOptions
 * @author   Mathias Bank (http://www.mathias-bank.de), original function
 * @author   Sam Collett (http://www.texotela.co.uk), addition of regular expression matching
 * @type     jQuery
 * @param    String|RegExp value  which options should be selected
 * can be a string or regular expression
 * @param    Boolean clear  clear existing selected options, default false
 * @example  $("#myselect").selectOptions("val1"); // with the value 'val1'
 * @example  $("#myselect").selectOptions(/^val/i); // with the value starting with 'val', case insensitive
 *
 */
$.fn.selectOptions = function(value, clear)
{
	var v = value;
	var vT = typeof value;
	var c = clear || false;
	// has to be a string or regular expression (object in IE, function in Firefox)
	if(vT != "string" && vT != "function" && vT != "object") return this;
	this.each(
		function()
		{
			if(this.nodeName.toLowerCase() != "select") return this;
			
			// get options
			var o = this.options;
			// get number of options
			var oL = o.length;
			
			for(var i = 0; i<oL; i++)
			{
				if(v.constructor == RegExp)
				{
					if (o[i].value.match(v))
					{
						o[i].selected = true;
					}
					else if(c)
					{
						o[i].selected = false;
					}
				}
				else
				{
					if (o[i].value == v)
					{
						o[i].selected = true;
					}
					else if(c)
					{
						o[i].selected = false;
					}
				}
			}
		}
	);
	return this;
};

/**
 * Copy options to another select
 *
 * @name     copyOptions
 * @author   Sam Collett (http://www.texotela.co.uk)
 * @type     jQuery
 * @param    String to  Element to copy to
 * @param    String which  (optional) Specifies which options should be copied - 'all' or 'selected'. Default is 'selected'
 * @example  $("#myselect").copyOptions("#myselect2"); // copy selected options from 'myselect' to 'myselect2'
 * @example  $("#myselect").copyOptions("#myselect2","selected"); // same as above
 * @example  $("#myselect").copyOptions("#myselect2","all"); // copy all options from 'myselect' to 'myselect2'
 *
 */
$.fn.copyOptions = function(to, which)
{
	var w = which || "selected";
	if($(to).size() == 0) return this;
	this.each(
		function()
		{
			if(this.nodeName.toLowerCase() != "select") return this;
			
			// get options
			var o = this.options;
			// get number of options
			var oL = o.length;
			
			for(var i = 0; i<oL; i++)
			{
				if(w == "all" ||
					(w == "selected" && o[i].selected)
					)
				{
					$(to).addOption(o[i].value, o[i].text);
				}
			}
		}
	);
	return this;
};

})(jQuery);
