﻿using System;
using System.Text;

namespace Curse.ServiceAuthentication.Models
{
    public class StringCipher
    {
        private const Int32 HexBase = 16;

        public static Byte[] HexStringToByteArray(String pHex)
        {
            var length = pHex.Length;
            var data = new Byte[(length + 1) / 2];
            for (var index = 0;
                 index < length;
                 index += 2)
            {
                data[index / 2] = Convert.ToByte(pHex.Substring(index, 2), HexBase);
            }
            return data;
        }

        /**
         * Private constants
         */
        private const Int32 PaddingSize = 4;
        private const Int32 StateSize = 256;

        /**
         * Private member data
         */
        private readonly Byte[] _state = new Byte[StateSize];

        /**
         * Initialization constructor
         * Initializes the internal base state using a byte array for the key
         *
         * @param pKey the byte array containing the key to initialize base state with
         */
        public StringCipher(Byte[] pKey)
        {
            InitializeState(pKey);
        }

        /**
         * Initialization constructor
         * Initializes the internal base state using a hex string for the key
         *
         * @param pKey the string containing hex data to initialize base state with
         */
        public StringCipher(String pKey)
        {
            InitializeState(HexStringToByteArray(pKey));
        }

        /**
         * Initializes the internal base state using a byte array for the key
         *
         * @param pKey the byte array containing the key to initialize base state with
         */
        private void InitializeState(Byte[] pKey)
        {
            for (Byte i = 0; ; )
            {
                _state[i] = i;
                if (++i == 0)
                {
                    break;
                }
            }

            for (Byte i = 0, j = 0, k = 0; ; )
            {
                j += _state[i];
                j += pKey[i % pKey.Length];
                k = _state[i];
                _state[i] = _state[j];
                _state[j] = k;
                i += 2;
                if (i == 0)
                {
                    break;
                }
            }
        }

        /**
         * Publically exposed method for encrypting a string
         * Takes consideration for Base64 encoding
         *
         * @param  pString the string to be encrypted
         * @return         the string after random padding, encrypting and Base64 encoding
         */
        public String Encrypt(String pString)
        {
            var state = new Byte[StateSize];
            var junk = new Byte[PaddingSize];
            var bytes = Encoding.UTF8.GetBytes(pString);
            var encrypted = new Byte[PaddingSize + bytes.Length];

            Buffer.BlockCopy(_state, 0, state, 0, state.Length);
            new Random().NextBytes(junk);
            Buffer.BlockCopy(junk, 0, encrypted, 0, PaddingSize);
            Buffer.BlockCopy(bytes, 0, encrypted, PaddingSize, bytes.Length);

            Encrypt(state, encrypted);
            return Convert.ToBase64String(encrypted);
        }

        /**
         * Publically exposed method for decrypting a string
         * Takes consideration for Base64 encoding
         *
         * @param  pString the string to be decrypted
         * @return         the string after Base64 decoding, decrypting and random padding stripping
         */
        public String Decrypt(String pString)
        {
            var state = new Byte[_state.Length];
            var bytes = Convert.FromBase64String(pString);
            Buffer.BlockCopy(_state, 0, state, 0, state.Length);

            Decrypt(state, bytes);
            return Encoding.UTF8.GetString(bytes, PaddingSize, bytes.Length - PaddingSize);
        }



        /**
         * Encrypts pData inline using a copy of the base state
         *
         * @param pState contains a copy of the base state
         * @param pData  contains the data to be encrypted
         */
        private static void Encrypt(Byte[] pState,
                                    Byte[] pData)
        {
            Byte feedback = 0xFF;            
            for (Int32 i = 0, j = 0, k = 0; k < pData.Length; ++k)
            {
                i = (i + 1) % StateSize;
                j = (j + pState[i]) % StateSize;

                var tmp = pState[i];
                pState[i] = pState[j];
                pState[j] = tmp;

                pData[k] ^= pState[(pState[i] + pState[j]) % StateSize];
                pData[k] ^= feedback;
                feedback = pData[k];
            }
        }

        /**
         * Decrypts pData inline using a copy of the base state
         *
         * @param pState contains a copy of the base state
         * @param pData  contains the data to be decrypted
         */
        private static void Decrypt(Byte[] pState,
                                    Byte[] pData)
        {
            Byte feedback = 0xFF;            
            for (Int32 i = 0, j = 0, k = 0; k < pData.Length; ++k)
            {
                i = (i + 1) % StateSize;
                j = (j + pState[i]) % StateSize;

                var tmp = pState[i];
                pState[i] = pState[j];
                pState[j] = tmp;

                tmp = pData[k];
                pData[k] ^= pState[(pState[i] + pState[j]) % StateSize];
                pData[k] ^= feedback;
                feedback = tmp;
            }
        }
    }
}
