﻿using System.Globalization;
using Curse.CloudServices.Authentication;
using Curse.ServiceAuthentication.AuthenticationService;
using Curse.ServiceAuthentication.Models;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.ServiceModel.Web;
using System.Web;
using System.Web.Caching;
using Curse.ServiceEncryption;

namespace Curse.ServiceAuthentication
{
    public static class AuthenticationProvider
    {
        private static string _serviceUrl;
        private static int _siteID;
        private static bool _freePremium = false;

        private static StringCipher _outgoingCipher;
        private static StringCipher _incomingCipher;

        private static string _apiKey;


        private const int SubscriptionSeedPremium = 181756351;
        private const int SubscriptionSeedNonPremium = 76172335;
        private const string IncomingCipherKey = "31ED549B43318C9CFAE926ADE1720D2202C2DEFFAB06AE9D78FB6F16DF2D70A4";

        private static bool _requireUserGrants = false;
        private static bool _generateAuthTokens = true;


        public static string ApiKey
        {
            get
            {
                return _apiKey;
            }
        }

        private static NetworkService CreateServiceClient()
        {
            return new NetworkService { Url = _serviceUrl };
        }

        public static void Initialize(string serviceUrl, LogDelegate logDelegate, int authSiteID, string authSiteKey, string apiKey, bool freePremium = false, bool requireUserGrants = false, string userGrantConnectionString = null, bool authLevelsDisabled = false, bool generateAuthTokens = true)
        {

            AuthenticationLogger.Initialize(logDelegate);
            if (!Uri.IsWellFormedUriString(serviceUrl, UriKind.Absolute))
            {
                throw new ArgumentException("The service url supplied is invalid: " + serviceUrl);
            }
            _serviceUrl = serviceUrl;
            _siteID = authSiteID;
            _outgoingCipher = new StringCipher(authSiteKey);
            _incomingCipher = new StringCipher(IncomingCipherKey);
            _freePremium = freePremium;
            _apiKey = apiKey;

            if (authLevelsDisabled)
            {
                RequiresAuthenticationAttribute.DisableLevels();
            }

            _requireUserGrants = requireUserGrants;

            if (_requireUserGrants)
            {
                UserGrantManager.Instance.Initialize(userGrantConnectionString);
            }

            _generateAuthTokens = generateAuthTokens;
        }

        public static int GetUserID(string usernameOrEmail)
        {

            if (usernameOrEmail.Contains("@"))
            {
                using (var authService = CreateServiceClient())
                {
                    return authService.v2GetUserIDFromEmail(_siteID, usernameOrEmail);
                }
            }
            else
            {
                using (var authService = CreateServiceClient())
                {
                    return authService.v2GetUserIDFromUsername(_siteID, usernameOrEmail);
                }
            }

        }

        internal static AuthenticationContext CreateAuthenticationContext()
        {

            try
            {
                var authTokenContext = CloudServices.Authentication.AuthenticationContext.Current;

                if (!authTokenContext.IsAnonymous)
                {
                    var token = EncryptionToken.FromValue(authTokenContext.Token);

                    if (!token.IsValid)
                    {
                        return new AuthenticationContext(AuthenticationSession.Anonymous, AuthenticationStatus.InvalidSession);
                    }

                    var timestamp = token.GetDateTime("Timestamp");

                    if (DateTime.UtcNow - timestamp > TimeSpan.FromDays(60))
                    {
                        return new AuthenticationContext(AuthenticationSession.Anonymous, AuthenticationStatus.InvalidSession);
                    }

                    return new AuthenticationContext(new AuthenticationSession
                    {
                        UserID = token.GetInteger("UserID"),
                        Username = token.GetValue("Username"),
                        Token = authTokenContext.Token,
                    }, AuthenticationStatus.Success);

                }

                var authToken = GetLegacyAuthToken();

                if (authToken.IsAnonymous)
                {
                    return new AuthenticationContext(AuthenticationSession.Anonymous, AuthenticationStatus.Success);
                }

                AuthenticationSession session;
                var authStatus = TryAuthenticateUser(authToken.Username, true, authToken.Password, true, out session);
                if (authStatus != AuthenticationStatus.Success)
                {
                    return new AuthenticationContext(AuthenticationSession.Anonymous, authStatus);
                }
                return new AuthenticationContext(session, authStatus);


            }
            catch (Exception ex)
            {
                AuthenticationLogger.Log("Failed to get current session", ex);
                return new AuthenticationContext(AuthenticationSession.Anonymous, AuthenticationStatus.UnknownError);
            }
        }

        public static AuthenticationSession CurrentSession
        {
            get { return AuthenticationContext.Current.AuthenticationSession; }

        }

        public static User CurrentUserProfile
        {
            get
            {
                return GetUserProfile(AuthenticationContext.Current.AuthenticationSession.UserID);
            }
        }

        private static LegacyAuthenticationToken GetLegacyAuthToken()
        {
            var webContext = WebOperationContext.Current;
            var operationContext = OperationContext.Current;

            if (operationContext != null && webContext != null)
            {
                var messageVersion = operationContext.IncomingMessageVersion;

                if (messageVersion == MessageVersion.None) // Soap 11 Version - Check standard Http headers
                {
                    var username = webContext.IncomingRequest.Headers["x-curse-username"];
                    var password = webContext.IncomingRequest.Headers["x-curse-password"];
                    if (username != null && password != null)
                    {
                        return LegacyAuthenticationToken.FromHeaders(username, password);
                    }
                }
                else if (operationContext.IncomingMessageHeaders.FindHeader(LegacyAuthenticationToken.HeaderName, LegacyAuthenticationToken.HeaderNamespace) != -1) // Soap 12 Version - Check standard Soap headers
                {
                    return operationContext.IncomingMessageHeaders.GetHeader<LegacyAuthenticationToken>(LegacyAuthenticationToken.HeaderName, LegacyAuthenticationToken.HeaderNamespace);
                }
            }
            return new LegacyAuthenticationToken() { IsAnonymous = true };

        }

        private static readonly ConcurrentDictionary<string, LoginResult> _cachedLogins = new ConcurrentDictionary<string, LoginResult>();

        private class LoginResult
        {
            public int UserID { get; set; }
            public string Username { get; set; }
            public string SessionID { get; set; }
            public bool HasPremium { get; set; }
        }

        private static AuthenticationStatus TryLoginUser(string username, string plainTextPassword, out LoginResult login)
        {
            login = null;
            string encryptedPassword;

            // If the password cannot be encrypted, it's invalid!
            try
            {
                encryptedPassword = _outgoingCipher.Encrypt(plainTextPassword);
            }
            catch
            {
                return AuthenticationStatus.InvalidPassword;                
            }

            try
            {                
                using (var authService = CreateServiceClient())
                {
                    var result = authService.v2ValidateClientUser(_siteID, username, encryptedPassword);

                    if (result.Status != ELoginStatus.Success)
                    {
                        return ToAuthenticationStatus(result.Status);
                    }

                    login = new LoginResult
                    {
                        SessionID = result.SessionId,
                        UserID = result.UserId,
                        HasPremium = result.SubscriptionLevel > 0,
                        Username = result.Username
                    };
                }

                return AuthenticationStatus.Success;
            }
            catch (Exception ex)
            {
                AuthenticationLogger.Log("Failed to validate user.", ex);
                return AuthenticationStatus.UnknownError;
            }
        }

        private static AuthenticationStatus TryLoginUser(string username, string encryptedPassword, bool allowCaching, out LoginResult login)
        {
            login = null;

            try
            {
                var plainTextPassword = _incomingCipher.Decrypt(encryptedPassword);

                var cacheKey = "Username:" + username + ";Password;" + plainTextPassword;

                if (allowCaching && _cachedLogins.TryGetValue(cacheKey, out login))
                {
                    return AuthenticationStatus.Success;
                }

                var status = TryLoginUser(username, plainTextPassword, out login);
                if (status == AuthenticationStatus.Success)
                {
                    if (allowCaching)
                    {
                        _cachedLogins.TryAdd(cacheKey, login);
                    }
                }
                return status;
            }
            catch (Exception ex)
            {
                AuthenticationLogger.Log("Failed to validate user.", ex);
                return AuthenticationStatus.UnknownError;
            }

        }

        public static AuthenticationStatus TryAuthenticateUser(string username, bool isPasswordEncrypted,  string password, bool allowCaching, out AuthenticationSession session)
        {

            LoginResult login;

            var authenticationStatus = isPasswordEncrypted ? TryLoginUser(username, password, allowCaching, out login)
                : TryLoginUser(username, password, out login);

            if (authenticationStatus != AuthenticationStatus.Success)
            {
                session = null;
                return authenticationStatus;
            }


            // Create an encryption token
            string tokenValue = null;
            string email = null;
            var hasPremium = login.HasPremium;

            if (_generateAuthTokens)
            {
                User userProfile = null;
                using (var authService = CreateServiceClient())
                {
                    try
                    {
                        if (_requireUserGrants && !UserGrantManager.Instance.HasGrant(login.UserID))
                        {
                            session = null;
                            return AuthenticationStatus.MissingGrant;
                        }

                        try
                        {
                            userProfile = authService.v2GetUserProfile(_siteID, login.UserID);

                            if (userProfile == null)
                            {
                                throw new Exception("Unable to retrieve user profile for user: " + login.UserID);
                            }

                            if (userProfile.Entitlements != null)
                            {
                                hasPremium = hasPremium || userProfile.Entitlements.Any(p => p.Type == ESubscriptionType.Premium && p.Expires >= DateTime.UtcNow);
                            }

                        }
                        catch (Exception ex)
                        {
                            AuthenticationLogger.Log("Failed to retrieve user profile!", ex, new { Username = username });
                            throw;
                        }

                    }
                    catch (Exception ex)
                    {
                        AuthenticationLogger.Log("Exception occurred while validating user!", ex);
                        session = null;
                        return AuthenticationStatus.UnknownError;
                    }
                }

                try
                {
                    var token = EncryptionToken.FromDictonary(new Dictionary<string, string>
                    {
                        {"UserID", login.UserID.ToString(CultureInfo.InvariantCulture)},
                        {"Username", login.Username},
                        {"Email", userProfile.Profile.Emails.First().Address},
                        {"Timestamp", EncryptionToken.ConvertToEpoch(DateTime.UtcNow).ToString(CultureInfo.InvariantCulture)},
                    });

                    tokenValue = token.Value;
                    email = userProfile.Profile.Emails.First().Address;
                }
                catch (Exception ex)
                {
                    AuthenticationLogger.Log("Failed to create authentication token!", ex);
                }

            }

            var effectivePremiumStatus = GetEffectivePremiumStatus(hasPremium);

            session = new AuthenticationSession()
            {
                UserID = login.UserID,
                Username = login.Username,
                SessionID = login.SessionID,
                ActualPremiumStatus = hasPremium,
                EffectivePremiumStatus = effectivePremiumStatus,
                SubscriptionToken = GetSubscriptionToken(login.UserID, effectivePremiumStatus),
                DateLastActive = DateTime.UtcNow,
                EmailAddress = email,
                Token = tokenValue
            };

            return AuthenticationStatus.Success;
        }

        private static AuthenticationStatus ToAuthenticationStatus(ELoginStatus status)
        {
            switch (status)
            {
                case ELoginStatus.InvalidPassword:
                    return AuthenticationStatus.InvalidPassword;
                case ELoginStatus.InvalidSession:
                    return AuthenticationStatus.InvalidSession;
                case ELoginStatus.Success:
                    return AuthenticationStatus.Success;
                case ELoginStatus.UnauthorizedLogin:
                    return AuthenticationStatus.UnauthorizedLogin;
                case ELoginStatus.UnknownEmail:
                    return AuthenticationStatus.UnknownEmail;
                case ELoginStatus.UnknownUsername:
                    return AuthenticationStatus.UnknownUsername;
                default:
                    return AuthenticationStatus.Unsuccessful;
            }
        }

        private static bool GetEffectivePremiumStatus(bool hasPremium)
        {
            if (_freePremium || hasPremium)
            {
                return true;
            }
            else
            {
                return false;

            }
        }

        public static int GetSubscriptionToken(int userID, bool hasPremium)
        {
            return userID ^ (hasPremium ? SubscriptionSeedPremium : SubscriptionSeedNonPremium);
        }

        private static string GetUserProfileCacheKey(int userID)
        {
            return "UserProfile-" + userID;
        }

        public static User GetUserProfile(int userID)
        {
            var cacheKey = GetUserProfileCacheKey(userID);

            var profile = HttpRuntime.Cache.Get(cacheKey) as User;

            if (profile == null)
            {
                using (var authService = CreateServiceClient())
                {
                    profile = authService.v2GetUserProfile(_siteID, userID);
                }

                HttpRuntime.Cache.Insert(cacheKey, profile, null, Cache.NoAbsoluteExpiration, TimeSpan.FromMinutes(10));
            }
            return profile;
        }

        public static bool IsUsernameAvailable(string username)
        {
            using (var authService = CreateServiceClient())
            {
                return authService.v2IsUsernameAvailable(_siteID, username);
            }
        }

        public static RegisterUserResult RegisterUser(string username, string password, string firstname, string lastname, string birthdate, string gender, string country, string region, string city, string email, bool newsletter)
        {
            AddUserResult result = null;
            using (var authService = CreateServiceClient())
            {
                try
                {
                    result = authService.v2AddUser(_siteID,
                        username,
                        _outgoingCipher.Encrypt(password),
                        firstname,
                        lastname,
                        birthdate,
                        gender,
                        country,
                        region,
                        city,
                        email,
                        newsletter,
                        null,
                        null,
                        null);
                }
                catch (Exception ex)
                {
                    AuthenticationLogger.Log("AuthenticationManager - Failed to register user.", ex);
                }
            }

            var registerUserResult = new RegisterUserResult();

            if (result == null)
            {
                AuthenticationLogger.Log("AuthenticationManager - v2AddUser returned empty result");
                registerUserResult.Status = RegisterUserStatus.UnknownError;
            }
            else
            {
                registerUserResult.Status = (RegisterUserStatus)result.Status;
                registerUserResult.UserID = result.UserID;
                registerUserResult.SessionID = result.SessionID;
            }

            return registerUserResult;
        }





    }
}
