﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.ServiceModel.Web;
using System.Web;

namespace Curse.ServiceAuthentication
{
    internal static class IPAddressExtensions
    {
        /// <summary>
        /// Returns the numeric representation of an IPAddress
        /// </summary>
        /// <param name="pIpAddress"></param>
        /// <returns></returns>
        public static Int64 ToInt64(this IPAddress value)
        {

            byte[] ipBytes = value.GetAddressBytes();
            Int64 ip = 0;
            for (int i = 0; i < ipBytes.Length; i++)
            {
                ip <<= 8;
                ip += ipBytes[i];
            }
            return ip;
        }

        /// <summary>
        /// Returns the IP address of the connected WCF client.
        /// </summary>
        /// <param name="operationContext"></param>
        /// <param name="silentFailure"></param>
        /// <returns></returns>
        public static IPAddress GetClientIPAddress(this OperationContext operationContext, bool silentFailure = true)
        {

            if (WebOperationContext.Current == null)
            {
                return GetClientIPAddress(GetOperationContextIPAddress(operationContext));
            }

            return GetClientIPAddress(WebOperationContext.Current.IncomingRequest.Headers["X-Forwarded-For"] ?? GetOperationContextIPAddress(operationContext) ?? string.Empty, silentFailure);            
        }

        private static string GetOperationContextIPAddress(OperationContext context)
        {            
            MessageProperties prop = context.IncomingMessageProperties;
            RemoteEndpointMessageProperty endpoint = prop[RemoteEndpointMessageProperty.Name] as RemoteEndpointMessageProperty;
            
            if (endpoint != null)
            {
                return endpoint.Address;
            }            
            return null;            
        }

        private static IPAddress GetClientIPAddress(string ipString, bool silentFailure = true)
        {
            if (ipString.Contains(","))
            {
                string[] ipArray = ipString.Split(',');
                ipString = ipArray[ipArray.Length - 1].Trim();
            }

            IPAddress ipAddress;
            if (IPAddress.TryParse(ipString, out ipAddress))
            {
                AddressFamily family = ipAddress.AddressFamily;
                if (family == AddressFamily.InterNetwork || family == AddressFamily.InterNetworkV6)
                {
                    byte[] bytes = ipAddress.GetAddressBytes();
                    if (bytes != null && bytes.Length > 0)
                    {
                        return ipAddress;
                    }
                }
            }
            if (silentFailure)
            {
                return IPAddress.None;
            }
            else
            {
                throw new InvalidOperationException(string.Format("Cannot parse IPAddress '{0}'", ipString));
            }
        }
    }
}