﻿using System;
using System.Web;
using System.ServiceModel.Description;
using System.ServiceModel.Dispatcher;
using System.ServiceModel.Channels;
using System.ServiceModel;
using System.ServiceModel.Web;
using Curse.ServiceAuthentication.Models;

namespace Curse.ServiceAuthentication
{    
    public enum AccessLevel
    {
        Anonymous = 0,
        Authenticated = 1,        
        PremiumPromotion = 2,
        PremiumSubscription = 4,
        Moderator = 8,
        Administrator = 16,
        ApiKey = 32
    }
    
    public class RequiresAuthenticationAttribute : Attribute, IOperationBehavior, IParameterInspector
    {
        private static bool _levelsDisabled = false;

        public static void DisableLevels()
        {
            _levelsDisabled = true;
        }

        private readonly AccessLevel _accessLevels;

        public RequiresAuthenticationAttribute(AccessLevel roles)
        {
            _accessLevels = roles;
        }

        #region IOperationBehavior Members

        public void AddBindingParameters(OperationDescription operationDescription, BindingParameterCollection bindingParameters) { }


        public void ApplyClientBehavior(OperationDescription operationDescription, ClientOperation clientOperation) { }        

        public void ApplyDispatchBehavior(OperationDescription operationDescription, DispatchOperation dispatchOperation)
        {
            dispatchOperation.ParameterInspectors.Add(this);
        }

        public void Validate(OperationDescription operationDescription) { }
        
        
        #endregion

        #region Authentication Failure Methods

        private void RaiseAuthenticationFailure(AuthenticationStatus status)
        {

            if (OperationContext.Current.IncomingMessageVersion == MessageVersion.None)
            {
                RaiseWebAuthenticationFailure((int)status);
            }
            else
            {
                RaiseSoapAuthenticationFailure((int)status);
            }
        }

        private void RaiseWebAuthenticationFailure(int status)
        {
            var ctx = WebOperationContext.Current;
            if (ctx == null)
            {
                return;
            }

            ctx.OutgoingResponse.StatusCode = System.Net.HttpStatusCode.Forbidden;
            ctx.OutgoingResponse.SuppressEntityBody = true;
            throw new HttpException(403, "You must login");
        }

        private void RaiseSoapAuthenticationFailure(int status)
        {
            var subCode = new FaultCode("FailedAuthentication");            
            var statusCode = status;
            var faultReason = new FaultReason(statusCode.ToString());
            throw new FaultException(faultReason, subCode);
        }

        #endregion

        #region IParameterInspector Members
       

        public object BeforeCall(string operationName, object[] inputs)
        {
            if (_accessLevels == AccessLevel.ApiKey)
            {
                var apiKey = WebOperationContext.Current.IncomingRequest.Headers["x-api-key"];

                if (string.IsNullOrEmpty(apiKey) || apiKey != AuthenticationProvider.ApiKey)
                {
                    RaiseAuthenticationFailure(AuthenticationStatus.InvalidApiKey);
                }
                return null;
            }
            
            var authContext = AuthenticationContext.Current;

            if (authContext.AuthenticationSession.UserID == 0 && _levelsDisabled)
            {
                return null;
            }

            if (_levelsDisabled || _accessLevels == AccessLevel.Anonymous)
            {
                return null;
            }

            if (authContext.AuthenticationStatus != AuthenticationStatus.Success)
            {
                RaiseAuthenticationFailure(authContext.AuthenticationStatus);
            }

            // AuthenticationSession is null, so raise an invalid session fault
            if (authContext.AuthenticationSession == null || authContext.AuthenticationSession.UserID == 0)
            {
                RaiseAuthenticationFailure(AuthenticationStatus.InvalidSession);
            }

            // User is authenticated. Check their access level:

            // 1. Regular Member Access Level
            if (_accessLevels == AccessLevel.Authenticated)
            {
                return null;
            }

            // 2. Premium Promotion Access Level
            if (_accessLevels == AccessLevel.PremiumPromotion && !authContext.AuthenticationSession.EffectivePremiumStatus)
            {
                RaiseAuthenticationFailure(AuthenticationStatus.SubscriptionExpired);
                return null;
            }

            // 3. Premium Subscription Access Level
            if (_accessLevels == AccessLevel.PremiumSubscription && !authContext.AuthenticationSession.ActualPremiumStatus)
            {
                RaiseAuthenticationFailure(AuthenticationStatus.SubscriptionExpired); 
                return null;
            }

            // 4. Moderator Only
            if (_accessLevels == AccessLevel.Moderator && AuthenticationProvider.CurrentUserProfile.Level < 200)
            {
                RaiseAuthenticationFailure(AuthenticationStatus.InsufficientAccessLevel);
                return null;
            }

            // 5. Administrator Only
            if (_accessLevels == AccessLevel.Administrator && AuthenticationProvider.CurrentUserProfile.Level < 300)
            {
                RaiseAuthenticationFailure(AuthenticationStatus.InsufficientAccessLevel);
                return null;
            }
            
            return null;
        }


        public void AfterCall(string operationName, object[] outputs, object returnValue, object correlationState) { }        

        #endregion


    }

}
