﻿using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.Security.Cryptography;
using System.Threading;
using Curse.Logging;

namespace Curse.ServiceEncryption
{
    public class EncryptionToken
    {
        public static bool IsInitialized;

        /// <summary>
        /// A dictionary of the raw data of the EncryptedCookie
        /// </summary>
        private readonly Dictionary<string, string> _dictionary;
        private readonly bool _isValid;
        private readonly string _tokenValue;

        /// <summary>
        /// The key to use when encryption and decrypting tokens.
        /// </summary>
        private static string _privateKey;

        /// <summary>
        /// The number of iterations to use when creating the key.
        /// </summary>
        private static int _iterations;

        private const int DefaultIterations = 1000;
        private const int FallbackIterations = 1;

        private static int[] _decryptionIterations;

        public static void Initialize(string privateKey, int iterations)
        {
            if (IsInitialized)
            {
                return;
            }

            _privateKey = privateKey;
            _iterations = iterations;

            var decryptionIterations = new List<int>();

            decryptionIterations.Add(iterations);

            if (!decryptionIterations.Contains(DefaultIterations))
            {
                decryptionIterations.Add(DefaultIterations);
            }

            if (!decryptionIterations.Contains(FallbackIterations))
            {
                decryptionIterations.Add(FallbackIterations);
            }

            _decryptionIterations = decryptionIterations.ToArray();

            IsInitialized = true;

            Logger.Info("Encryption token initialized", new { Iterations = _decryptionIterations });
        }

        public string Value
        {
            get { return _tokenValue; }
        }

        public bool IsValid
        {
            get { return _isValid; }
        }

        private static string Decrypt(string encryptedValue, int iteration)
        {            
            // Try to decrypt with the current iterations config
            return EncryptionProvider.Instance.Decrypt(encryptedValue, _privateKey, iteration);         
        }

        private Dictionary<string, string> TryDecodeDictionary(string value)
        {
            foreach (var iteration in _decryptionIterations)
            {
                // Attempt to decrypt the token
                string unencryptedValue = null;
                try
                {
                    unencryptedValue = Decrypt(value, iteration);
                }
                catch
                {
                    continue;
                }

                // Attempt to decode the dictionary                
                try
                {
                    var decodedDictionary = DecodeDictionary(unencryptedValue);

                    if (decodedDictionary != null)
                    {
                        return decodedDictionary;
                    }
                }
                catch 
                {
                    continue;
                }
            }

            return null;
        }

        private static DateTime _lastLoggedFailure = DateTime.UtcNow;
        private static readonly TimeSpan LogThrottle = TimeSpan.FromSeconds(5);
        private static int _totalFailures = 0;

        private void LogTokenFailure(string value)
        {
            Interlocked.Increment(ref _totalFailures);

            if (DateTime.UtcNow.Subtract(_lastLoggedFailure) < LogThrottle)
            {
                return;
            }
            
            Logger.Warn("Failed to decrypt or decode encrypted value token.", new { Value = value, TotalFailures = _totalFailures });
            _lastLoggedFailure = DateTime.UtcNow;
        }

        private EncryptionToken(string encryptedValue)
        {
            if (!IsInitialized)
            {
                throw new Exception("EncryptionToken must be initialized before use.");
            }

            if (string.IsNullOrEmpty(encryptedValue))
            {                
                _dictionary = new Dictionary<string, string>();
                _isValid = false;
                return;
            }

            var decodedDictionary = TryDecodeDictionary(encryptedValue);
            
            if (decodedDictionary == null)
            {
                LogTokenFailure(encryptedValue);
                _dictionary = new Dictionary<string, string>();
                _isValid = false;
                return;
            }
            
            _dictionary = decodedDictionary;
            _isValid = true;            
            _tokenValue = encryptedValue;
        }

        private EncryptionToken(Dictionary<string, string> dictionary)
        {
            if (!IsInitialized)
            {
                throw new Exception("EncryptionToken must be initialized before use.");
            }

            if (dictionary == null)
            {
                throw new ArgumentNullException("dictionary");
            }          

            _dictionary = dictionary;
            var encodedDictionary = EncodeDictionary(_dictionary);
            _tokenValue = EncryptionProvider.Instance.Encrypt(encodedDictionary, _privateKey, _iterations);
            _isValid = true;
        }

        /// <summary>
        /// Create EncryptionToken from a dictionary, producing an encrypted value.
        /// </summary>
        /// <param name="dictionary">The dictionary to use to create the token.</param>
        /// <param name="key">The key to use to encrypt the dictionary.</param>
        public static EncryptionToken FromDictonary(Dictionary<string, string> dictionary)
        {
            return new EncryptionToken(dictionary);
        }
        
        /// <summary>
        /// Create EncryptionToken from an encrypted value.
        /// </summary>
        /// <param name="encryptedValue">The encrypted value to use to create the token.</param>
        public static EncryptionToken FromValue(string encryptedValue)
        {
            return new EncryptionToken(encryptedValue);
        }
      
        /// <summary>
        /// Convert the provided <paramref name="values"/> into an unencrypted string.
        /// </summary>
        /// <param name="values">A dictionary to convert.</param>
        /// <returns>An unencrypted cookie string.</returns>
        private static string EncodeDictionary(IDictionary<string, string> values)
        {
            if (values == null)
            {
                return string.Empty;
            }

            return JsonConvert.SerializeObject(values);            
        }

        /// <summary>
        /// Convert an unencrypted cookie value into a dictionary.
        /// </summary>
        /// <param name="unencryptedValue">The unencrypted cookie value to try to convert.</param>
        /// <returns>A dictionary.</returns>
        private static Dictionary<string, string> DecodeDictionary(string unencryptedValue)
        {
            if (string.IsNullOrEmpty(unencryptedValue))
            {
                return new Dictionary<string, string>();
            }

            return JsonConvert.DeserializeObject<Dictionary<string, string>>(unencryptedValue);              
        }

        public string GetValue(string key)
        {
            if(!_isValid || _dictionary == null)
            {
                throw new InvalidOperationException("The token's dictionary is null or invalid.");
            }

            string value = null;
            if(_dictionary.TryGetValue(key, out value))
            {
                return value;
            }

            return null;
        }

        public int GetInteger(string key)
        {
            var stringValue = GetValue(key);
            if(string.IsNullOrEmpty(stringValue))
            {
                return 0;
            }
            int value;

            if(int.TryParse(stringValue, out value))
            {
                return value;
            }

            return 0;
        }

        public static long ConvertToEpoch(DateTime value)
        {
            var epoch = new DateTime(1970, 1, 1, 0, 0, 0, DateTimeKind.Utc);
            return Convert.ToInt64((value - epoch).TotalSeconds);
        }

        public static DateTime ConvertFromEpoch(long seconds)
        {
            var epoch = new DateTime(1970, 1, 1, 0, 0, 0, DateTimeKind.Utc);
            return epoch.AddSeconds(seconds);
        }

        public DateTime GetDateTime(string key)
        {
            var stringValue = GetValue(key);
            if (string.IsNullOrEmpty(stringValue))
            {
                return default(DateTime);
            }

            long epoch;

            if (!long.TryParse(stringValue, out epoch))
            {
                return default(DateTime);
            }

            return ConvertFromEpoch(epoch);

        }
    }
}
