﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.ServiceModel.Description;
using System.ServiceModel.Dispatcher;
using System.ServiceModel.Channels;
using System.ServiceModel;
using System.ServiceModel.Web;
using Curse.ServiceModels.Authentication;
using System.Web;

namespace Curse.ServiceModels.Attributes
{
    [Flags]
    public enum AccessLevel
    {

        Anonymous = 0,
        Authenticated = 1,
        PremiumPromotion = 2,
        PremiumSubscription = 4,
        Moderator = 8,
        Administrator = 16,
        ApiKey = 32,
        ApiKeyAndUserID = 64
    }

    public class RequiresAuthenticationAttribute : Attribute, IOperationBehavior, IParameterInspector
    {
        private AccessLevel _accessLevels;

        public RequiresAuthenticationAttribute(AccessLevel roles)
        {
            this._accessLevels = roles;
        }

        #region IOperationBehavior Members
        public void AddBindingParameters(OperationDescription operationDescription, BindingParameterCollection bindingParameters) { }

        public void ApplyClientBehavior(OperationDescription operationDescription, ClientOperation clientOperation) { }

        public void ApplyDispatchBehavior(OperationDescription operationDescription, DispatchOperation dispatchOperation)
        {
            dispatchOperation.ParameterInspectors.Add(this);
        }

        public void Validate(OperationDescription operationDescription) { }
        #endregion

        #region Authentication Failure Methods
        private void RaiseAuthenticationFailure(AuthenticationStatus status)
        {
            if (OperationContext.Current.IncomingMessageVersion == MessageVersion.None)
            {
                RaiseWebAuthenticationFailure((int)status);
            }
            else
            {
                RaiseSoapAuthenticationFailure((int)status);
            }
        }

        private void RaiseWebAuthenticationFailure(int status)
        {
            WebOperationContext ctx = WebOperationContext.Current;
            ctx.OutgoingResponse.StatusCode = System.Net.HttpStatusCode.Forbidden;
            ctx.OutgoingResponse.SuppressEntityBody = true;
            throw new HttpException(403, status.ToString());
        }

        private void RaiseSoapAuthenticationFailure(int status)
        {
            var subCode = new FaultCode("FailedAuthentication");
            var faultCode = FaultCode.CreateSenderFaultCode(subCode);
            int statusCode = status;
            var faultReason = new FaultReason(statusCode.ToString());
            throw new FaultException(faultReason, subCode);
        }
        #endregion

        #region IParameterInspector Members
        public object BeforeCall(string operationName, object[] inputs)
        {
            if ((this._accessLevels | AccessLevel.Anonymous) == AccessLevel.Anonymous)
            {
                return null;
            }

            if ((this._accessLevels | AccessLevel.ApiKey) == AccessLevel.ApiKey)
            {
                string apiKey = WebOperationContext.Current.IncomingRequest.Headers["x-api-key"];

                AuthenticationStatus apiStatus = ServiceAuthentication.AuthenticateApiUser(apiKey);
                if (apiStatus != AuthenticationStatus.Success)
                {
                    RaiseAuthenticationFailure(apiStatus);
                }
                return null;
            }

            if ((this._accessLevels | AccessLevel.ApiKeyAndUserID) == AccessLevel.ApiKeyAndUserID)
            {
                string apiKey = WebOperationContext.Current.IncomingRequest.Headers["x-api-key"];

                AuthenticationStatus apiStatus = ServiceAuthentication.AuthenticateApiUser(apiKey, ServiceAuthentication.CurrentSession.UserID);
                if (apiStatus != AuthenticationStatus.Success)
                {
                    RaiseAuthenticationFailure(apiStatus);
                }
                return null;
            }
            
            
            // Request is anonymouse, so raise an invalid session fault
            if (ServiceAuthentication.CurrentAuthToken.IsAnonymous)
            {
                Logger.Log("Anonymous call made to non-anymous operation: {0}", ELogLevel.Debug, operationName);
                RaiseAuthenticationFailure(AuthenticationStatus.InvalidSession);
            }

            AuthenticationStatus authenticationStatus = ServiceAuthentication.AuthenticateUser(ServiceAuthentication.CurrentAuthToken.Username, ServiceAuthentication.CurrentAuthToken.Password);

            if (authenticationStatus != AuthenticationStatus.Success)
            {
                RaiseAuthenticationFailure(authenticationStatus);
            }

            ServiceAuthenticationSession session = ServiceAuthentication.CurrentSession;

            // Session is null, so raise an invalid session fault
            if (session == null)
            {
                RaiseAuthenticationFailure(AuthenticationStatus.InvalidSession);
            }

            // User is authenticated. Check their access level:

            // 1. Regular Member Access Level
            if ((this._accessLevels | AccessLevel.Authenticated) == AccessLevel.Authenticated)
            {
                return null;
            }

            // 2. Premium Promotion Access Level
            if ((this._accessLevels | AccessLevel.PremiumPromotion) == AccessLevel.PremiumPromotion && !session.EffectivePremiumStatus)
            {
                RaiseAuthenticationFailure(AuthenticationStatus.SubscriptionExpired);
                return null;
            }

            // 3. Premium Subscription Access Level
            if ((this._accessLevels | AccessLevel.PremiumSubscription) == AccessLevel.PremiumSubscription && !session.ActualPremiumStatus)
            {
                RaiseAuthenticationFailure(AuthenticationStatus.SubscriptionExpired);
                return null;
            }

            // 4. Moderator Only
            if ((this._accessLevels | AccessLevel.Moderator) == AccessLevel.Moderator && ServiceAuthentication.CurrentUserProfile.Level < 200)
            {
                RaiseAuthenticationFailure(AuthenticationStatus.InsufficientAccessLevel);
                return null;
            }

            // 5. Administrator Only
            if ((this._accessLevels | AccessLevel.Administrator) == AccessLevel.Administrator && ServiceAuthentication.CurrentUserProfile.Level < 300)
            {
                RaiseAuthenticationFailure(AuthenticationStatus.InsufficientAccessLevel);
                return null;
            }

            return null;
        }
        
        public void AfterCall(string operationName, object[] outputs, object returnValue, object correlationState) { }
        #endregion
    }
}
