﻿using System;
using System.Runtime.InteropServices;

namespace Curse
{

    /**
 * Handles very precise timing between the beginning and the end of a task
 *
 * @author Michael Comperda
 */
    public sealed class CPUMonitor
    {
        [DllImport("KERNEL32")]
        private static extern void QueryPerformanceCounter(ref Int64 x);
        [DllImport("KERNEL32")]
        private static extern void QueryPerformanceFrequency(ref Int64 x);

        /**
         * Static constructor
         * Assigns the CPU frequency and calibrates the CPU cycles per second
         */
        static CPUMonitor()
        {
            Int64 start = 0;
            Int64 stop = 0;
            QueryPerformanceFrequency(ref sFrequency);
            for (Int32 count = 0; count < 1000; ++count)
            {
                QueryPerformanceCounter(ref start);
                QueryPerformanceCounter(ref stop);
                sCalibration += stop - start;
            }
            sCalibration /= 1000;
        }

        /**
         * Initialization constructor
         * Assigns a header for the resulting output and sets the starting point for the timer
         */
        public CPUMonitor(String pFormat,
                          params Object[] pArgs)
        {
            mName = String.Format(pFormat, pArgs);
            QueryPerformanceCounter(ref mStarted);
        }

        /**
         * Provides an override of ToString to return a fully formated result from the start till this call
         * Can be called multiple times for stopwatch effect
         * 
         * @return  the string containing formatted output for logging purposes
         */
        public override String ToString()
        {
            Int64 current = 0;
            QueryPerformanceCounter(ref current);
            String elapsed = ((((current - mStarted - sCalibration) * 1000000.0 / sFrequency)) / 1000.0).ToString("0.000");
            return String.Format("{0} took {1} MS", mName, elapsed);
        }

        /**
         * Static member variables
         */
        private static Int64 sCalibration = 0;
        private static Int64 sFrequency = 0;

        /**
         * Member variables
         */
        private String mName;
        private Int64 mStarted = 0;
    }

}
