﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Curse.Extensions
{
    public static class ObjectExtensions
    {
        /// <summary>
        /// Run a <paramref name="selector"/> on the <paramref name="target"/> if it is non-null, otherwise return default(<typeparamref name="TResult"/>).
        /// </summary>
        /// <typeparam name="T">The type of the target</typeparam>
        /// <typeparam name="TResult">The expected result type.</typeparam>
        /// <param name="target">The target to test.</param>
        /// <param name="selector">The function to run, passing in the <paramref name="target"/></param>
        /// <returns>The result of <paramref name="selector"/>, or default(<typeparamref name="TResult"/>)</returns>
        /// <example>
        /// <code>
        /// string text = "Hello";
        /// int length = text.Try(x => x.Length);
        /// text = null;
        /// length = text.Try(x => x.Length);
        /// </code>
        /// </example>
        public static TResult Try<T, TResult>(this T target, Func<T, TResult> selector) where T : class
        {
            return Try<T, TResult>(target, selector, default(TResult));
        }

        /// <summary>
        /// Run a <paramref name="selector"/> on the <paramref name="target"/> if it is non-null, otherwise return <paramref name="defaultValue"/>.
        /// </summary>
        /// <typeparam name="T">The type of the target</typeparam>
        /// <typeparam name="TResult">The expected result type.</typeparam>
        /// <param name="target">The target to test.</param>
        /// <param name="selector">The function to run, passing in the <paramref name="target"/></param>
        /// <param name="defaultValue">The value to return if <paramref name="target"/> is null.</param>
        /// <returns>The result of <paramref name="selector"/>, or <paramref name="defaultValue"/></returns>
        /// <example>
        /// <code>
        /// string text = "Hello";
        /// int length = text.Try(x => x.Length);
        /// text = null;
        /// length = text.Try(x => x.Length);
        /// </code>
        /// </example>
        public static TResult Try<T, TResult>(this T target, Func<T, TResult> selector, TResult defaultValue) where T : class
        {
            if (selector == null)
            {
                throw new ArgumentNullException("selector");
            }

            if (target == null)
            {
                return defaultValue;
            }
            else
            {
                return selector(target);
            }
        }
    }
}
