using System.Text.RegularExpressions;
using System.Net.Mail;
using System.Net;

namespace Curse
{
    public class InputValidation
    {
        static readonly Regex ValidEmail =
            new Regex("^[a-zA-Z0-9\\.\\-_]+@([a-zA-Z0-9\\.\\-_]+\\.)+[a-zA-Z]{2,4}$");

        static readonly Regex ValidAlphaNumUnderscore =
            new Regex("^[a-zA-Z0-9_]+$");

        static readonly Regex ValidNumber =
            new Regex("^-?[0-9]+$");

        static readonly Regex ValidHost =
            new Regex("^([a-zA-Z0-9_\\-]+\\.)+[a-zA-Z]{2,3}$");

        static readonly Regex ValidUrl =
            new Regex("^http://([a-zA-Z0-9_\\-]+\\.)+[a-zA-Z]{2,3}(/[a-zA-Z0-9_\\-\\.]+)*/?$");

        static readonly Regex ValidIp =
            new Regex("^([0-9]{1,3}\\.){3}[0-9]{1,3}$");

        static readonly Regex ValidSession =
            new Regex("^[0-9A-Z]+$");

        static readonly Regex ValidDate =
            new Regex("^[0-9]{4}-[0-9]{2}-[0-9]{2}$");

        static public bool IsValidBirthDate(string date)
        {
            if (string.IsNullOrEmpty(date))
                return true;           

            if (date.Length != 10)
                return false;

            return ValidDate.IsMatch(date);
        }

        static public bool IsValidPassword(string password)
        {
            if (string.IsNullOrEmpty(password))
            {
                return false;
            }
            if (password.Length < 6 || password.Length > 32)
            {
                return false;
            }

            return true;
        }

        static public bool IsValidSession(string session)
        {
            if (session.Length != 16 && session.Length != 48)
                return false;

            return ValidSession.IsMatch(session);
        }

        static public bool IsValidGender(string gender)
        {
            if (string.IsNullOrEmpty(gender) || gender == "1" || gender == "2")
                return true;

            return false;
        }

        static public bool IsValidIp(string ip)
        {
            if (ip.Length < 7 || ip.Length > 15)
                return false;

            try
            {
                IPAddress.Parse(ip);
                return true;
            }
            catch
            {
                return false;
            }

        }

        static public bool IsValidSiteName(string name)
        {
            if (name.Length < 2 || name.Length > 16)
                return false;

            return true;
        }

        static public bool IsValidLastName(string lastName)
        {
            if (string.IsNullOrEmpty(lastName))
                return true;

            if (lastName.Length < 2 || lastName.Length > 32)
                return false;

            return true;
        }

        static public bool IsValidFirstName(string firstname)
        {
            if (firstname == null)
            {
                return false;
            }

            if (firstname == string.Empty)
                return true;

            if (firstname.Length < 2 || firstname.Length > 32)
                return false;

            return true;
        }

        static public bool IsValidRegionCode(string region)
        {
            if (string.IsNullOrEmpty(region))
                return true;

            return IsValidCountryCode(region);
        }

        static public bool IsValidCity(string city)
        {
            if (string.IsNullOrEmpty(city))
                return true;

            if (city.Length < 2 || city.Length > 32)
                return false;

            return true;
        }

        static public bool IsValidCountryCode(string country)
        {
            if (country.Length != 2)
                return false;

            return ValidAlphaNumUnderscore.IsMatch(country);
        }

        static public bool IsValidUserName(string name)
        {
            if (name.Length < 2 || name.Length > 32)
                return false;

            return ValidAlphaNumUnderscore.IsMatch(name);
        }

        static public bool IsValidNumber(string num)
        {
            if (num.Length < 1 || num.Length > 11)
                return false;

            return ValidNumber.IsMatch(num);
        }

        static public bool IsValidEmail(string email)
        {
            if (email.Length < 5 || email.Length > 64)
                return false;

            try
            {
                new MailAddress(email);
            }
            catch
            {
                return false;
            }

            return true;
        }

        static public bool IsValidUrl(string url)
        {
            if (url.Length < 11)
                return false;

            return ValidUrl.IsMatch(url);
        }

        static public bool IsValidIconUrl(string url)
        {
            if (url.Length > 32)
                return false;

            return IsValidUrl(url);
        }

        static public bool IsValidReferrerHost(string host)
        {
            if (host.Length < 4 || host.Length > 32)
                return false;

            return ValidHost.IsMatch(host);
        }

        static public bool IsValidUpdateUrl(string url)
        {
            if (url.Length > 64)
                return false;

            return ValidUrl.IsMatch(url);
        }

        static public bool IsValidHost(string host)
        {
            if (host.Length < 4 || host.Length > 32)
                return false;

            return ValidHost.IsMatch(host);
        }

    }
}
