﻿using System;
using System.Collections.Generic;
using System.Diagnostics;

namespace Curse
{
    /**
     * Class for reading and writing UInt16 elements in a list format
     * Based on inheritance of List<UInt16> and implements IPackable
     * Includes the Count in the packed data form
     *
     * @author Michael Comperda
     */
    public sealed class PackableUInt16List
        : List<UInt16>, IPackable
    {
        /**
         * Reads the list count and the elements from pBuf
         *
         * @param  pBuf the IReadable buffer in which to read from
         * @return      true if list count and all the elements could be read, false otherwise
         */
        public Boolean Read(IReadable pBuf)
        {
            UInt16 count = 0;
            if (!pBuf.Read(ref count))
            {
                return false;
            }
            Capacity = count;
            Boolean ok = true;
            UInt16 buf = 0;
            for (Int32 i = 0;
                 ok &&
                 i < Capacity;
                 ++i)
            {
                if ((ok = pBuf.Read(ref buf)))
                {
                    Add(buf);
                }
            }
            return ok;
        }

        /**
         * Writes the list count and elements to pBuf
         * Asserts if Count > UInt16.MaxValue
         *
         * @param  pBuf the IWriteable buffer in which to write to
         * @return      true if list count and all the elements could be written, false otherwise
         */
        public Boolean Write(IWriteable pBuf)
        {
            Debug.Assert(Count <= UInt16.MaxValue);
            return pBuf.Write(Convert.ToUInt16(Count)) &&
                   TrueForAll(b => pBuf.Write(b));
        }
    }
}
