﻿using System;
using System.Collections.Generic;
using System.Threading;
using System.Configuration;
using System.IO;
using Curse.GameServers.Configuration;
using System.Data.SqlClient;
using Curse.GameServers.Extensions;
using Curse.Extensions;
using System.Linq;
using System.Runtime.Serialization.Formatters.Binary;

namespace Curse.GameServers.Caching
{
    public class GameReleaseCache
    {
        private Dictionary<int, CGameRelease> _gameReleases;
        private object _cacheLock = new object();

        private string _staticFileDestination;
        private bool _createFeedFiles;
        
        private Thread _updateThread = null;
        private int _updateThreadInterval;
        private DateTime _lastUpdateTime = DateTime.Parse("01/01/2012");

        static GameReleaseCache _instance = new GameReleaseCache();
        public static GameReleaseCache Instance { get { return _instance; } }

        public GameReleaseCache()
        {
            _gameReleases = new Dictionary<int, CGameRelease>();

            _staticFileDestination = ConfigurationManager.AppSettings["FeedPath"];
            if (_createFeedFiles && !Directory.Exists(_staticFileDestination))
            {
                Directory.CreateDirectory(_staticFileDestination);
            }
            _createFeedFiles = (System.Environment.MachineName.ToLower() == ConfigurationManager.AppSettings["JobMachineName"].ToLower());

            UpdateCache();

            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }
        public void Initialize() { }

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }
        private void UpdateCache()
        {
            var gameReleases = new Dictionary<int, CGameRelease>(_gameReleases);
            var updateTime = DateTime.UtcNow;

            using (var conn = DatabaseConfiguration.GetGameServerConnection())
            {
                var lastQueryTime = _lastUpdateTime.AddSeconds(-30);

                var query = "SELECT * FROM GameRelease WHERE DateModified > @lastUpdateTime";

                var cmd = new SqlCommand(query, conn);
                cmd.Parameters.Add("lastUpdateTime", System.Data.SqlDbType.DateTime).Value = lastQueryTime;
                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var gameReleaseID = reader.GetInt32(reader.GetOrdinal("ID"));

                        var gameRelease = new CGameRelease();
                        gameRelease.SetFromDataReader(reader);

                        if (gameReleases.ContainsKey(gameReleaseID))
                        {
                            gameReleases.Remove(gameReleaseID);
                        }
                        gameReleases.Add(gameReleaseID, gameRelease);
                    }

                    lock (_cacheLock)
                    {
                        _gameReleases = gameReleases;
                    }
                }

                lock (_cacheLock)
                {
                    _gameReleases = gameReleases;
                }               

                _lastUpdateTime = updateTime;

                var usableReleases = gameReleases.Values.Where(p => p.ReleaseType != EGameReleaseType.None).ToList();
                try
                {
                    SaveToDisk(usableReleases);
                }
                catch (Exception exc)
                {
                    Logger.Log("Unable to create GameRelease feed file! Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
                }

                GC.Collect();
            }
        }

        #region Methods
        public List<CGameRelease> GetAllReleases()
        {
            return _gameReleases.Values.ToList();
        }

        public List<CGameRelease> GetAllReleasesByGameID(int gameId)
        {
            return _gameReleases.Values.Where(p => p.GameID == gameId).ToList();
        }

        public CServiceResponse UpdateReleaseNotes(int releaseId, string releaseNotes, string releaseNotesLink)
        {
            using (SqlConnection conn = DatabaseConfiguration.GetGameServerConnection())
            {
                var cmd = new SqlCommand("UPDATE GameRelease SET ReleaseNotes = @ReleaseNotes, ReleaseNotesLink = @ReleaseNotesLink, DateModified = GETUTCDATE() WHERE ID = @ID", conn);
                cmd.Parameters.Add("ID", System.Data.SqlDbType.Int).Value = releaseId;
                cmd.Parameters.Add("ReleaseNotes", System.Data.SqlDbType.NVarChar).Value = releaseNotes;
                cmd.Parameters.Add("ReleaseNotesLink", System.Data.SqlDbType.NVarChar).Value = releaseNotesLink;

                try
                {
                    cmd.ExecuteNonQuery();
                    return new CServiceResponse(EServiceResponseStatus.Successful);
                }
                catch (Exception exc)
                {
                    return new CServiceResponse(EServiceResponseStatus.UnknownException, exc.Message);
                }
            }
        }
        #endregion

        #region Feed File Generation

        private void SaveToDisk(List<CGameRelease> releases)
        {
            if (_createFeedFiles)
            {
                SaveBinaryToDisk(releases);
            }
        }

        private void SaveBinaryToDisk(List<CGameRelease> releases)
        {
            using (MemoryStream ms = new MemoryStream())
            {
                BinaryFormatter bf = new BinaryFormatter();
                bf.Serialize(ms, releases);
                ms.Position = 0;

                byte[] uncompressed = ms.ToArray();
                byte[] compressed = Utility.GetCompressedBytes(uncompressed);

                string basePath = _staticFileDestination;
                string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string backupfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string currentfile = Path.Combine(basePath, "GameReleases.zip");
                try
                {
                    using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
                    {
                        binWriter.Write(compressed);
                    }
                }
                catch (Exception ex)
                {
                    throw ex;
                }

                if (File.Exists(currentfile) && FileComparer.CompareFiles(currentfile, tempfile))
                {
                    try
                    {
                        File.Delete(tempfile);
                    }
                    catch (Exception ex)
                    {
                        Logger.Log("CGameReleaseCache - Unable to delete temp file! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                    }
                    return;
                }

                try
                {
                    if (!File.Exists(currentfile))
                    {
                        File.Copy(tempfile, currentfile, true);
                    }
                    else
                    {
                        File.Replace(tempfile, currentfile, backupfile, true);
                    }
                }
                catch (Exception ex)
                {
                    Logger.Log("CGameReleaseCache - SaveToDisk Exception! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                }
                finally
                {
                    File.Delete(tempfile);
                    if (File.Exists(backupfile))
                    {
                        File.Delete(backupfile);
                    }
                }
            }
        }
        #endregion
    }
}