﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using Curse.GameServers.Configuration;
using Curse.GameServers.Extensions;

namespace Curse.GameServers.Caching
{
    public class GameServerDataSourceCache
    {
        private List<CGameServerDataSource> _gameServerDataSources;
        private object _cacheLock = new object();

        private Thread _updateThread = null;
        private int _updateThreadInterval;
        private bool _isCacheBuilt;
        
        static GameServerDataSourceCache _instance = new GameServerDataSourceCache();
        public static GameServerDataSourceCache Instance { get { return _instance; } }

        public GameServerDataSourceCache()
        {
            _gameServerDataSources = new List<CGameServerDataSource>();

            UpdateCache();

            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }
        public void Initialize() { }

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }
        public void UpdateCache()
        {
            var dataSources = new List<CGameServerDataSource>();
            using (var conn = DatabaseConfiguration.GetGameServerConnection())
            {
                var cmd = new SqlCommand("SELECT * FROM GameServerDataSource", conn);
                using (var reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var dataSource = new CGameServerDataSource();
                        dataSource.SetFromDataReader(reader);
                        dataSources.Add(dataSource);
                    }

                    lock (_cacheLock)
                    {
                        _gameServerDataSources = dataSources;
                    }
                }
            }

            _isCacheBuilt = true;
        }

        public CGameServerDataSource GetByID(int dataSourceId)
        {
            if (!_isCacheBuilt)
            {
                return null;
            }

            return _gameServerDataSources.FirstOrDefault(p => p.ID == dataSourceId);
        }
        public CGameServerDataSource GetByName(string name)
        {
            return _gameServerDataSources.FirstOrDefault(p => p.Name == name);
        }
        public List<CGameServerDataSource> GetAllDataSources()
        {
            if (!_isCacheBuilt)
            {
                return null;
            }
            return _gameServerDataSources;
        }
        public List<CGameServerDataSource> GetDataSourcesByGameID(int gameId)
        {
            if (!_isCacheBuilt)
            {
                return null;
            }
            return _gameServerDataSources.Where(p => p.GameID == gameId).ToList();
        }
    }
}