﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Caching;
using Curse.Extensions;

namespace Curse.GameServers.Extensions
{
    public static class CacheExtensions
    {
        private const int FakeNullValue = int.MinValue;
        private static TimeSpan _defaultCacheDuration = TimeSpan.FromMinutes(30);
        private static TimeSpan _nullCacheDuration = TimeSpan.FromSeconds(30);

        public static T Get<T>(this Cache cache, string key, Func<T> func, Action<Cache, string, T> setter, bool bypassCache)
        {

            object result = null;

            if (!bypassCache)
            {
                result = HttpRuntime.Cache.Get(key);
            }

            if (result == null)
            {
                T value = func.Invoke();
                setter.Invoke(cache, key, value);
                return (T)value;
            }
            else if (result is T)
            {
                return (T)result;
            }
            else
            {
                return default(T);
            }
        }

        public static T Get<T>(this Cache cache, bool bypassCache, string key, Func<T> func)
        {
            return cache.Get(key, func, (c, k, result) => c.Replace(k, result), bypassCache);
        }

        public static T Get<T>(this Cache cache, string key, Func<T> func)
        {
            return cache.Get(key, func, (c, k, result) => c.Replace(k, result), false);
        }

        public static void Expire(this Cache cache, string keyFormat, params string[] keyValues)
        {
            HttpRuntime.Cache.Expire(keyFormat.FormatWith(keyValues));
        }

        public static void Expire(this Cache cache, string key)
        {
            cache.Remove(key);
        }

        public static void Replace(this Cache cache, string key, object value, TimeSpan cacheDuration)
        {
            if (value == null)
            {
                value = FakeNullValue;
                cacheDuration = _nullCacheDuration;
            }

            cache.Insert(key, value, null, DateTime.UtcNow + cacheDuration, Cache.NoSlidingExpiration, CacheItemPriority.Normal, null);
        }

        public static void Replace(this Cache cache, string key, object value)
        {
            cache.Replace(key, value, _defaultCacheDuration);
        }
    }
}