﻿using Curse;
using Curse.Auth;

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.IO;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Web;
using System.Web.Services;
using System.Web.Services.Protocols;

namespace HardwareCenter
{
    /**
     * Hardware Front-End Web Service Class
     * Inherits AuthenticatableWebService
     *
     * @author Shane Bryldt
     */
    [WebService(Namespace = "http://data.curse.com/")]
    [WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
    [ToolboxItem(false)]
    public class HardwareService
        : AuthenticatableWebService
    {
        /**
         * Static constructor
         * Does one-time initialization of the hardware service
         */
        static HardwareService()
        {
            Logger.SetLogLevel = (ELogLevel)Array.IndexOf<String>(Enum.GetNames(typeof(ELogLevel)),
                                                                  ConfigurationManager.AppSettings["LogLevel"]);
            Logger.SetLogPath = ConfigurationManager.AppSettings["LogPath"];
            sClientCipher = new StringCipher(ConfigurationManager.AppSettings["ClientKey"]);
            String authService = ConfigurationManager.AppSettings["AuthenticationService"];
            Int32 authId = Convert.ToInt32(ConfigurationManager.AppSettings["AuthenticationId"]);
            String authKey = ConfigurationManager.AppSettings["AuthenticationKey"];
            Int32 sessionExpiration = Convert.ToInt32(ConfigurationManager.AppSettings["SessionExpiration"]);
            sMaxQueuedUpdates = Convert.ToInt32(ConfigurationManager.AppSettings["MaxQueuedUpdates"]);

            sAuthentication = new Authentication(authService,
                                                 authId,
                                                 authKey,
                                                 sessionExpiration);

            sHardwareDB = ConfigurationManager.ConnectionStrings["HardwareDB"].ConnectionString;
            DB.Load(sHardwareDB);

            sUpdateThread = new Thread(ProcessUpdateThread);
            sUpdateThread.Priority = ThreadPriority.AboveNormal;
            sUpdateThread.Start();

            Logger.Log(ELogLevel.Info,
                       null,
                       "Ready");
        }

        /**
         * Publically exposed web method for uploading hardware updates
         * Parameters are taken through form fields
         * Returns a StatusCode in the response stream
         */
        [WebMethod]
        public void UploadUpdate()
        {
            try
            {
                String session = Context.Request.Form["pSession"];

                if (Context.Request.Files.Count == 0)
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "No file to read from the stream");
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.InvalidStream);
                    return;
                }

                Int32 userId;
                StatusCode status = sAuthentication.ValidateSessionKeyToUserId(session, out userId);
                
                if (status != StatusCode.Ok)
                {
                    Logger.Log(ELogLevel.Access,
                               Context.Request.UserHostAddress,
                               "ValidateSessionKeyToUserId failed with status {0}",
                               status);
                    Context.Response.OutputStream.WriteByte((Byte)status);
                    return;
                }
                Package buf = new Package(Encoding.UTF8, Context.Request.Files[0].InputStream);

#if Debug
                buf.DumpToFile(ConfigurationManager.AppSettings["DumpPath"] + userId.ToString() + ".txt",
                               DateTime.Now.ToString("yyyy-MM-dd-HHmmss") + "] Compressed is " + buf.Length + " bytes");
#endif


                if (!buf.Decompress())
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "Invalid compressed update file stream");
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.InvalidStream);
                    return;
                }

#if Debug
                buf.DumpToFile(ConfigurationManager.AppSettings["DumpPath"] + userId.ToString() + ".txt",
                               DateTime.Now.ToString("yyyy-MM-dd-HHmmss") + "] Decompressed is " + buf.Length + " bytes");
#endif


                Update update = new Update();
                if (!update.Read(buf))
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "Invalid update file stream");
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.InvalidStream);
                    return;
                }

                Boolean overworked = false;
                lock (sUpdateQueue)
                {
                    overworked = sUpdateQueue.Count >= sMaxQueuedUpdates;
                    if (!overworked)
                    {
                        sUpdateQueue.Enqueue(new UpdateThreadParam(userId,
                                                                   Context.Request.UserHostAddress,
                                                                   update));
                    }
                }
                if (overworked)
                {
                    Logger.IgnoredAction(userId,
                                         Context.Request.UserHostAddress,
                                         "Update",
                                         "Update queue has exceeded maximum capacity");
                }

                Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
                return;
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "UploadUpdate Exception: {0}",
                           exc.Message);
            }

            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Unknown);
            return;
        }

        /**
         * Private update processing thread
         */
        private static void ProcessUpdateThread()
        {
            Boolean aborted = false;
            UpdateThreadParam param;
            while (!aborted)
            {
                try
                {
                    param = null;
                    lock (sUpdateQueue)
                    {
                        if (sUpdateQueue.Count > 0)
                        {
                            param = sUpdateQueue.Dequeue();
                        }
                    }

                    if (param == null)
                    {
                        Thread.Sleep(1);
                        continue;
                    }

                    Int32 userId = ((UpdateThreadParam)param).ID;
                    String userHost = ((UpdateThreadParam)param).Host;
                    Update update = ((UpdateThreadParam)param).Update;

                    DB.UpdateCPU(userId,
                                 userHost,
                                 update.CPU, 
                                 update.IsLaptop);
                    DB.UpdateGPUs(userId,
                                  userHost,
                                  update.GPUs);
                    DB.UpdateHDs(userId,
                                 userHost,
                                 update.HDs);
                    DB.UpdateRAM(userId,
                                 userHost,
                                 update.RAM);
                    DB.UpdateOS(userId,
                                userHost,
                                update.OS);
                    DB.UpdateSound(userId,
                                   userHost,
                                   update.Sound);
                    DB.UpdateDisplay(userId,
                                     userHost,
                                     update.Display);
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "ProcessUpdateThread Aborted");
                }
                catch (Exception exc)
                {
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "ProcessUpdateThread Exception: {0}",
                               exc.Message);
                    Logger.Log(ELogLevel.Debug,
                               null,
                               exc.StackTrace);
                }
            }
        }

        /**
         * Private static member data
         */
        private static Int32 sMaxQueuedUpdates = 0;
        private static String sHardwareDB = null;
        private static Thread sUpdateThread = null;
        private static Queue<UpdateThreadParam> sUpdateQueue = new Queue<UpdateThreadParam>();

        /**
         * Internal class for passing update thread parameters
         */
        private class UpdateThreadParam
        {
            public Int32 ID;
            public String Host;
            public Update Update;

            /**
             * Initialization constructor
             */
            public UpdateThreadParam(Int32 pId,
                                     String pHost,
                                     Update pUpdate)
            {
                ID = pId;
                Host = pHost;
                Update = pUpdate;
            }
        }
    }
}
