﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Curse.Logging;

namespace Curse.Jobs
{
    public abstract class BaseJob
    {
        protected abstract int RunIntervalMinutes { get; }
        protected readonly LogCategory Logger;

        private bool _isRunning;

        protected BaseJob()
        {
            Logger = new LogCategory(GetType().Name);
        }

        public void Run(CancellationToken cancelToken)
        {
            if (_isRunning)
            {
                return;
            }

            _isRunning = true;

            var sw = Stopwatch.StartNew();

            try
            {
                DoRun(cancelToken);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error occurred while running job");
            }
            finally
            {                
                _isRunning = false;
                Logger.Info("Job completed in " + sw.Elapsed.TotalSeconds.ToString("##0.00") + " seconds");
            }
        }

        /// <summary>
        /// Note: This expects the work to be complete before returning. If using background threads, ensure they are Joined before returning.
        /// </summary>
        /// <param name="cancelToken">Token to perform cancellation.</param>
        protected abstract void DoRun(CancellationToken cancelToken);

        public bool ShouldRun(DateTime lastRunDate)
        {

            if (DateTime.UtcNow - lastRunDate < TimeSpan.FromMinutes(RunIntervalMinutes))
            {
                return false;
            }

            if (_isRunning)
            {
                Logger.Warn("Unable to start job, it is already running.");
                return false;

            }

            return true;
        }
    }
}
