﻿using System;
using System.IO;
using System.Linq;
using System.Net;
using System.Security.AccessControl;
using System.Text.RegularExpressions;
using Curse.Logging;
using Curse.Minecraft.Jobs.Utilities;
using ICSharpCode.SharpZipLib.Zip;
using Newtonsoft.Json;

namespace Curse.Minecraft.Jobs.Forge
{
    internal class ForgeFileProcessor
    {
        private readonly string _name;
        private readonly string _mcVersion;
        private readonly LogCategory Logger;

        public enum ForgePackageType
        {
            Installer,
            Client
        }

        public ForgeFileProcessor(string name, string mcVersion)
        {
            _name = name;
            _mcVersion = mcVersion;
            Logger = new LogCategory(_name);
        }

        public ForgeFileDetails Process(ForgePackageType type, ForgeDownloadInfo forgeDownloadInfo)
        {
            switch (type)
            {
                case ForgePackageType.Client:
                    return ProcessForgeClient(forgeDownloadInfo);

                case ForgePackageType.Installer:
                    return ProcessForgeInstaller(forgeDownloadInfo);
            }

            return null;
        }

        private ForgeFileDetails ProcessForgeClient(ForgeDownloadInfo forgeDownloadInfo)
        {
            Console.WriteLine("Processing forge build " + forgeDownloadInfo.DownloadUrl);

            // Download forge version
            var target = Path.GetTempFileName();
            var modLoaderUrl = ModLoaderHelper.GetModLoaderUrl(forgeDownloadInfo.DownloadUrl, target, true);

            // Copy this down to a new location            
            forgeDownloadInfo.DownloadUrl = modLoaderUrl.FullUrl;
            forgeDownloadInfo.FileName = Path.GetFileName(modLoaderUrl.FullUrl.LocalPath);
            return new ForgeFileDetails();
        }

        private ForgeFileDetails ProcessForgeInstaller(ForgeDownloadInfo forgeDownloadInfo)
        {
            Console.WriteLine("Downloading forge from " + forgeDownloadInfo.DownloadUrl);
            const string installerProfileFilename = "install_profile.json";

            // Download forge version
            var installerFile = Path.GetTempFileName();
            var jarFile = Path.GetTempFileName();

            try
            {
                // Try to download the file
                var modLoaderUrl = ModLoaderHelper.GetModLoaderUrl(forgeDownloadInfo.DownloadUrl, installerFile, true);
                
                Logger.Info(string.Format("Extracting {0} from forge archive", installerProfileFilename));
                ForgeInstallProfile result;

                // Extract installer profile from archive
                using (var forgeArchive = new ZipFile(installerFile))
                {
                    // Get the installer profile
                    var installerProfileEntry = forgeArchive.GetEntry(installerProfileFilename);

                    var zipStream = forgeArchive.GetInputStream(installerProfileEntry);

                    var serializer = new JsonSerializer();



                    using (var streamReader = new StreamReader(zipStream))
                    {
                        using (var textReader = new JsonTextReader(streamReader))
                        {
                            result = serializer.Deserialize<ForgeInstallProfile>(textReader);
                        }
                    }


                    if (result == null)
                    {
                        throw new IOException("Cannot extract installer_profile.json form forge archive.");
                    }

                    var jarPath = result.install.filePath;
                    var jarEntry = forgeArchive.GetEntry(jarPath);
                    if (jarEntry == null)
                    {
                        throw new IOException("Cannot extract jar form forge archive.");
                    }

                    using (var stream = forgeArchive.GetInputStream(jarEntry))
                    {
                        using (var outputStream = new FileStream(jarFile, FileMode.Truncate))
                        {
                            stream.CopyTo(outputStream);
                        }
                    }
                }

                // Ensure it can be opened as a zip
                if (!ZipHelper.IsValidZip(jarFile))
                {
                    throw new Exception("Unable to open zip file");
                }

                ApplyJarFixes(jarFile, result);

                // Apply any version specific changes

                // Get the loader
                var forgeLib = result.versionInfo.libraries.Select(p => new MavenPackageString(p.name, new Uri(p.url, UriKind.Absolute)))
                    .FirstOrDefault(p => p.Package.Equals("net.minecraftforge") || p.Package.Equals("net.forge"));

                if (forgeLib == null)
                {
                    return null;
                }

                var newUrl = ModLoaderHelper.GetModLoaderUrl(forgeLib, jarFile);
                forgeDownloadInfo.DownloadUrl = newUrl.FullUrl;
                forgeDownloadInfo.FileName = forgeLib.FileName;

                // Read the install_profile.json to determine version specific /libraries location for the forge.jar
                return ProcessInstallerProfile(result, jarFile);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Cannot process forge archive.");
                throw;
            }
            finally
            {
                try
                {
                    File.Delete(installerFile);
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex);
                }

                try
                {
                    File.Delete(jarFile);
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex);
                }
            }
        }


        private void ApplyJarFixes(string jarFile, ForgeInstallProfile profile)
        {
            if (profile.install.minecraft.Equals("1.5.2"))
            {
                // Clear the META-INF in the minecraft JAR. Why do we have to do this? Only LexManos knows.
                ClearMetaInf(jarFile);
            }
        }

        /// <summary>
        ///     Clears all META-INF from a Jar file.
        /// </summary>
        /// <param name="jarfile"></param>
        private void ClearMetaInf(string jarfile)
        {
            try
            {
                using (var zip = System.IO.Compression.ZipFile.Open(jarfile, System.IO.Compression.ZipArchiveMode.Update))
                {
                    var metaEntries = zip.Entries.Where(p => p.FullName.ToUpperInvariant().StartsWith("META-INF")).ToArray();
                    foreach (var entry in metaEntries)
                    {
                        entry.Delete();
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[Minecraft] Error clearing META-INF from " + jarfile);
            }
        }

        /// <summary>
        ///     Parses the installer profile file within the forge archive
        /// </summary>        
        private ForgeFileDetails ProcessInstallerProfile(ForgeInstallProfile profile, string jarFilePath)
        {
            var versionInfo = profile.versionInfo;           
            var packageString = new MavenPackageString(profile.install.path);
            return new ForgeFileDetails(packageString.ToString(), packageString.LibPath, ProcessVersionInfoJson(versionInfo, jarFilePath));            
        }

        /// <summary>
        /// Processes custom minecraft version.json located within Forge archive
        /// </summary>
        private string ProcessVersionInfoJson(ForgeInstallProfileVersionInfo versionInfo, string jarFilePath)
        {            
            versionInfo.id = _name;


            var libsToInstall = versionInfo.libraries.Where(l => !string.IsNullOrEmpty(l.url)).ToArray();

            foreach (var lib in libsToInstall)
            {
                var package = new MavenPackageString(lib.name, new Uri(lib.url, UriKind.Absolute));

                if (package.Package.Equals("net.minecraftforge") || package.Package.Equals("net.forge"))
                {
                    var newUrl = ModLoaderHelper.GetModLoaderUrl(package, jarFilePath);
                    lib.url = newUrl.PartialUrl.ToString();
                    continue;
                }

                try
                {
                    var newUrl = ModLoaderHelper.GetModLoaderUrl(package);
                    lib.url = newUrl.PartialUrl.ToString();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to download or process library: " + package.Name);
                    throw;
                }
            }

            var settings = new JsonSerializerSettings { NullValueHandling = NullValueHandling.Ignore };
            var baseJson = JsonConvert.SerializeObject(versionInfo, Formatting.Indented, settings);
            return UpdateJsonBasedOnVersion(baseJson);
           
        }

        private string UpdateJsonBasedOnVersion(string json)
        {
            var version = VersionJson.Load(json);
            version.UpdateBasedOnVersion(_name, _mcVersion);
            return version.Save();
        }
    }
}
