﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using Curse.Logging;
using Newtonsoft.Json;

namespace Curse.Minecraft.Jobs.Forge
{
    class VersionJson
    {

        [JsonProperty(PropertyName = "id")]
        public string Id { get; set; }

        [JsonProperty(PropertyName = "time")]
        public string Time { get; set; }

        [JsonProperty(PropertyName = "releaseTime")]
        public string ReleaseTime { get; set; }

        [JsonProperty(PropertyName = "type")]
        public string Type { get; set; }

        [JsonProperty(PropertyName = "minecraftArguments")]
        public string MinecraftArguments { get; set; }

        [JsonProperty(PropertyName = "minimumLauncherVersion")]
        public int MinimumLauncherVersion { get; set; }

        [JsonProperty(PropertyName = "assets")]
        public string Assets { get; set; }

        [JsonProperty(PropertyName = "inheritsFrom")]
        public string InheritsFrom { get; set; }

        [JsonProperty(PropertyName = "jar")]
        public string Jar { get; set; }

        [JsonProperty(PropertyName = "libraries")]
        public List<Library> Libraries { get; set; }

        [JsonProperty(PropertyName = "mainClass")]
        public string MainClass { get; set; }

        public static VersionJson Load(string json)
        {
            try
            {
                return JsonConvert.DeserializeObject<VersionJson>(json);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[Minecraft] Could not deserialize Minecraft version.json");
                return null;
            }
        }

        public string Save()
        {
            try
            {
                var settings = new JsonSerializerSettings { NullValueHandling = NullValueHandling.Ignore };
                return JsonConvert.SerializeObject(this, Formatting.Indented, settings);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error serializing Minecraft version.json");
                throw;
            }
        }

        public void UpdateBasedOnVersion(string versionName, string minecraftVersion)
        {
            if (minecraftVersion == "1.7.10")
            {
                Assets = "1.7.10";
            }
            else if (minecraftVersion == "1.5.2")
            {
                // Change the startup class
                Logger.Info("Setting tweak class to net.minecraft.launchwrapper.VanillaTweaker");
                var args = MinecraftArguments.Split(' ').ToList();

                // Sub out the --tweakClass argument for the vanilla one. We've already installed dependencies
                var i = args.IndexOf("--tweakClass");
                args[i + 1] = "net.minecraft.launchwrapper.VanillaTweaker";
                MinecraftArguments = String.Join(" ", args);
            }
            else if (minecraftVersion.StartsWith("1.8"))
            {
                Assets = "1.8";
            }

            InheritsFrom = minecraftVersion;
            Jar = versionName;
        }

        [Obfuscation(ApplyToMembers = true)]
        public class Extract
        {
            [JsonProperty(PropertyName = "exclude")]
            public List<string> Exclude { get; set; }
        }

        [Obfuscation(ApplyToMembers = true)]
        public class Library
        {
            [JsonProperty(PropertyName = "name")]
            public string Name { get; set; }
            [JsonProperty(PropertyName = "url")]
            public string Url { get; set; }
            [JsonProperty(PropertyName = "serverreq")]
            public bool? ServerReq { get; set; }
            [JsonProperty(PropertyName = "clientreq")]
            public bool? Clientreq { get; set; }
            [JsonProperty(PropertyName = "natives")]
            public Natives Natives { get; set; }
            [JsonProperty(PropertyName = "extract")]
            public Extract Extract { get; set; }
            [JsonProperty(PropertyName = "rules")]
            public List<Rule> Rules { get; set; }
        }

        [Obfuscation(ApplyToMembers = true)]
        public class Natives
        {
            [JsonProperty(PropertyName = "linux")]
            public string Linux { get; set; }
            [JsonProperty(PropertyName = "windows")]
            public string Windows { get; set; }
            [JsonProperty(PropertyName = "osx")]
            public string OSX { get; set; }
        }

        [Obfuscation(ApplyToMembers = true)]
        public class Os
        {
            [JsonProperty(PropertyName = "name")]
            public string Name { get; set; }
        }

        [Obfuscation(ApplyToMembers = true)]
        public class Rule
        {
            [JsonProperty(PropertyName = "action")]
            public string Action { get; set; }
            [JsonProperty(PropertyName = "os")]
            public Os Os { get; set; }
        }
    }
}
