﻿using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Data.SqlTypes;
using System.Linq;
using System.Threading;
using Curse.Database.Helpers;
using Curse.Jobs;
using Curse.Logging;
using Curse.Minecraft.Jobs.Forge;
using Curse.Minecraft.Jobs.Minecraft;
using Curse.Minecraft.Jobs.Utilities;
using Curse.Minecraft.Models;
using Newtonsoft.Json;
using Curse.AddOns;
using Curse.Jobs.Runner.Configuration;

namespace Curse.Minecraft.Jobs
{
    [Export(typeof(BaseJob))]
    public class SniffForgeVersionsJob : BaseJob
    {
        protected override int RunIntervalMinutes
        {
            get { return 20; }
        }

        private Dictionary<string, MinecraftGameVersion> _minecraftGameVersions;
        private readonly HashSet<string> _unknownMinecraftVersions = new HashSet<string>();

        protected override void DoRun(CancellationToken cancelToken)
        {
            _minecraftGameVersions = MinecraftGameVersionExtensions.GetAll().ToDictionary(p => p.VersionString);

            // The Gradle List is newer and preferred over legacy
            BuildForgeGradleList();

            // Legacy is disabled, but the code will remain for the time being
            // The Gradle List most likely has all these versions, but this is left just in case that changes
            //BuildForgeLegacyList();
        }

        #region Forge - Legacy

        /// <summary>
        ///     Retrieves version information from the minecraftforge.net servers for the 1.6.4- versions of Forge (legacy builds)
        /// </summary>
        /// <returns>Modloader dictionary of legacy builds</returns>
        /// =======================================================================================================================
        private void BuildForgeLegacyList()
        {
            try
            {
                using (var client = new WebClientWithTimeout(Constants.DownloadTimeoutPeriod, Constants.HttpUserAgent))
                {
                    client.Headers["User-Agent"] = Constants.HttpUserAgent;

                    // Retrieve legacy version JSON from minecraftforge.net
                    Logger.Info("Downloading Forge Legacy List");
                    var json = client.DownloadString(new Uri(Constants.ForgeLegacyVersionListUrl));
                    ProcessForgeLegacyList(JsonConvert.DeserializeObject<LegacyForgeListJson>(json));
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to download Forge Legacy List.");
            }
        }

        private void ProcessForgeLegacyList(LegacyForgeListJson forgeList)
        {
            try
            {
                // get distinct list of build files. Prefer installer to universal.
                var builds = (from build in forgeList.Builds
                    from file in build.Files
                    where (file.BuildType == "installer" || file.BuildType == "client" || file.BuildType == "universal")
                    orderby file.BuildNumber descending, file.BuildType
                    select file).DistinctBy(f => f.BuildNumber).ToArray();

                var promos = CreateLegacyPromos(builds);

                // Get once to cache locally to not hit DB as much                
                var gameVersionType = GetForgeGameVersionType();

                var total = builds.Length;
                var counter = 0;                                

                // Build Modloader list
                foreach (var build in builds)
                {
                    try
                    {
                        Logger.Debug("Processing legacy forge build " + (++counter) + " of " + total + "...");

                        var forgeInfo = new ForgeInfo(build.ForgeVersion);

                        MinecraftGameVersion minecraftGameVersion;
                        if (!_minecraftGameVersions.TryGetValue(build.MinecraftVersion, out minecraftGameVersion))
                        {
                            if (!_unknownMinecraftVersions.Contains(build.MinecraftVersion))
                            {
                                Logger.Warn(string.Format("Forge version {0} references unknown minecraft version {1}, skipping!", forgeInfo.VersionName, build.MinecraftVersion));
                            }

                            _unknownMinecraftVersions.Add(build.MinecraftVersion);
                            continue;
                        }                          

                        var modLoader = MinecraftModLoaderVersionExtensions.GetByName(forgeInfo.ModLoaderName);
                        if (modLoader == null || MinecraftJobConfiguration.Instance.RebuildMode)
                        {
                            var download = new ForgeDownloadInfo(build.DownloadUrl);
                            var lastModified = SqlDateTime.MinValue.Value;
                            var type = build.BuildType == "installer" ? ForgeFileProcessor.ForgePackageType.Installer : ForgeFileProcessor.ForgePackageType.Client;
                            var forgeDetails = new ForgeFileProcessor(forgeInfo.ModLoaderName, build.MinecraftVersion).Process(type, download);

                            
                            if (forgeDetails == null)
                            {
                                Logger.Warn(string.Format("Unable to process {0}. It will not be persisted.", forgeInfo.ModLoaderName));
                                continue;
                            }

                            PersistForgeVersion(forgeInfo, download, lastModified, promos[build.MinecraftVersion], forgeDetails, minecraftGameVersion, gameVersionType, modLoader != null ? modLoader.ID : 0);
                        }
                    }
                    catch (Exception ex)
                    {
                        Logger.Warn(ex, "Failed to process forge version: " + build.ForgeVersion);
                    }
                   
                }
            }

            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process Forge Legacy List");
            }
        }

        private Dictionary<string,ForgePromo> CreateLegacyPromos(LegacyForgeFileJson[] builds)
        {
            var latestBuilds = builds.DistinctBy(x => x.MinecraftVersion).ToArray();
            var promos = latestBuilds.ToDictionary(
                x => x.MinecraftVersion,
                x => new ForgePromo
                {
                    LatestForgeVersion = x.ForgeVersion,
                    RecommendedForgeVersion = x.ForgeVersion
                });

            // Don't overwrite newer latest/recommended
            foreach (var kvp in promos)
            {
                var info = new MinecraftGameVersionInfo(kvp.Key);
                var latest = MinecraftModLoaderVersionExtensions.GetLatestByMinecraftVersion(info.VersionSlug);
                if (latest != null && CompareVersions(kvp.Value.LatestForgeVersion, latest.ForgeVersion) < 0)
                {
                    kvp.Value.LatestForgeVersion = latest.ForgeVersion;
                }
                var recommended = MinecraftModLoaderVersionExtensions.GetRecommendedByMinecraftVersion(info.VersionSlug);
                if (recommended != null && CompareVersions(kvp.Value.LatestForgeVersion, recommended.ForgeVersion) < 0)
                {
                    kvp.Value.LatestForgeVersion = recommended.ForgeVersion;
                }
            }

            return promos;
        }

        private int CompareVersions(string version1, string version2)
        {
            var version1Split = version1.Split('.');
            var version2Split = version2.Split('.');
            for (int i = 0; i < (version1Split.Length < version2Split.Length ? version1Split.Length : version2Split.Length); i++)
            {
                var v1Part = Int32.Parse(version1Split[i]);
                var v2Part = Int32.Parse(version2Split[i]);

                if (v1Part < v2Part)
                {
                    return -1;
                }
                if (v1Part > v2Part)
                {
                    return 1;
                }
            }
            return 0;
        }

        #endregion

        #region Forge - Gradle

        /// <summary>
        ///     Retrieves version information from the minecraftforge.net servers for the 1.6.4+ versions of Forge (Gradle builds)
        /// </summary>
        /// <returns>Modloader dictionary of Gradle builds</returns>
        /// =======================================================================================================================
        private void BuildForgeGradleList()
        {
            try
            {
                // Build array of resuilts
                using (var client = new WebClientWithTimeout(Constants.DownloadTimeoutPeriod, Constants.HttpUserAgent))
                {
                    // Setup client
                    client.Headers["User-Agent"] = Constants.HttpUserAgent;
                    Logger.Info("Downloading Forge Gradle List");

                    // Retrieve version JSON from minecraftforge.net proxy
                    Console.WriteLine("Downloading: " + Constants.ForgeGradleVersionListUrl);
                    var json = client.DownloadString(new Uri(Constants.ForgeGradleVersionListUrl));
                    ProcessForgeGradleList(JsonConvert.DeserializeObject<GradleForgeListJson>(json));
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Could not download Forge Gradle list");
            }
        }
    
        private void ProcessForgeGradleList(GradleForgeListJson forgeList)
        {
            const string artifact = "forge";

            try
            {               
                var promoVersions = GetForgePromoVersions(forgeList);
                var forgeGameVersionType = GetForgeGameVersionType();

                var downloadRoot = forgeList.WebPath;
                var total = forgeList.Builds.Values.Count;
                var counter = 0;
                // Build Modloader list
                foreach (var build in forgeList.Builds.Values)
                {
                    try
                    {
                        ++counter;
                        Logger.Info("Processing forge build " + (counter) + " of " + total + "...");

                        Console.WriteLine();
                        Console.WriteLine("Processing forge build " + (counter) + " of " + total + "...");

                        var forgeInfo = new ForgeInfo(build.ForgeVersion);

                        MinecraftGameVersion minecraftGameVersion;
                        if (!_minecraftGameVersions.TryGetValue(build.MinecraftVersion, out minecraftGameVersion))
                        {
                            if (!_unknownMinecraftVersions.Contains(build.MinecraftVersion))
                            {
                                Logger.Warn(string.Format("Forge version {0} references unknown minecraft version {1}, skipping!", forgeInfo.VersionName, build.MinecraftVersion));
                            }

                            _unknownMinecraftVersions.Add(build.MinecraftVersion);
                            continue;                            
                        }                        

                        var modLoader = MinecraftModLoaderVersionExtensions.GetByName(forgeInfo.ModLoaderName);
                        if (modLoader == null || MinecraftJobConfiguration.Instance.RebuildMode)
                        {
                            var branch = build.Branch;
                            var branchString = (branch != null) ? string.Format("-{0}", branch) : string.Empty;
                            var fullVersion = string.Format("{0}-{1}{2}", build.MinecraftVersion, forgeInfo.VersionName, branchString);

                            string buildType;
                            string fileExtension;
                            DetermineBestForgeFileType(build, out buildType, out fileExtension);

                            if (buildType != string.Empty && fileExtension != string.Empty)
                            {
                                var download = new ForgeDownloadInfo(downloadRoot, artifact, fullVersion, buildType, fileExtension);
                                var type = buildType == "installer" ? ForgeFileProcessor.ForgePackageType.Installer : ForgeFileProcessor.ForgePackageType.Client;
                              
                                var forgeDetails = new ForgeFileProcessor(forgeInfo.ModLoaderName, build.MinecraftVersion).Process(type, download);

                                if (forgeDetails == null)
                                {
                                    Logger.Warn(string.Format("Unable to process {0}. It will not be persisted.", forgeInfo.ModLoaderName));
                                    continue;
                                }

                                var lastModified = new DateTime(build.ModifiedEpoch, DateTimeKind.Utc);

                                PersistForgeVersion(forgeInfo, download, lastModified, promoVersions[build.MinecraftVersion], forgeDetails, minecraftGameVersion, forgeGameVersionType, modLoader != null ? modLoader.ID : 0);
                            }
                        }
                        else
                        {
                            var latest = promoVersions[build.MinecraftVersion].LatestForgeVersion == forgeInfo.VersionName;
                            var recommended = promoVersions[build.MinecraftVersion].RecommendedForgeVersion == forgeInfo.VersionName;
                            modLoader.UpdateLatestAndRecommended(latest, recommended);
                        }
                    }
                    catch (Exception ex)
                    {
                        Logger.Warn(ex, "Failed to process gradle version for forge version: " + build.ForgeVersion);
                    }                    
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error processing Forge Gradle List");
            }
        }


        // Parse the forge json and ensures we have distinct entries for each version, sorted by a priority scheme
        private void DetermineBestForgeFileType(GradleForgeBuildJson build, out string buildType, out string fileExtension)
        {
            try
            {
                var buildTypePrefs = new List<string> { "installer", "universal", "client" };
                var fileExtensionPrefs = new List<string> { "jar", "zip" };

                // Filter file list to types we might care about
                var filtered =
                    build.Files.Where(file => fileExtensionPrefs.Contains(file.FileType))
                        .Where(file => buildTypePrefs.Contains(file.BuildType))
                        .ToList();
                var sorted =
                    filtered.OrderBy(file => fileExtensionPrefs.IndexOf(file.FileType))
                        .ThenBy(file => buildTypePrefs.IndexOf(file.BuildType))
                        .ToList();

                // Take first item out of list sorted by preferred versions
                if (sorted.Any())
                {
                    buildType = sorted.First().BuildType;
                    fileExtension = sorted.First().FileType;
                }

                else
                {
                    buildType = string.Empty;
                    fileExtension = string.Empty;
                }
            }

                // If matched file isn't one of the preferred types, ignore it
            catch
            {
                buildType = string.Empty;
                fileExtension = string.Empty;
            }
        }

        // Parses the recommended-latest promos from the gradle-list. Sets recommended/latest flags 
        // for highest version in group if not specified
        private Dictionary<string,ForgePromo> GetForgePromoVersions(GradleForgeListJson forgeList)
        {
            var versions = forgeList.Builds.Where(b => b.Value.MinecraftVersion != null).DistinctBy(x => x.Value.MinecraftVersion).ToDictionary(x => x.Value.MinecraftVersion, x => new ForgePromo());

            try
            {
                foreach (var promo in forgeList.Promos.Where(p=>p.Key.Contains("recommended")))
                {
                    // Extract recommended versions
                    try
                    {

                        var recommended = forgeList.Builds[promo.Value];
                        versions[recommended.MinecraftVersion].RecommendedForgeVersion = recommended.ForgeVersion;
                    }

                    // Skip failures, perhaps the file format changed on us.
                    catch(Exception ex)
                    {
                        Logger.Warn(ex, "Unexpected forge version...skipping", new {promo});
                    }
                }
            }

            catch (Exception ex)
            {
                Logger.Warn(ex, "Error parsing forge recommended versions -- treating latest as recommended");
            }

            // Default scheme for undefined versions, set recommended to latest if not already set by the 'promo' key
            var maxPerVersion = forgeList.Builds.GroupBy(x => x.Value.MinecraftVersion).Select(g => g.MaxBy(b => b.Value.ForgeVersion).Value);
            //var maxPerVersion = builds.GroupBy(x => x["mcversion"]).Select(g => g.MaxBy(ml => ml["version"]));

            foreach (var latest in maxPerVersion)
            {
                var mcVersion = latest.MinecraftVersion;
                var forgeVersion = latest.ForgeVersion;

                if (mcVersion == null)
                {
                    continue;
                }

                versions[mcVersion].LatestForgeVersion = forgeVersion;

                if (versions[mcVersion].RecommendedForgeVersion == null)
                {
                    versions[mcVersion].RecommendedForgeVersion = forgeVersion;
                }
            }

            return versions;
        }

        #endregion Forge - Gradle

        private void PersistForgeVersion(ForgeInfo forge, ForgeDownloadInfo forgeDownload, DateTime lastModified, ForgePromo forgePromo, ForgeFileDetails forgeDetails, MinecraftGameVersion minecraftGameVersion, GameVersionType gameVersionType, int existingID = 0)
        {
            try
            {
                using (var conn = DatabaseConnectionHelper.GetConnection(DatabaseType.Elerium))
                {
                    using (var transaction = conn.BeginTransaction())
                    {
                        try
                        {
                            Logger.Info(string.Format("New version of Forge found: {0}", forge.VersionName));

                            // Game Version
                            var gameVersions = GameVersionExtensions.GetAllBySlugAndType(conn, transaction, forge.VersionSlug, gameVersionType);
                            if (gameVersions.All(v => v.IsDeleted()) && gameVersions.Any())
                            {
                                // Skip if version is completely deleted
                                return;
                            }

                            var gameVersion = gameVersions.FirstOrDefault(v => !v.IsDeleted());
                            if (gameVersion == null)
                            {
                                gameVersion = new GameVersion()
                                {
                                    GameVersionTypeID = gameVersionType.ID,
                                    Name = forge.VersionName,
                                    Slug = forge.VersionSlug,
                                };
                                gameVersion.SaveToDatabase(conn, transaction);
                            }

                            
                            // Minecraft Mod Loader Version
                            var modLoader = new MinecraftModLoaderVersion(gameVersion, minecraftGameVersion)
                            {
                                ID = existingID,
                                Name = forge.ModLoaderName,
                                Type = ModLoaderType.Forge,
                                DownloadUrl = forgeDownload.DownloadUrl.ToString(),
                                Filename = forgeDownload.FileName,
                                InstallMethod = forgeDetails.InstallMethod,
                                Latest = forgePromo.LatestForgeVersion == forge.VersionName,
                                Recommended = forgePromo.RecommendedForgeVersion == forge.VersionName,
                                LastModified = lastModified < SqlDateTime.MinValue.Value ? SqlDateTime.MinValue.Value : lastModified,
                                MavenVersionString = forgeDetails.MavenString,
                                LibrariesInstallLocation = forgeDetails.LibraryPath,
                                VersionJson = forgeDetails.VersionJson,
                                MinecraftVersion = minecraftGameVersion.VersionString,
                                AdditionalFilesJson = GetAdditionalFiles(minecraftGameVersion.VersionString)
                            };
                            
                            
                           
                            modLoader.SaveToDatabase(conn, transaction);

                            // Mapping
                            var mapping = GameVersionMappingExtensions.Get(gameVersion.ID, minecraftGameVersion.GameVersionID);
                            if (mapping == null)
                            {
                                mapping = new GameVersionMapping
                                {
                                    GameVersionID = gameVersion.ID,
                                    RelatedGameVersionID = minecraftGameVersion.GameVersionID
                                };
                                mapping.SaveToDatabase(conn, transaction);
                            }


                            transaction.Commit();
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex, string.Format("Failed to persist Forge {0}, rolling back", forge.VersionName));
                            transaction.Rollback();
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to connect to Minecraft Database");
            }
        }

        private static readonly Dictionary<string, string> _additionalFilesJson = new Dictionary<string, string>();

        static SniffForgeVersionsJob()
        {
            
            // 1.4
            var additionalFiles = new List<MinecraftModLoaderFile>();
            additionalFiles.Add(new MinecraftModLoaderFile(new ModLoaderUrl("fixes/argo-2.25.jar").FullUrl, @"{instancefolder}\lib\{filename}"));
            additionalFiles.Add(new MinecraftModLoaderFile(new ModLoaderUrl("fixes/asm-all-4.0.jar").FullUrl, @"{instancefolder}\lib\{filename}"));
            additionalFiles.Add(new MinecraftModLoaderFile(new ModLoaderUrl("fixes/bcprov-jdk15on-147.jar").FullUrl, @"{instancefolder}\lib\{filename}"));
            additionalFiles.Add(new MinecraftModLoaderFile(new ModLoaderUrl("fixes/guava-12.0.1.jar").FullUrl, @"{instancefolder}\lib\{filename}"));
            _additionalFilesJson["1.4"] = JsonConvert.SerializeObject(additionalFiles.ToArray());

            // 1.5
            additionalFiles = new List<MinecraftModLoaderFile>();
            additionalFiles.Add(new MinecraftModLoaderFile(new ModLoaderUrl("fixes/argo-small-3.2.jar").FullUrl, @"{instancefolder}\lib\{filename}"));
            additionalFiles.Add(new MinecraftModLoaderFile(new ModLoaderUrl("fixes/asm-all-4.1.jar").FullUrl, @"{instancefolder}\lib\{filename}"));
            additionalFiles.Add(new MinecraftModLoaderFile(new ModLoaderUrl("fixes/bcprov-jdk15on-148.jar").FullUrl, @"{instancefolder}\lib\{filename}"));
            additionalFiles.Add(new MinecraftModLoaderFile(new ModLoaderUrl("fixes/guava-14.0-rc3.jar").FullUrl, @"{instancefolder}\lib\{filename}"));
            additionalFiles.Add(new MinecraftModLoaderFile(new ModLoaderUrl("fixes/scala-library.jar").FullUrl, @"{instancefolder}\lib\{filename}"));
            additionalFiles.Add(new MinecraftModLoaderFile(new ModLoaderUrl("fixes/deobfuscation_data_1.5.2.zip").FullUrl, @"{instancefolder}\lib\{filename}"));
            _additionalFilesJson["1.5"] = JsonConvert.SerializeObject(additionalFiles.ToArray());

        }

        static string GetAdditionalFiles(string minecraftVersion)
        {
            var majorVersion = string.Join(".", minecraftVersion.Split('.').Take(2).ToArray());
            string json;

            if (_additionalFilesJson.TryGetValue(majorVersion, out json))
            {
                return json;
            }

            return null;
        }

        private GameVersionType GetForgeGameVersionType()
        {
            var gameVersionTypes = GameVersionTypeExtensions.GetAllBySlug(Constants.ForgeGameVersionTypeSlug);
            var gameVersionType = gameVersionTypes.FirstOrDefault(t => !t.IsDeleted());

            if (gameVersionType != null)
            {
                return gameVersionType;
            }

            gameVersionType = new GameVersionType
            {
                Name = "Forge",
                Slug = Constants.ForgeGameVersionTypeSlug,
                ValidationGroupName = Constants.ForgeValidationGroup,
            };
            gameVersionType.SaveToDatabase();
            return gameVersionType;
        }
    }
}
