﻿using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Threading;
using Curse.AddOns;
using Curse.Database.Helpers;
using Curse.Jobs;
using Curse.Logging;
using Curse.Minecraft.Jobs.Minecraft;
using Curse.Minecraft.Jobs.Utilities;
using Curse.Minecraft.Models;
using Newtonsoft.Json.Linq;

namespace Curse.Minecraft.Jobs
{
    [Export(typeof(BaseJob))]
    public class SniffMinecraftVersionsJob : BaseJob
    {
        protected override int RunIntervalMinutes
        {
            get { return 360; }
        }

        protected override void DoRun(CancellationToken cancelToken)
        {
            try
            {
                using (var client = new WebClientWithTimeout(Constants.DownloadTimeoutPeriod, Constants.HttpUserAgent))
                {
                    // Retrieve version list from Mojang
                    var versionListString = client.DownloadString(new Uri(Constants.MinecraftVersionListUrl));
                    ProcessVersionListString(versionListString);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Cannot fetch Minecraft versions");
            }
        }

        private void ProcessVersionListString(string versionListString)
        {
            try
            {
                var versions = JObject.Parse(versionListString);
                IList<string> versionList =
                    versions["versions"].Where(t => t["type"].Value<string>() == "release")
                        .Select(m => (string)m.SelectToken("id"))
                        .ToList();

                foreach (var version in versionList)
                {
                    var versionInfo = new MinecraftGameVersionInfo(version);

                    using (var conn = DatabaseConnectionHelper.GetConnection(DatabaseType.Elerium))
                    {
                        using (var transaction = conn.BeginTransaction())
                        {
                            try
                            {
                                // Game Version Type
                                var gameVersionTypes = GameVersionTypeExtensions.GetAllBySlug(conn, transaction, versionInfo.TypeSlug);
                                if (gameVersionTypes.All(t => t.IsDeleted()) && gameVersionTypes.Any())
                                {
                                    // Skipping if this version type is completely deleted
                                    continue;
                                }

                                var gameVersionType = gameVersionTypes.FirstOrDefault(t => !t.IsDeleted());
                                if (gameVersionType == null)
                                {
                                    gameVersionType = new GameVersionType
                                    {
                                        Slug = versionInfo.TypeSlug,
                                        Name = versionInfo.TypeName,
                                        ValidationGroupName = Constants.MinecraftValidationGroup
                                    };
                                    gameVersionType.SaveToDatabase(conn, transaction);
                                }

                                // Game Version
                                var gameVersions = GameVersionExtensions.GetAllBySlugAndType(conn, transaction, versionInfo.VersionSlug, gameVersionType);
                                if (gameVersions.All(t => t.IsDeleted()) && gameVersions.Any())
                                {
                                    // Skipping if this game version is completely deleted
                                    continue;
                                }

                                var gameVersion = gameVersions.FirstOrDefault(t => !t.IsDeleted());
                                if (gameVersion == null)
                                {
                                    gameVersion = new GameVersion()
                                    {
                                        GameVersionTypeID = gameVersionType.ID,
                                        Name = versionInfo.VersionName,
                                        Slug = versionInfo.VersionSlug,
                                    };
                                    gameVersion.SaveToDatabase(conn, transaction);
                                }

                                // Minecraft Game Version
                                if (MinecraftGameVersionExtensions.GetByGameVersion(conn, transaction, gameVersion.ID) == null)
                                {
                                    var minecraftGameVersion = new MinecraftGameVersion(gameVersion)
                                    {
                                        JarDownloadUrl = string.Format(Constants.MinecraftVersionJarMask, version),
                                        JsonDownloadUrl = string.Format(Constants.MinecraftVersionJsonMask, version),
                                    };
                                    minecraftGameVersion.SaveToDatabases(conn, transaction);
                                }

                                transaction.Commit();
                            }
                            catch (Exception ex)
                            {
                                Logger.Error(ex, string.Format("Error while persisting Minecraft Version {0}", version));
                                transaction.Rollback();
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error processing Minecraft version list");
            }
        }
    }
}
