﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using Curse.Jobs.Runner.Configuration;
using Curse.Logging;
using Curse.MurmurHash;
using Curse.Minecraft.Jobs.Forge;

namespace Curse.Minecraft.Jobs.Utilities
{
    public static class ModLoaderHelper
    {
        private static readonly HashSet<string> ExistingFiles = new HashSet<string>(StringComparer.InvariantCultureIgnoreCase);

        public static ModLoaderUrl GetModLoaderUrl(MavenPackageString mavenPackage, string downloadedFilePath = null)
        {
            return GetModLoaderUrl(mavenPackage.PrimaryBaseUrl, mavenPackage.AllBaseUrls, mavenPackage.RelativeUrl, mavenPackage.FileName, downloadedFilePath);
        }

        public static ModLoaderUrl GetModLoaderUrl(Uri fullUri, string downloadedFilePath, bool downloadFile = false)
        {
            // Convert the download URI into a local path
            var localPath = GetPathFromUri(fullUri);
          
            // See if the file is known to already exist
            if (ExistingFiles.Contains(localPath))
            {
                Console.WriteLine("File already exists: " + localPath);
                return new ModLoaderUrl(localPath);
            }

            // Check the path
            var fullPath = new Uri(MinecraftJobConfiguration.Instance.ModLoaderSharePath + "/" + localPath, UriKind.Absolute);

            // Check if it exists
            if (File.Exists(fullPath.LocalPath))
            {
                Console.WriteLine("File already exists: " + localPath);
                ExistingFiles.Add(fullUri.ToString());
                return new ModLoaderUrl(localPath);
            }

            var directoryPath = Path.GetDirectoryName(fullPath.LocalPath);

            if (directoryPath == null)
            {
                throw new Exception("Unable to get a directory path from url: " + fullPath.LocalPath);
            }

            if (!Directory.Exists(directoryPath))
            {
                Directory.CreateDirectory(directoryPath);
            }

            if (downloadFile)
            {
                if (File.Exists(downloadedFilePath))
                {
                    File.Delete(downloadedFilePath);
                }

                using (var client = new WebClientWithTimeout(Constants.DownloadTimeoutPeriod, Constants.HttpUserAgent))
                {
                    Logger.Info(string.Format("Downloading forge version at {0}", fullUri));
                    client.DownloadFile(fullUri, downloadedFilePath);
                    Logger.Info("Downloading completed", fullUri);
                }
            }

            // Ensure it can be opened as a zip
            if (!ZipHelper.IsValidZip(downloadedFilePath))
            {
                throw new Exception("Unable to open zip file");
            }

            File.Copy(downloadedFilePath, fullPath.LocalPath, true);
            return new ModLoaderUrl(localPath);
        }


        public static ModLoaderUrl GetModLoaderUrl(Uri primaryBaseUri, Uri[] downloadBaseUrls, Uri relativeUri, string fileName, string downloadedFilePath = null)
        {

            var primaryUrl = new Uri(primaryBaseUri, relativeUri);

            // Convert the download URI into a local path
            var baseLocalPath = GetPathFromUri(primaryBaseUri);

            // Convert the download URI into a local path
            var localPath = GetPathFromUri(primaryUrl);

            // See if the file is known to already exist
            if (ExistingFiles.Contains(localPath))
            {
                Console.WriteLine("File already exists: " + localPath);
                return new ModLoaderUrl(baseLocalPath, localPath);
            }

            // Check the path
            var fullPath = new Uri(MinecraftJobConfiguration.Instance.ModLoaderSharePath + "/" + localPath, UriKind.Absolute);

            // Check if it exists
            if (File.Exists(fullPath.LocalPath))
            {
                Console.WriteLine("File already exists: " + localPath);
                ExistingFiles.Add(primaryUrl.ToString());
                return new ModLoaderUrl(baseLocalPath, localPath);
            }

            if (downloadedFilePath == null)
            {
                Console.WriteLine("Downloading: " + localPath);

                // If not, try to download it
                downloadedFilePath = Path.GetTempFileName();
                

                var succeeded = false;
                foreach (var baseUrl in downloadBaseUrls)
                {
                    var url = new Uri(baseUrl, relativeUri);

                    Logger.Info("Attempting download from: " + url);

                    using (var client = new WebClientWithTimeout(Constants.DownloadTimeoutPeriod, Constants.HttpUserAgent))
                    {
                        try
                        {
                            client.DownloadFile(url, downloadedFilePath);
                            // Make sure we can extract this
                            ZipHelper.IsValidZip(downloadedFilePath, true);
                            succeeded = true;
                            break;
                        }
                        catch (Exception ex)
                        {
                            Logger.Warn(ex, "Failed to download from: " + url);
                        }
                    }
                }

                if (!succeeded)
                {
                    // Try local fallbacks
                    var fixFilePath = Path.Combine(MinecraftJobConfiguration.Instance.ModLoaderSharePath, "fixes", fileName);

                    if (!File.Exists(fixFilePath))
                    {
                        throw new Exception("Failed to download file: " + primaryUrl);    
                    }

                    Logger.Warn("File could not be downloaded from primary or mirrors, but was found in our fixes folder", fileName);
                    File.Copy(fixFilePath, downloadedFilePath, true);                    
                }
            }

            var directoryPath = Path.GetDirectoryName(fullPath.LocalPath);

            if (directoryPath == null)
            {
                throw new Exception("Unable to get a directory path from url: " + fullPath.LocalPath);
            }

            if (!Directory.Exists(directoryPath))
            {
                Directory.CreateDirectory(directoryPath);
            }

            File.Copy(downloadedFilePath, fullPath.LocalPath, true);
            return new ModLoaderUrl(baseLocalPath, localPath);
        }


        private static string GetPathFromUri(Uri uri)
        {

            // First convert the host to an Int64
            var host = Encoding.Default.GetBytes(uri.Host.ToLowerInvariant());
            var hostHash = MurmurHash2.GetHashCode(host, host.Length, 1);

            // Combine the host hash with the path and query to get a local full path
            return hostHash + uri.LocalPath;
        }       
    }
}
