﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Runtime.Serialization;
using Curse.Database.Helpers;

namespace Curse.Minecraft.Models
{

	public static class MinecraftModLoaderVersionExtensions
	{

        public static MinecraftModLoaderVersion SetFromDataReader(this MinecraftModLoaderVersion model, SqlDataReader reader)
		{
            model.ID = reader.Get<int>("ID");
            model.GameVersionID = reader.Get<int>("GameVersionID");
            model.MinecraftGameVersionID = reader.Get<int>("MinecraftGameVersionID");
            model.Name = reader.Get<string>("Name");
            model.Type = (ModLoaderType)reader.Get<byte>("Type");
            model.DownloadUrl = reader.Get<string>("DownloadUrl");
            model.Filename = reader.Get<string>("Filename");
            model.InstallMethod = (ModLoaderInstallMethod)reader.Get<byte>("InstallMethod");
            model.Latest = reader.Get<bool>("Latest");
            model.Recommended = reader.Get<bool>("Recommended");
            model.Approved = reader.Get<bool>("Approved");
            model.LastModified = reader.Get<DateTime>("LastModified").FixDBTimeToUTC();
            model.MavenVersionString = reader.Get<string>("MavenVersionString");
            model.LibrariesInstallLocation = reader.Get<string>("LibrariesInstallLocation");
            model.VersionJson = reader.Get<string>("VersionJson");
            model.ForgeVersion = reader.Get<string>("ForgeVersion");
            model.MinecraftVersion = reader.Get<string>("VersionString");
            model.AdditionalFilesJson = reader.Get<string>("AdditionalFilesJson");
            return model;
		}

        public static void UpdateLatestAndRecommended(this MinecraftModLoaderVersion model, bool latest, bool recommended)
		{
            if (model.Latest != latest || model.Recommended != recommended)
			{
                model.Latest = latest;
                model.Recommended = recommended;
				using (var conn = DatabaseConnectionHelper.GetConnection(DatabaseType.Elerium))
				{
					using (var transaction = conn.BeginTransaction())
					{
						try
						{
                            SaveToDatabase(model, conn, transaction);
							transaction.Commit();
						}
						catch
						{
							transaction.Rollback();
							throw;
						}
					}
				}
			}
		}

		public static void SaveToDatabase(this MinecraftModLoaderVersion model, SqlConnection connection, SqlTransaction transaction)
		{
			using (var command = connection.CreateCommand())
			{
				command.Transaction = transaction;
                command.Parameters.AddWithValue("@Latest", model.Latest);
                command.Parameters.AddWithValue("@Recommended", model.Recommended);

                command.Parameters.AddWithValue("@GameVersionID", model.GameVersionID);
                command.Parameters.AddWithValue("@MinecraftGameVersionID", model.MinecraftGameVersionID);
                command.Parameters.AddWithValue("@Name", model.Name);
                command.Parameters.AddWithValue("@Type", model.Type);
                command.Parameters.AddWithValue("@DownloadUrl", model.DownloadUrl);
                command.Parameters.AddWithValue("@Filename", model.Filename);
                command.Parameters.AddWithValue("@InstallMethod", (byte)model.InstallMethod);
                command.Parameters.AddWithValue("@Approved", model.Approved);
                command.Parameters.AddWithValue("@LastModified", model.LastModified);
                command.Parameters.AddWithValue("@MavenVersionString", model.MavenVersionString.ValueOrDBNull());
                command.Parameters.AddWithValue("@VersionJson", model.VersionJson.ValueOrDBNull());
                command.Parameters.AddWithValue("@LibrariesInstallLocation", model.LibrariesInstallLocation.ValueOrDBNull());
                command.Parameters.AddWithValue("@AdditionalFilesJson", model.AdditionalFilesJson.ValueOrDBNull());

                if (model.ID == 0)
				{                    
					command.CommandText =
						string.Format(
							"INSERT INTO [MinecraftModLoaderVersion] ({0},{1},{2},{3},{4},{5},{6},{7},{8},{9},{10},{11},{12},{13},{14}) OUTPUT INSERTED.ID VALUES (@{0},@{1},@{2},@{3},@{4},@{5},@{6},@{7},@{8},@{9},@{10},@{11},@{12},@{13},@{14})",
							"GameVersionID", "MinecraftGameVersionID", "Name", "Type", "DownloadUrl", "Filename", "InstallMethod", "Latest", "Recommended", "Approved", "LastModified",
                            "MavenVersionString", "LibrariesInstallLocation", "VersionJson", "AdditionalFilesJson");
                    model.ID = (int)command.ExecuteScalar();
				}
				else
				{
                    command.CommandText = "UPDATE [MinecraftModLoaderVersion] SET Latest=@Latest, Recommended=@Recommended, AdditionalFilesJson = @AdditionalFilesJson WHERE ID=@ID";
                    command.Parameters.AddWithValue("@ID", model.ID);
					command.ExecuteNonQuery();
				}
			}
		}

		private const string GetQuery = @"SELECT modloader.*,modloaderGameVersion.Name as ForgeVersion,minecraftGameVersion.Name as VersionString
										FROM [MinecraftModLoaderVersion] as modloader
										JOIN [GameVersion] AS modloaderGameVersion 
										ON modloader.GameVersionID=modloaderGameVersion.ID
										JOIN [MinecraftGameVersion] mgv
											JOIN [GameVersion] AS minecraftGameVersion 
											ON mgv.GameVersionID=minecraftGameVersion.ID
										ON modloader.MinecraftGameVersionID=mgv.ID";

		public static MinecraftModLoaderVersion[] GetAll()
		{
			var modLoaders = new List<MinecraftModLoaderVersion>();
			using (var conn = DatabaseConnectionHelper.GetConnection(DatabaseType.Elerium))
			{
				using (var command = conn.CreateCommand())
				{
					command.CommandText = GetQuery;

					using (var reader = command.ExecuteReader())
					{
						while (reader.Read())
						{
                            modLoaders.Add(new MinecraftModLoaderVersion().SetFromDataReader(reader));
						}
					}
				}
			}

			return modLoaders.ToArray();
		}

		public static MinecraftModLoaderVersion GetByName(string name)
		{
			using (var conn = DatabaseConnectionHelper.GetConnection(DatabaseType.Elerium))
			{
				using (var command = conn.CreateCommand())
				{
                    command.CommandText = string.Format("{0} {1}", GetQuery, "WHERE modloader.Name=@Name");
					command.Parameters.AddWithValue("@Name", name);
					using (var reader = command.ExecuteReader())
					{
                        return reader.Read() ? new MinecraftModLoaderVersion().SetFromDataReader(reader) : null;
					}
				}
			}
		}

		public static MinecraftModLoaderVersion GetLatestByMinecraftVersion(string minecraftVersionSlug)
		{
			using (var conn = DatabaseConnectionHelper.GetConnection(DatabaseType.Elerium))
			{
				using (var command = conn.CreateCommand())
				{
                    command.CommandText = string.Format("{0} {1}", GetQuery, "WHERE modloader.Latest = 1 AND minecraftGameVersion.Slug=@VersionString");
					command.Parameters.AddWithValue("@VersionString", minecraftVersionSlug);
					using (var reader = command.ExecuteReader())
					{
                        return reader.Read() ? new MinecraftModLoaderVersion().SetFromDataReader(reader) : null;
					}
				}
			}
		}

		public static MinecraftModLoaderVersion GetRecommendedByMinecraftVersion(string minecraftVersionSlug)
		{
			using (var conn = DatabaseConnectionHelper.GetConnection(DatabaseType.Elerium))
			{
				using (var command = conn.CreateCommand())
				{
                    command.CommandText = string.Format("{0} {1}", GetQuery, "WHERE modloader.Recommended = 1 AND minecraftGameVersion.Slug=@VersionString");
					command.Parameters.AddWithValue("@VersionString", minecraftVersionSlug);
					using (var reader = command.ExecuteReader())
					{
                        return reader.Read() ? new MinecraftModLoaderVersion().SetFromDataReader(reader) : null;
					}
				}
			}
		}

		public static MinecraftModLoaderVersion[] GetAllNewAndNormal()
		{
			var versions = new List<MinecraftModLoaderVersion>();
			using (var conn = DatabaseConnectionHelper.GetConnection(DatabaseType.Elerium))
			{
				using (var command = conn.CreateCommand())
				{
					command.CommandText =
						@"WITH ValidVersions AS
						(
						  SELECT gv.ID,gv.Name FROM [GameVersion] as gv
						  JOIN [GameVersionType] as gvt 
						  ON gv.GameVersionTypeID=gvt.ID 
						  WHERE (gv.[Status]=1 OR gv.[Status]=3) AND (gvt.[Status]=1)
						)

						SELECT modloader.*,modloaderGameVersion.Name as ForgeVersion,minecraftGameVersion.Name as VersionString
						FROM [MinecraftModLoaderVersion] as modloader
						JOIN ValidVersions AS modloaderGameVersion ON modloader.GameVersionID=modloaderGameVersion.ID
						JOIN [MinecraftGameVersion] mgv
						  JOIN ValidVersions AS minecraftGameVersion ON mgv.GameVersionID=minecraftGameVersion.ID
						ON modloader.MinecraftGameVersionID=mgv.ID";

					using (var reader = command.ExecuteReader())
					{
						while (reader.Read())
						{
                            versions.Add(new MinecraftModLoaderVersion().SetFromDataReader(reader));
						}
					}
				}
			}
			return versions.ToArray();
		}
	}
}