﻿using System.Collections.Generic;
using System.Data.SqlClient;
using System.Runtime.Serialization;

namespace MinecraftService.Minecraft.Data
{
    [DataContract]
    public class MinecraftGameVersion
    {
        public int ID { get; private set; }
        public int GameVersionID { get; private set; }

        [DataMember]
        public string JarDownloadUrl { get; set; }
        [DataMember]
        public string JsonDownloadUrl { get; set; }
        [DataMember]
        public bool Approved { get; private set; }

        // Requires a join on GameVersion
        [DataMember]
        public string VersionString { get; set; }

        public MinecraftGameVersion(GameVersion version)
        {
            GameVersionID = version.ID;
            VersionString = version.Name;
        }

        private MinecraftGameVersion(SqlDataReader reader)
        {
            ID = reader.GetInt32(0);
            GameVersionID = reader.GetInt32(1);
            JarDownloadUrl = reader.GetString(2);
            JsonDownloadUrl = reader.GetString(3);
            Approved = reader.GetBoolean(4);
        }

        public void SaveToDatabases()
        {
            using (var conn = DatabaseHelper.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.Parameters.AddWithValue("@JarDownloadUrl", JarDownloadUrl);
                    command.Parameters.AddWithValue("@JsonDownloadUrl", JsonDownloadUrl);

                    if (ID == 0)
                    {
                        command.CommandText = "INSERT INTO [MinecraftVersion] (GameVersionID,JarDownloadUrl,JsonDownloadUrl,Approved) OUTPUT INSERTED.ID VALUES (@GameVersionID,@JarDownloadUrl,@JsonDownloadUrl,false)";
                        command.Parameters.AddWithValue("@GameVersionID", GameVersionID);
                        ID = (int)command.ExecuteScalar();
                    }
                    else
                    {
                        command.CommandText = "UPDATE [MinecraftVersion] SET JarDownloadUrl=@JarDownloadUrl, JsonDownloadUrl=@JsonDownloadUrl, WHERE ID = @ID";
                        command.Parameters.AddWithValue("@ID", ID);
                        command.ExecuteNonQuery();
                    }
                }
            }
        }

        public static MinecraftGameVersion[] GetAll()
        {
            var versions = new List<MinecraftGameVersion>();
            using (var conn = DatabaseHelper.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "SELECT m.*,g.Name FROM [MinecraftVersion] as m JOIN [GameVersion] as g ON m.GameVersionID=g.ID";
                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            versions.Add(new MinecraftGameVersion(reader) {VersionString = (string) reader["Name"]});
                        }
                    }
                }
            }

            return versions.ToArray();
        }

        public static MinecraftGameVersion GetByGameVersion(int gameVersionID)
        {
            using (var conn = DatabaseHelper.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.Parameters.AddWithValue("@GameVesionID", gameVersionID);
                    command.CommandText = "SELECT m.*,g.Name FROM [MinecraftVersion] as m JOIN [GameVersion] as g ON m.GameVersionID=g.ID WHERE m.GameVersionID=@GameVersionID";

                    using (var reader = command.ExecuteReader())
                    {
                        return reader.Read()
                            ? new MinecraftGameVersion(reader) { VersionString = (string)reader["Name"] }
                            : null;
                    }
                }
            }
        }
    }
}