﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Web;
using Curse.Logging;
using MinecraftService.Minecraft.Data;
using Newtonsoft.Json.Linq;

namespace MinecraftService.Minecraft.Polling
{
    public class MinecraftPoller
    {
        #region Minecraft version list

        public static void GetMinecraftVersionsList()
        {
            try
            {
                using (var client = new WebClientWithTimeout(MCConstants.DownloadTimeoutPeriod))
                {
                    // Retrive version list from Mojang
                    client.DownloadStringCompleted += GetMinecraftVersionList_DownloadStringCompleted;
                    client.DownloadStringAsync(new Uri(MCConstants.MinecraftVersionListUrl));
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Cannot fetch Minecraft versions");
            }
        }

        private static void GetMinecraftVersionList_DownloadStringCompleted(object sender,
            DownloadStringCompletedEventArgs e)
        {
            try
            {
                if (e.Error != null)
                {
                    throw new TimeoutException("Timeout retrieving Minecraft versions from Mojang.");
                }

                var versions = JObject.Parse(e.Result);
                IList<string> versionList =
                    versions["versions"].Where(t => t["type"].Value<string>() == "release")
                        .Select(m => (string)m.SelectToken("id"))
                        .ToList();

                foreach (var val in versionList)
                {
                    var gameVersion = GameVersion.GetByNameAndType(val, GameVersionType.Minecraft);
                    if (gameVersion == null)
                    {
                        gameVersion = new GameVersion {Name = val};
                        gameVersion.SaveToDatabase();
                    }

                    var mcVersion = MinecraftGameVersion.GetByGameVersion(gameVersion.ID);
                    if (mcVersion == null)
                    {
                        // New Version Detected
                        mcVersion = new MinecraftGameVersion(gameVersion)
                        {
                            JarDownloadUrl = string.Format(MCConstants.MinecraftVersionJarMask,val),
                            JsonDownloadUrl = string.Format(MCConstants.MinecraftVersionJsonMask,val),
                        };
                        mcVersion.SaveToDatabases();
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error parsing Minecraft version list");
            }
        }

        #endregion
    }
}