﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using System.Runtime.Serialization;
using System.ServiceModel;
using System.ServiceModel.Web;
using System.ServiceModel.Activation;
using System.Text;
using System.Configuration;
using System.Web;
using Curse;
using Curse.Extensions;
using Curse.ServiceModels.Authentication;
using Curse.ServiceModels.Services;
using Curse.ServiceModels.Configuration;
using Curse.ServerModService.Caching;
using Curse.ServerModService.Models;
using System.Web.Script.Serialization;
using System.Net;

namespace Curse.ServerModService
{
    [ServiceBehavior(Name = "ServerModService", Namespace = "http://servermodservice-live.curse.us/", AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Multiple, InstanceContextMode = InstanceContextMode.PerSession)]
    [AspNetCompatibilityRequirements(RequirementsMode = AspNetCompatibilityRequirementsMode.Allowed)]
    public class ServerModService : IServerModService
    {
        private static object _startLock = new object();

        static ServerModService()
        {
            lock (_startLock)
            {
                var config = ServiceConfiguration.Instance;

                #region Init Logger

                string logLevel = config.Logging.LogLevel;
                switch (logLevel.ToLower())
                {
                    case "debug":
                        Logger.SetLogLevel = ELogLevel.Debug;
                        break;
                    case "access":
                        Logger.SetLogLevel = ELogLevel.Access;
                        break;
                    case "error":
                        Logger.SetLogLevel = ELogLevel.Error;
                        break;
                    case "info":
                        Logger.SetLogLevel = ELogLevel.Info;
                        break;
                    case "warning":
                        Logger.SetLogLevel = ELogLevel.Warning;
                        break;
                }
                Logger.SetAutoGrowthMegabytes = 50;
                Logger.SetLogPath = config.Logging.LogPath;

                #endregion

                ServerModCache.Instance.Initialize();
                Logger.Log("ServerModCache Initialized....", ELogLevel.Info);
            }
        }

        public List<ServerMod> Projects(string slug)
        {
            //This needs to be refactored. Putting ion quick fix to meet emergency request.
            string apiKey = WebOperationContext.Current.IncomingRequest.Headers["x-api-key"];

            if (!String.IsNullOrEmpty(apiKey))
            {
                AuthenticationStatus status = CheckApiKey(apiKey);
                if (status != AuthenticationStatus.Success)
                {
                    WebOperationContext ctx = WebOperationContext.Current;
                    ctx.OutgoingResponse.StatusCode = System.Net.HttpStatusCode.Forbidden;
                    ctx.OutgoingResponse.SuppressEntityBody = true;
                    throw new HttpException(403, "Invalid API Key");
                }
            }
            //end refactor

            switch (slug)
            {
                case "":
                    {
                        ErrorData errorData = new ErrorData("You submitted an empty query parameter. (For example:?search= should be ?search=world)", 1001);
                        throw new WebFaultException<ErrorData>(errorData, HttpStatusCode.BadRequest);
                    }
                case "all":
                    slug = "";
                    break;
            }
            try
            {
                return ServerModCache.Instance.GetProjectsBySlug(slug);
            }
            catch (ArgumentNullException ane)
            {
                Logger.Log("Error in retreiving Project List! \n" + ane.Message + "\n" + ane.StackTrace, ELogLevel.Error);
                ErrorData errorData = new ErrorData("You did not provide a correct search parameter. (For example, a correct search parameter is:?search=world)", 1002);
                throw new WebFaultException<ErrorData>(errorData, HttpStatusCode.BadRequest);
            }
            catch (HttpException he)
            {
                Logger.Log("Error in retreiving Project List! \n" + he.Message + "\n" + he.StackTrace, ELogLevel.Error);
                ErrorData errorData = new ErrorData("There was an error in processing your api key", 1003);
                throw new WebFaultException<ErrorData>(errorData, HttpStatusCode.Forbidden);
            }
            catch (Exception ex)
            {
                Logger.Log("Error in retreiving Project List! \n" + ex.Message + "\n" + ex.StackTrace, ELogLevel.Error);
                ErrorData errorData = new ErrorData("There was an unknown error processing your request.", 1999);
                throw new WebFaultException<ErrorData>(errorData, HttpStatusCode.BadRequest);
            }
        }

        public List<ServerModFile> Files(string ids)
        {
            //This needs to be refactored. Putting ion quick fix to meet emergency request.
            string apiKey = WebOperationContext.Current.IncomingRequest.Headers["x-api-key"];

            if (!String.IsNullOrEmpty(apiKey))
            {
                AuthenticationStatus status = CheckApiKey(apiKey);
                if (status != AuthenticationStatus.Success)
                {
                    WebOperationContext ctx = WebOperationContext.Current;
                    ctx.OutgoingResponse.StatusCode = System.Net.HttpStatusCode.Forbidden;
                    ctx.OutgoingResponse.SuppressEntityBody = true;
                    throw new HttpException(403, "Invalid API Key");
                }
            }

            //end refactor

            switch (ids)
            {
                case "":
                    {
                        ErrorData errorData = new ErrorData("You submitted an empty query parameter. (For example:?projectids= should be ?projectids=1,2,3)", 2001);
                        throw new WebFaultException<ErrorData>(errorData, HttpStatusCode.BadRequest);
                    }
            }
            try
            {
                string[] parts = ids.Split(',');
                List<int> listIds = parts.Select(int.Parse).ToList();
                return ServerModFileCache.Instance.GetFilesByIDs(listIds);
            }
            catch (NullReferenceException ane)
            {
                Logger.Log("Error in retreiving File List! \n" + ane.Message + "\n" + ane.StackTrace, ELogLevel.Error);
                ErrorData errorData = new ErrorData("You did not provide a correct search parameter. (for example:?projectids=1,2,3)", 2002);
                throw new WebFaultException<ErrorData>(errorData, HttpStatusCode.BadRequest);
            }
            catch (FormatException fme)
            {
                Logger.Log("Error in retreiving File List! \n" + fme.Message + "\n" + fme.StackTrace, ELogLevel.Error);
                ErrorData errorData = new ErrorData("You did not provide a correct format for the files in the search parameter. (for example, a valid files list is:?projectids=1,2,3)", 2003);
                throw new WebFaultException<ErrorData>(errorData, HttpStatusCode.BadRequest);
            }
            catch (HttpException he)
            {
                Logger.Log("Error in retreiving Project List! \n" + he.Message + "\n" + he.StackTrace, ELogLevel.Error);
                ErrorData errorData = new ErrorData("There was an error in processing your api key", 1003);
                throw new WebFaultException<ErrorData>(errorData, HttpStatusCode.Forbidden);
            }
            catch (Exception ex)
            {
                Logger.Log("Error in retreiving File List! \n" + ex.Message + "\n" + ex.StackTrace, ELogLevel.Error);
                ErrorData errorData = new ErrorData("There was an error processing your request", 2999);
                throw new WebFaultException<ErrorData>(errorData, HttpStatusCode.BadRequest);
            }
        }

        public AuthenticationStatus CheckApiKey(string apiKey)
        {

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["Radon"].ConnectionString))
            {
                SqlCommand cmd;

                cmd = new SqlCommand("SELECT ApiKey, IsBanned FROM ServerModsApiKey WHERE apikey = @apikey", conn);
                cmd.Parameters.AddWithValue("@apikey", apiKey);

                try
                {
                    conn.Open();
                    using (var reader = cmd.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            var userKey = reader.GetString(reader.GetOrdinal("ApiKey"));
                            var isBanned = reader.GetBoolean(reader.GetOrdinal("IsBanned"));

                            if (isBanned || apiKey != userKey)
                            {
                                return AuthenticationStatus.InvalidApiKey;
                            }

                            return AuthenticationStatus.Success;
                        }
                        else
                        {
                            return AuthenticationStatus.InvalidApiKey;
                        }
                    }
                }
                catch (Exception exc)
                {
                    Logger.Log("Failed to authenticate ApiKey. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
                    return AuthenticationStatus.UnknownError;
                }
            }
        }

        public string HealthCheck()
        {
            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["Radon"].ConnectionString))
            {
                conn.Open();
            }

            return "Success";
        }
    }
}
