﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Globalization;
using System.Data.SqlClient;
using System.Configuration;
using System.Net.Mail;
using System.Net;
using System.Threading;
using System.IO;
using System.Reflection;
using System.Web;

namespace CurseReportingService
{
    public static class ReportingEngine
    {
        private static AuthReporter _authReporter = null;        
        private static SubscriptionsReporter _subscriptionsReporter = null;
        private static LegacySubscriptionsReporter _legacySubscriptionsReporter = null;        
        private static V4ClientReporter _v4ClientReporter = null;
        
        private static string _connectionString = null;
        private static string _reportTemplate = null;

        private static bool DEBUG_MODE = false;
        private static string[] DEBUG_EMAILS = new[] { "mcomperda@curse.com" };

        public static void Initialize()
        {
#if DEBUG
            DEBUG_MODE = true;
#endif
            _authReporter = new AuthReporter();
            _subscriptionsReporter = new SubscriptionsReporter();            
            _v4ClientReporter = new V4ClientReporter();
            
            _connectionString = ConfigurationManager.ConnectionStrings["Reporting"].ConnectionString;
            _legacySubscriptionsReporter = new LegacySubscriptionsReporter();
            
            string templateLocation = Path.Combine(Directory.GetCurrentDirectory(), "ReportTemplate.html");
            
            using (StreamReader reader = new StreamReader(templateLocation))
            {
                _reportTemplate = reader.ReadToEnd();
            }
        }


        public static void DoDailyReports()
        {
                                   
            DateTime today = new DateTime(DateTime.UtcNow.Year, DateTime.UtcNow.Month, DateTime.UtcNow.Day, 9, 0, 0, DateTimeKind.Local);
            DateTime yesterday = today.AddDays(-1);
            TryPeriodicReport(EReportType.Daily, EReporterType.SubscriptionReports, yesterday, today);
            TryPeriodicReport(EReportType.Daily, EReporterType.ClientMetricsReports, yesterday, today);
            
        }

        public static void DoWeeklyReports()
        {

            DateTime today = new DateTime(DateTime.UtcNow.Year, DateTime.UtcNow.Month, DateTime.UtcNow.Day, 0, 0, 0, DateTimeKind.Utc);
            DateTime lastWeek = today.AddDays(-7);
            TryPeriodicReport(EReportType.Weekly, EReporterType.SubscriptionReports, lastWeek, today);
            TryPeriodicReport(EReportType.Weekly, EReporterType.ClientMetricsReports, lastWeek, today);
        }

        public static void DoMonthlyReports()
        {

            // Get new user registerations, for the day:            
            DateTime theFirst = new DateTime(DateTime.UtcNow.Year, DateTime.UtcNow.Month, 1, 0, 0, 0, DateTimeKind.Utc);            
            DateTime lastMonth = theFirst.AddMonths(-1);            
            theFirst = theFirst.AddSeconds(-1);
            TryPeriodicReport(EReportType.Monthly, EReporterType.SubscriptionReports, lastMonth, theFirst);
            TryPeriodicReport(EReportType.Monthly, EReporterType.ClientMetricsReports, lastMonth, theFirst);
        }

        public static void DoHourlyReports()
        {
            DateTime now = new DateTime(DateTime.Now.Year, DateTime.Now.Month, DateTime.Now.Day, DateTime.Now.Hour, 0, 0, DateTimeKind.Local);;
            DateTime oneHourAgo = now.AddHours(-1);
            TryPeriodicReport(EReportType.Hourly, EReporterType.SubscriptionReports, oneHourAgo, now);
        }

        public static bool TryPeriodicReport(EReportType pReportType, EReporterType pReporterType, DateTime dateFrom, DateTime dateTo)
        {
            try
            {
                return DoPeriodicReport(pReportType, pReporterType, dateFrom, dateTo);
            }                
            catch (SqlException ex)
            {
                if (ex.Message.Contains("Timeout"))
                {
                    Console.WriteLine("");
                    Console.WriteLine("SQL Timeout, retrying report...");

                    return TryPeriodicReport(pReportType, pReporterType, dateFrom, dateTo);
                }
                else
                {
                    Console.WriteLine("");
                    Console.WriteLine("Unable to create report: " + ex.Message);
                    Console.WriteLine(ex.StackTrace);
                    return false;
                }
            }            
        }

        public static bool DoPeriodicReport(EReportType pReportType, EReporterType pReporterType, DateTime dateFrom, DateTime dateTo)
        {
            DateTime past = DateTime.Parse("1/1/1970");
            DateTime future = DateTime.UtcNow.AddDays(1);

            if (pReporterType == EReporterType.SubscriptionReports)
            {

                string reportTypeLabel = Enum.GetName(typeof(EReportType), pReportType);
                string reportBody = "<h1>Curse " + reportTypeLabel + " Report</h1>";
                string timezoneLabel = dateFrom.Kind == DateTimeKind.Utc ? "UTC" : TimeZone.CurrentTimeZone.IsDaylightSavingTime(dateTo) ? TimeZone.CurrentTimeZone.DaylightName : TimeZone.CurrentTimeZone.StandardName;
                reportBody += "<div>" + dateFrom.ToString() + " to " + dateTo.ToString() + " (" + timezoneLabel + ")</div>";

                // Convert to Universal Time for all querying
                if (dateFrom.Kind == DateTimeKind.Local)
                {
                    dateFrom = dateFrom.ToUniversalTime();
                    dateTo = dateTo.ToUniversalTime();
                }

                #region User Registrations

                Console.WriteLine("User Registrations...");
                int newUserRegistrations = _authReporter.GetNewUserCount(dateFrom, dateTo);
                int totalUserRegistrations = _authReporter.GetNewUserCount(past, future);
                Dictionary<string, int> newUserRegistrationsBySite = _authReporter.GetNewUserCountWithSite(dateFrom, dateTo);
                Dictionary<string, int> totalUserRegistrationsBySite = _authReporter.GetNewUserCountWithSite(past, future);

                reportBody += "<h2>User Registratons</h2>";

                reportBody += "<h3>New Registrations: " + newUserRegistrations.ToString("#,##0") + "</h3>";
                reportBody += GetTableHtml();
                foreach (KeyValuePair<string, int> kvp in newUserRegistrationsBySite)
                {
                    reportBody += "<tr>" + GetLabelCellHtml() + kvp.Key + "</td>" + GetValueCellHtml() + kvp.Value.ToString("#,##0") + "</td></tr>";
                }
                reportBody += "</table>";


                reportBody += "<h3><b>Total Registrations: " + totalUserRegistrations.ToString("#,##0") + "</h3>";
                reportBody += GetTableHtml();
                foreach (KeyValuePair<string, int> kvp in totalUserRegistrationsBySite)
                {
                    reportBody += "<tr>" + GetLabelCellHtml() + kvp.Key + "</td>" + GetValueCellHtml() + kvp.Value.ToString("#,##0") + "</td></tr>";
                }
                reportBody += "</table>";

                #endregion

                # region Subscriptions

                {
                    Console.WriteLine("Curse Premium Subscriptions...");


                    // Sales Numbers
                    Dictionary<string, int> newSubscriptionsByType = _subscriptionsReporter.GetTransactionCountByProduct(dateFrom, dateTo, new[] { "CP" });
                    int newSubscriptions = newSubscriptionsByType.Sum(p => p.Value);
                    
                    // Expirations                    
                    Dictionary<string, int> expirationsByPlan = _subscriptionsReporter.GetExpiredSubscriptions(dateFrom, dateTo, new[] { "CP" });
                    int totalExpirations = expirationsByPlan.Sum(p => p.Value);
                    
                    // Coupons                    
                    
                    Dictionary<string, int> newCouponRedemptionsByType = _subscriptionsReporter.GetSubscriptionCountWithType(dateFrom, dateTo, false, new[] { "CP" }, true, ESubscriptionAction.New);
                    int newCouponRedemptions = newCouponRedemptionsByType.Sum(p => p.Value);
                    int totalActiveCouponRedemptions = _subscriptionsReporter.GetSubscriptionCount(past, future, true, new[] { "CP" }, true, ESubscriptionAction.New);
                    int totalCouponRedemptions = _subscriptionsReporter.GetSubscriptionCount(past, future, false, new[] { "CP" }, true, ESubscriptionAction.New);
                    
                    // Historical Total
                    int totalSubscriptions = _subscriptionsReporter.GetSubscriptionCount(past, future, false, new[] { "CP" }, false, ESubscriptionAction.New);
                    Dictionary<string, int> totalSubscriptionsByType = _subscriptionsReporter.GetSubscriptionCountWithType(past, future, false, new[] { "CP" }, false, ESubscriptionAction.New);
                    Dictionary<string, int> totalCouponRedemptionsByType = _subscriptionsReporter.GetSubscriptionCountWithType(past, future, false, new[] { "CP" }, true, ESubscriptionAction.New);                                        

                    // Active
                    var activeSubscriptionsByType = _subscriptionsReporter.GetActiveSubscriptions(past, future, new[] { "CP" }, new[] { "CP" });
                    var activeSubscriptions = activeSubscriptionsByType.Sum(p => p.Value);
                    
                    Dictionary<string, int> couponsByPurpose = _subscriptionsReporter.GetCouponUsageByReasonAndDateRange(dateFrom, dateTo);

                    reportBody += "<h2>Premium Subscriptions</h2>";
                    reportBody += "<h3>Total New Paid Subs & Redeemed Coupons: " + (newSubscriptions + newCouponRedemptions).ToString("#,##0") + "</h3>";
                    reportBody += "<h3>Net Gain in Subscriptions: " + (newSubscriptions - totalExpirations).ToString("#,##0") + "</h3>";
                    reportBody += "<span>Net Gain equals (New subs - Expired subs), as renewals have no net affect on current total.</span>";
                    reportBody += "<h3>New Paid Subscriptions: " + newSubscriptions.ToString("#,##0") + "</h3>";
                    reportBody += "<p>The total number of new subscriptions purchased during this time interval. It does not include subscription renewals, or plan changes.</p>";
                    reportBody += GetTableHtml();
                    foreach (KeyValuePair<string, int> kvp in newSubscriptionsByType.OrderByDescending(p => p.Value))
                    {
                        reportBody += "<tr>" + GetLabelCellHtml() + kvp.Key + "</td>" + GetValueCellHtml() + kvp.Value.ToString("#,##0") + "</td></tr>";
                    }
                    reportBody += "</table>";

                    reportBody += "<h3>Expired Subscriptions: " + totalExpirations.ToString("#,##0") + "</h3>";
                    reportBody += "<p>The total number of subscriptions that were active and in good standing prior to this time interval, and have since expired.</p>";
                    reportBody += GetTableHtml();
                    foreach (KeyValuePair<string, int> kvp in expirationsByPlan.OrderByDescending(p => p.Value))
                    {
                        reportBody += "<tr>" + GetLabelCellHtml() + kvp.Key + "</td>" + GetValueCellHtml() + kvp.Value.ToString("#,##0") + "</td></tr>";
                    }
                    reportBody += "</table>";

                    if (newCouponRedemptions > 0)
                    {
                        reportBody += "<h3>New Coupon Redemptions: " + newCouponRedemptions.ToString("#,##0") + "</h3>";

                        reportBody += GetTableHtml();
                        foreach (KeyValuePair<string, int> kvp in newCouponRedemptionsByType.OrderByDescending(p => p.Value))
                        {
                            reportBody += "<tr>" + GetLabelCellHtml() + kvp.Key + "</td>" + GetValueCellHtml() + kvp.Value.ToString("#,##0") + "</td></tr>";
                        }

                        if (couponsByPurpose.Count > 0)
                        {
                            reportBody += "<h4>Categories of Redeemed Coupons</h4>";
                            foreach (KeyValuePair<string, int> kvp in couponsByPurpose.OrderByDescending(p => p.Value))
                            {
                                reportBody += "<tr>" + GetLabelCellHtml() + kvp.Key + "</td>" + GetValueCellHtml() + kvp.Value.ToString("#,##0") + "</td></tr>";
                            }
                        }
                    }

                    reportBody += "</table>";
                    
                    reportBody += "<h3>Active Subscriptions: " + activeSubscriptions.ToString("#,##0") + "</h3>";
                    reportBody += "<p>The total number of subscriptions that are active and in good standing at the time of this report (excluding legacy customers).</p>";
                    reportBody += GetTableHtml();
                    foreach (KeyValuePair<string, int> kvp in activeSubscriptionsByType.OrderByDescending(p => p.Value))
                    {
                        reportBody += "<tr>" + GetLabelCellHtml() + kvp.Key + "</td>" + GetValueCellHtml() + kvp.Value.ToString("#,##0") + "</td></tr>";
                    }
                    reportBody += "</table>";
                   
                    // Legacy Subscription
                    int activeLegacySubscriptions = _legacySubscriptionsReporter.GetSubscriptionCount(past, future, true, false, ESubscriptionAction.Either);
                    int activeLegacyPayedSubscriptions = _legacySubscriptionsReporter.GetSubscriptionCount(past, future, true, true, ESubscriptionAction.Either);                    
                    int compedSubs = activeLegacySubscriptions - activeLegacyPayedSubscriptions;

                    reportBody += "<h3>Active Comped Subscriptions: " + compedSubs.ToString("#,##0") + "</h3>";
                    reportBody += "<p>The total number of active subscriptions that have been comped to users (primarily authors).</p>";

                    reportBody += "<h3>Active Legacy Subscriptions: " + activeLegacyPayedSubscriptions.ToString("#,##0") + "</h3>";
                    reportBody += "<p>The total number of active subscriptions that are still on the old PayPal-based billing system.</p>";

                    // Total Legacy and Currency
                    reportBody += "<h3>Total Active Paid Subscriptions: " + (activeLegacyPayedSubscriptions + activeSubscriptions).ToString("#,##0") + "</h3>";
                    reportBody += "<p>The grand total of all paid active subscriptions.</p>";
                }

                #endregion

                #region v4 Client Users (Windows)

                Console.WriteLine("Curse Client - Windows...");
                reportBody += GetV4ClientUsageStatistics(false, pReportType, pReporterType, dateFrom, dateTo, reportTypeLabel, EPlatform.Windows, 10, 1.5, 0.9);
                
                #endregion

                //#region v4 Client Users (OSX)

                //Console.WriteLine("Curse Client - Mac...");
                //reportBody += GetV4ClientUsageStatistics(false, pReportType, pReporterType, dateFrom, dateTo, reportTypeLabel, EPlatform.Mac, 1, 1, 1);

                //#endregion

                #region Client Metrics

                Console.WriteLine("Curse Client Metrics...");
                reportBody += GetClientMetrics(false, pReportType, pReporterType, dateFrom, dateTo, reportTypeLabel);

                #endregion

                SaveReport(pReportType, reportBody, dateFrom, dateTo);

                reportBody = string.Format(_reportTemplate, reportBody);

                EmailReport(pReportType, pReporterType, reportBody, dateFrom, dateTo);

                
            }
            else if (pReporterType == EReporterType.ClientMetricsReports)
            {
                string reportTypeLabel = Enum.GetName(typeof(EReportType), pReportType);
                string reportBody = "<h1>Curse " + reportTypeLabel + " Report</h1>";
                string timezoneLabel = dateFrom.Kind == DateTimeKind.Utc ? "UTC" : TimeZone.CurrentTimeZone.IsDaylightSavingTime(dateTo) ? TimeZone.CurrentTimeZone.DaylightName : TimeZone.CurrentTimeZone.StandardName;
                reportBody += "<div>" + dateFrom.ToString() + " to " + dateTo.ToString() + " (" + timezoneLabel + ")</div>";

                // Convert to Universal Time for all querying
                if (dateFrom.Kind == DateTimeKind.Local)
                {
                    dateFrom = dateFrom.ToUniversalTime();
                    dateTo = dateTo.ToUniversalTime();
                }

                #region v4 Client Users (Windows)

                Console.WriteLine("Curse Client - Windows...");
                reportBody += GetV4ClientUsageStatistics(true, pReportType, pReporterType, dateFrom, dateTo, reportTypeLabel, EPlatform.Windows, 1, 1, 1);

                #endregion

                #region v4 Client Users (OSX)

                Console.WriteLine("Curse Client - Mac...");
                reportBody += GetV4ClientUsageStatistics(true, pReportType, pReporterType, dateFrom, dateTo, reportTypeLabel, EPlatform.Mac, 1, 1, 1);

                #endregion

                #region Client Metrics

                Console.WriteLine("Curse Client Metrics...");
                reportBody += GetClientMetrics(true, pReportType, pReporterType, dateFrom, dateTo, reportTypeLabel);

                #endregion

                reportBody = string.Format(_reportTemplate, reportBody);

                EmailReport(pReportType, pReporterType, reportBody, dateFrom, dateTo);

            }
            
        
            return true;

        }      

        private static string GetV4ClientUsageStatistics(bool activeOnly, EReportType reportType, EReporterType reporterType,
            DateTime dateFrom,
            DateTime dateTo,
            string reportTypeLabel,
            EPlatform platform,
            int smallSampleThreadhold, double recentFactor, double monthlyFactor)
        {
            
            string reportBody = "<h2>Curse Client (" + platform.ToString() + ")</h2>";

            int newClientUsers = _v4ClientReporter.GetNewUserCount(dateFrom, dateTo, platform);
            int recentClientUsers = _v4ClientReporter.GetRecentUserCount(dateFrom, platform);
            int monthlyClientUsers = _v4ClientReporter.GetTotalUserCount(true, platform);
            int totalClientUsers = _v4ClientReporter.GetTotalUserCount(false, platform);
            int activeClientUsers = _v4ClientReporter.GetActiveUserCount(dateFrom, dateTo, platform);

            
            
            if (reportType != EReportType.Monthly)
            {
                recentClientUsers = (int)(recentClientUsers * recentFactor);

                if (recentClientUsers > monthlyClientUsers)
                {
                    recentClientUsers = (int)(monthlyClientUsers * monthlyFactor);
                }                                
                reportBody += "<h3>" + reportTypeLabel + " New Client Users: " + newClientUsers.ToString("#,##0") + "</h3>";
            }

          
                reportBody += "<h3>" + reportTypeLabel + " Active Client Users: " + recentClientUsers.ToString("#,##0") + "</h3>";
                if (reportType != EReportType.Monthly)
                {
                    reportBody += "<h3>Monthly Active Client Users: " + monthlyClientUsers.ToString("#,##0") + "</h3>";
                }
                reportBody += "<h3>Historical Total Client Users: " + totalClientUsers.ToString("#,##0") + "</h3>";
          

            //Windows Usage by Client Verson
            Dictionary<string, int> usageByClientVersion = _v4ClientReporter.GetUsageByClientVersion(activeOnly, dateFrom, dateTo, platform, smallSampleThreadhold);
            reportBody += "<h3>Usage by Curse Client Build</h3>";
            reportBody += GetTableHtml();
            foreach (KeyValuePair<string, int> kvp in usageByClientVersion)
            {
                reportBody += "<tr>" + GetLabelCellHtml() + kvp.Key.Replace(".", "&middot;") + "</td>" + GetValueCellHtml() + kvp.Value.ToString("#,##0") + "</td></tr>";
            }
            reportBody += "</table>";

            //Windows Usage by OS Verson
            Dictionary<string, int> usageByOSVersion = _v4ClientReporter.GetUsageByOSVersion(activeOnly, dateFrom, dateTo, platform, smallSampleThreadhold);
            reportBody += "<h3>Usage by Operating System Version</h3>";
            reportBody += GetTableHtml();
            foreach (KeyValuePair<string, int> kvp in usageByOSVersion)
            {
                reportBody += "<tr>" + GetLabelCellHtml() + kvp.Key.Replace(".", "&middot;") + "</td>" + GetValueCellHtml() + kvp.Value.ToString("#,##0") + "</td></tr>";
            }
            reportBody += "</table>";

            return reportBody;
        }

        public static string GetClientMetrics(bool activeOnly, EReportType reportType, EReporterType reporterType,
            DateTime dateFrom,
            DateTime dateTo,
            string reportTypeLabel)
        {


            string reportBody = "<h2>Curse Client Metrics</h2>";

            #region Local Saves

            Console.WriteLine("Local Saves Usage...");
            reportBody += "<h3>Total Local Saves Usage</h3>";
            reportBody += GetTableHtml();
            reportBody += "<tr><td>Game</td><td>Users</td><td>#&nbsp;of&nbsp;Saves</td></tr>";
            foreach (SavedGameRecord r in _v4ClientReporter.GetSavedGameCounts())
            {
                reportBody += "<tr>" + GetLabelCellHtml() + r.GameId + "</td>" + GetValueCellHtml() + r.UserCount.ToString("#,##0") + "</td>" + GetValueCellHtml() + r.Saves.ToString("#,##0") + " </td></tr>";
            }
            reportBody += "</table>";

            #endregion

            #region Cloud Storge Info

            Console.WriteLine("Cloud Storage Usage...");
            if (activeOnly)
            {
                reportBody += "<h3> " + reportTypeLabel + " Cloud Storage Usage</h3>";
            }
            else
            {
                reportBody += "<h3> Total Cloud Storage Usage</h3>";
            }
            
            reportBody += GetTableHtml();
            reportBody += "<tr><td>Game</td><td>Users</td><td>Space&nbsp;in&nbsp;MB</td></tr>";
            foreach (CloudUsageRecord r in _v4ClientReporter.GetCloudUsage(activeOnly, dateFrom, dateTo))
            {
                reportBody += "<tr>" + GetLabelCellHtml() + r.GameId + "</td>" + GetValueCellHtml() + r.UserCount.ToString("#,##0") + "</td>" + GetValueCellHtml() + r.SpaceUsed.ToString("#,##0") + " </td></tr>";
            }
            reportBody += "</table>";

            #endregion

            #region Game Stats Info

            if (reporterType != EReporterType.ClientMetricsReports)
            {

                Console.WriteLine("Game Metrics Info...");
                reportBody += "<h3> Total Game Metrics</h3>";
                reportBody += GetTableHtml();
                reportBody += "<tr><td>Game</td><td>Total&nbsp;Installs</td><td>Set&nbsp;as&nbsp;Default</td></tr>";
                foreach (GameStatsRecord r in _v4ClientReporter.GetGameStatsInfo(activeOnly, dateFrom, dateTo))
                {
                    reportBody += "<tr>" + GetLabelCellHtml() + r.GameId + "</td>" + GetValueCellHtml() + r.Installs.ToString("#,##0") + "</td>" + GetValueCellHtml() + r.Defaults.ToString("#,##0") + " </td></tr>";
                }
                reportBody += "</table>";
            }

            #endregion

            #region Game Stats Info

            Console.WriteLine("Game Feature Usage Info...");
            if (activeOnly)
            {
                reportBody += "<h3> " + reportTypeLabel + " Total Game Feature Usage</h3>";
            }
            else
            {
                reportBody += "<h3> Total Game Feature Usage</h3>";    
            }
            
            reportBody += GetTableHtml();
            reportBody += "<tr><td>Game</td><td>Feature</td><td>Uses</td></tr>";
            foreach (GameFeatureUsageRecord r in _v4ClientReporter.GetGameFeatureUsageInfo(activeOnly, dateFrom, dateTo))
            {
                reportBody += "<tr><td>" + r.GameId + "</td><td>" + r.Description + "</td>" + GetValueCellHtml() + r.Uses.ToString("#,##0") + " </td></tr>";
            }
            reportBody += "</table>";

            #endregion

            return reportBody;

        }

        public static void SaveReport(EReportType pReportType, string pReportBody, DateTime pDateFrom, DateTime pDateTo)
        {

            using (SqlConnection conn = Utility.GetDatabaseConnection(_connectionString))
            {
                SqlCommand command = conn.CreateCommand();
                command.CommandText = "delete from Report where Type = @Type and DateFrom = @DateFrom and DateTo = @DateTo";
                command.Parameters.AddWithValue("@Type", pReportType);
                command.Parameters.AddWithValue("@Body", pReportBody);
                command.Parameters.AddWithValue("@DateFrom", pDateFrom);
                command.Parameters.AddWithValue("@DateTo", pDateTo);
                command.ExecuteNonQuery();

                command.CommandText = "insert into Report(Type, Body, DateFrom, DateTo) values(@Type, @Body, @DateFrom, @DateTo);";                
                command.ExecuteNonQuery();
            }

        }

        public static void EmailReport(EReportType pReportType, EReporterType pReporterType, string pReportBody, DateTime dateFrom, DateTime dateTo)
        {

            string reportTypeLabel = Enum.GetName(typeof(EReportType), pReportType);

            MailMessage message = new MailMessage();

            if (DEBUG_MODE)
            {                
                foreach (string address in DEBUG_EMAILS)
                {
                    message.To.Add(new MailAddress(address));
                }
            }
            else
            {
                if (pReporterType == EReporterType.SubscriptionReports)
                {
                    message.To.Add(new MailAddress("report." + reportTypeLabel.ToLower() + "@curse.com"));
                }
                else if (pReporterType == EReporterType.CustomerServiceReports)
                {
                    message.To.Add(new MailAddress("supportreport." + reportTypeLabel.ToLower() + "@curse.com"));
                }
                else if (pReporterType == EReporterType.ClientMetricsReports)
                {
                    message.To.Add(new MailAddress("cmreport." + reportTypeLabel.ToLower() + "@curse.com"));
                }
            }

            // Convert Hourly and Daily Reports to Local Time
            if ((pReportType == EReportType.Hourly || pReportType == EReportType.Daily) && dateFrom.Kind == DateTimeKind.Utc)
            {
                dateFrom = dateFrom.ToLocalTime();
                dateTo = dateTo.ToLocalTime();
            }

            string subject = String.Empty;


            if (pReporterType == EReporterType.SubscriptionReports)
            {
                message.From = new MailAddress("reporting@curse.com", "Curse Reporting");
                subject = "Curse " + reportTypeLabel + " Report";                      
                switch (pReportType)
                {
                    case EReportType.Daily:
                        subject += " (" + dateTo.ToString("MMMM d, yyyy") + ")";
                        break;
                    case EReportType.Hourly:
                        subject += " (" + dateTo.ToString("MMMM d, yyyy h:mm tt") + ")";
                        break;
                    case EReportType.Weekly:
                        subject += " (" + dateFrom.ToString("MMMM d, yyyy") + " to " + dateTo.ToString("MMMM d, yyyy") + ")";
                        break;
                    case EReportType.Monthly:
                        subject += " (" + dateFrom.ToString("MMMM yyyy") + ")";
                        break;
                    default:
                        break;
                }
            }
            else if (pReporterType == EReporterType.CustomerServiceReports)
            {
                message.From = new MailAddress("reporting@curse.com", "Curse Support Reporting");
                subject = "Curse " + reportTypeLabel + " Support Report";                      
                switch (pReportType)
                {
                    case EReportType.Daily:
                        subject += " (" + dateTo.ToString("MMMM d, yyyy") + ")";
                        break;
                    case EReportType.Hourly:
                        subject += " (" + dateTo.ToString("MMMM d, yyyy h:mm tt") + ")";
                        break;
                    case EReportType.Weekly:
                        subject += " (" + dateFrom.ToString("MMMM d, yyyy") + " to " + dateTo.ToString("MMMM d, yyyy") + ")";
                        break;
                    case EReportType.Monthly:
                        subject += " (" + dateFrom.ToString("MMMM yyyy") + ")";
                        break;
                    default:
                        break;
                }
            }
            else if (pReporterType == EReporterType.ClientMetricsReports)
            {
                message.From = new MailAddress("reporting@curse.com", "Curse Client Metrics Reporting");
                subject = "Curse Client " + reportTypeLabel + " Metrics Report";
                switch (pReportType)
                {
                    case EReportType.Daily:
                        subject += " (" + dateTo.ToString("MMMM d, yyyy") + ")";
                        break;
                    case EReportType.Hourly:
                        subject += " (" + dateTo.ToString("MMMM d, yyyy h:mm tt") + ")";
                        break;
                    case EReportType.Weekly:
                        subject += " (" + dateFrom.ToString("MMMM d, yyyy") + " to " + dateTo.ToString("MMMM d, yyyy") + ")";
                        break;
                    case EReportType.Monthly:
                        subject += " (" + dateFrom.ToString("MMMM yyyy") + ")";
                        break;
                    default:
                        break;
                }
            }
            else
            {
                throw new Exception("Incorrect or null ReporterType specified.");
            }

            message.Subject = subject;
            message.IsBodyHtml = true;
            message.Body = pReportBody;

            Thread t = new Thread(SendThreadedMail);
            t.Start(message);

        }

        public static void SendThreadedMail(Object o)
        {
            bool success = false;
            while (!success)
            {
                SmtpClient client = new SmtpClient();
                client.Host = DEBUG_MODE ? "smtp.curse.local" : "smtp01-live.curse.us";
                client.DeliveryMethod = SmtpDeliveryMethod.Network;                
                try
                {
                    Console.Write("Sending mail...");
                    client.Send((MailMessage)o);
                    success = true;
                    Console.WriteLine("Success!");
                }
                catch(Exception ex)
                {
                    success = false;
                    Console.WriteLine("Failure: " + ex.Message);
                }
            }

        }

        private static string GetTableHtml()
        {
            return "<table style=\"width: 320px; border: 1px solid #eeeeee; border-width: 0px 1px 1px 1px; border-spacing: 0px;\">";
        }

        private static string GetLabelCellHtml()
        {
            return "<td style=\"border-top: 1px solid #eeeeee;background-color: #F9F9F9;width: 90%;\">";
        }

        private static string GetValueCellHtml()
        {
            return "<td style=\"border-top: 1px solid #eeeeee; text-align: right;min-width: 65px;width: 10%;\">";
        }

    }
}
