﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using MySql.Data.MySqlClient;
using MySql.Data.Types;
using Curse;
using Curse.Extensions;

namespace WowSteadUserImporter
{
    internal static class Program
    {
        private static Random _randomProvider = new Random();
        private const int MAX_RANDOM_VALUE = 999999;                                    

        private static List<User> GetLegacyUsers(int startingID, int limit)
        {
            List<User> users = new List<User>();

            Console.WriteLine("Opening WowStead Database Connection...");

            using (MySqlConnection conn = new MySqlConnection(Settings.Default.SourceDB))
            {
                try
                {
                    conn.Open();
                }
                catch(Exception ex)
                {
                    Console.WriteLine("Unable to open connection: " + ex.Message);
                    Console.ReadKey();
                    return null;
                }

                MySqlCommand cmd = conn.CreateCommand();
                cmd.CommandTimeout = 3000;                
                cmd.CommandText = "SELECT count(0) FROM ws_users where id >= " + startingID;

                if (limit > 0)
                {
                    cmd.CommandText += " limit " + limit + ";";
                }

                long count = (long)cmd.ExecuteScalar();
                
                cmd.CommandText = "SELECT *, hex(username) as hex_username FROM ws_users where id >= " + startingID + " ORDER BY id";

                if (limit > 0)
                {
                    cmd.CommandText += " limit " + limit + ";";
                }

                Console.WriteLine("Caching Legacy Users...");
                int i = 0;

                using (MySqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {

                        ++i;

                        Console.SetCursorPosition(25, 0);
                        Console.Write(i.ToString("###,##") + " of " + count.ToString("###,##"));
                        
                        User u = new User();
                        u.Email = ((string)reader["email"]).ToLower();
                        u.LegacyUserID = (int)reader["id"];

                        string hexUsername = (string)reader["hex_username"];
                        byte[] usernameBytes = hexUsername.ToByteArray();
                        string username = System.Text.Encoding.UTF8.GetString(usernameBytes);

                        u.LegacyUsername = username;
                        u.Salt = (string)reader["salt"];
                        if (u.Salt == string.Empty)
                        {
                            u.Password = (string)reader["old_password"];
                        }
                        else
                        {
                            u.Password = (string)reader["password"];
                        }

                        u.FirstName = (string)reader["username"];
                        u.Country = (string)reader["country"];
                        if (u.FirstName.Length > 32)
                        {
                            u.FirstName = "";
                        }

                        try
                        {
                            u.BirthDate = (DateTime)reader["dob"];
                        }
                        catch (MySqlConversionException) { }

                        if (u.BirthDate.Year < 1900)
                        {
                            u.BirthDate = new DateTime(1900, 1, 1);
                        }

                        u.Gender = 0;
                        u.Newsletter = (bool)reader["mailinglist"];

                        users.Add(u);
                    }
                }
            }            
            return users;
        }

        private static bool ProcessPreviousImport(SqlConnection conn, User user)
        {
            // First see if this user was already imported
            SqlCommand cmd = conn.CreateCommand();
            cmd.CommandType = CommandType.StoredProcedure;
            cmd.CommandText = "spGetLegacyUser";
            cmd.Parameters.AddWithValue("@numLegacyUserID", user.LegacyUserID);
            cmd.Parameters.AddWithValue("@numSiteID", Settings.Default.SiteID);

            using (SqlDataReader reader = cmd.ExecuteReader())
            {
                if (!reader.Read())
                {
                    return false;
                }
                else
                {
                    user.NewUserID = (int)reader["_uid"];
                    user.Status = (EImportStatus)reader["_status"];
                }
            }
            
            // Update their password
            cmd = conn.CreateCommand();
            cmd.CommandText = "update userlegacy set _password = @password, _salt = @salt where _site = @site and _legacyid = @legacyid";
            cmd.Parameters.AddWithValue("@legacyid", user.LegacyUserID);
            cmd.Parameters.AddWithValue("@site", Settings.Default.SiteID);
            cmd.Parameters.AddWithValue("@password", user.Password);
            if (user.Salt.IsNullOrEmpty())
            {
                cmd.Parameters.AddWithValue("@salt", System.DBNull.Value);
            }
            else
            {
                cmd.Parameters.AddWithValue("@salt", user.Salt);
            }
            cmd.ExecuteNonQuery();

            return true;
        }

        private static bool ProcessExistingUser(SqlConnection conn, User user)
        {
            SqlCommand cmd = conn.CreateCommand();

            cmd.CommandType = CommandType.StoredProcedure;
            cmd.CommandText = "spUserIDFromEmail";
            cmd.Parameters.AddWithValue("@strEmail", user.Email);

            object uid = null;

            // If the email is already in useremails, we can consolidate the accounts
            if ((uid = cmd.ExecuteScalar()) != null)
            {                

                cmd.Parameters.Clear();
                cmd.CommandText = "spUsernameFromID";
                cmd.Parameters.AddWithValue("@numUserID", (int)uid);

                user.NewUsername = (string)cmd.ExecuteScalar();
                user.NewUserID = (int)uid;

                if (string.Equals(user.LegacyUsername, user.NewUsername, StringComparison.InvariantCultureIgnoreCase))
                {
                    // Usernames match
                    user.Status = EImportStatus.Merge_Clean;
                }
                else
                {                    
                    // Usernames do not match
                    user.Status = EImportStatus.Merge_UsernameChanged;                    
                }

                CreateLegacyRecord(conn, user, null);

                return true;
            }
            else
            {
                return false;
            }
        }
        
        private static bool ProcessNewUser(SqlConnection conn, User user)
        {


            user.NewUsername = GetValidUsername(conn, user);

            SqlTransaction transaction = conn.BeginTransaction();

            try
            {

                // Update the users table
                SqlCommand cmd = conn.CreateCommand();
                cmd.Transaction = transaction;
                cmd.CommandText = "INSERT INTO users(_username,_password,_level,_salt) OUTPUT Inserted._uid VALUES(@username,NULL,100,NULL);";
                cmd.Parameters.AddWithValue("@username", user.NewUsername);

                user.NewUserID = (int)cmd.ExecuteScalar();

                // Update the useremails table
                cmd = conn.CreateCommand();
                cmd.Transaction = transaction;
                cmd.CommandText = "INSERT INTO useremails(_uid,_email,_verified) OUTPUT Inserted._id  VALUES(@uid,@email,0);";
                cmd.Parameters.AddWithValue("@uid", user.NewUserID);
                cmd.Parameters.AddWithValue("@email", user.Email);
                int emailid = (int)cmd.ExecuteScalar();

                // Update the userprofiles table
                cmd = conn.CreateCommand();
                cmd.Transaction = transaction;
                cmd.CommandText = "INSERT INTO userprofiles("
                        + "_uid,_firstName,_birthDate,_gender,_defaultEmail,_country,_wantsmail,_registeredVia)"
                        + " VALUES("
                        + "@uid, @firstName, @birthDate, @gender, @defaultEmail, @country, @wantsmail, @registeredVia)";

                cmd.Parameters.AddWithValue("@uid", user.NewUserID);
                cmd.Parameters.AddWithValue("@firstName", user.FirstName);
                cmd.Parameters.AddWithValue("@birthDate", user.BirthDate);
                cmd.Parameters.AddWithValue("@gender", user.Gender);
                cmd.Parameters.AddWithValue("@defaultEmail", emailid);
                cmd.Parameters.AddWithValue("@country", user.Country);
                cmd.Parameters.AddWithValue("@wantsmail", user.Newsletter);
                cmd.Parameters.AddWithValue("@registeredVia", Settings.Default.SiteID);                                                       
                cmd.ExecuteNonQuery();


                CreateLegacyRecord(conn, user, transaction);

                transaction.Commit();                

                return true;

            }
            catch (Exception ex)
            {
                transaction.Rollback();

                user.FailureMessage = ex.Message;
                user.Status = EImportStatus.Failed;

                return false;
            }

        }

        private static string GetValidUsername(SqlConnection conn, User user)
        {
            SqlCommand cmd = conn.CreateCommand();
            cmd.CommandType = CommandType.StoredProcedure;
            cmd.CommandText = "spUserIDFromUsername";
            cmd.Parameters.AddWithValue("@strUsername", user.LegacyUsername);

            string newUsername = null;

            // The account could not be consolidated with an existing curse account, so we need to verify the username is valid
            if (InputValidation.IsValidUserName(user.LegacyUsername) && cmd.ExecuteScalar() == null)
            {
                user.Status = EImportStatus.NewAccount_Clean;
                return user.LegacyUsername;
            }

            newUsername = string.Format("user_{0}", _randomProvider.Next(0, MAX_RANDOM_VALUE));
            user.Status = EImportStatus.NewAccount_UsernameChanged;
            cmd.Parameters["@strUsername"].Value = newUsername;

            while (cmd.ExecuteScalar() != null)
            {
                newUsername = string.Format("user_{0}", _randomProvider.Next(0, MAX_RANDOM_VALUE));
                cmd.Parameters["@strUsername"].Value = newUsername;
            }

            return newUsername;
        }
       
        private static void Main(string[] args)
        {

            List<User> users = GetLegacyUsers(0, 0);
            if (users == null)
            {
                return;
            }

            string connString = Settings.Default.DestinationDBLive;
            //string connString = Settings.Default.DestinationDBDev;

            Console.WriteLine("Opening Auth Database Connection...");

            using (SqlConnection conn = new SqlConnection(connString))
            {
                conn.Open();
                                                               
                int count = 0;
                int percent = 0;

                Console.Clear();
                Console.Write("Updating Legacy Users...");

                foreach (User u in users)
                {
                    int newPercent = (int)(((float)(++count) / users.Count) * 100);

                    Console.SetCursorPosition(25, 0);
                    Console.Write(count.ToString("###,##") + " of " + users.Count.ToString("###,##"));

                    if (percent == 0 || newPercent != percent)
                    {
                        percent = newPercent;
                        Console.Title = string.Format("Import Status: {0}%", percent);
                    }

                    // 1. Previously Imported
                    if (ProcessPreviousImport(conn, u))
                    {
                        continue;
                    }
                    

                    // 2. Simple Merge
                    if (ProcessExistingUser(conn, u))
                    {
                        continue;
                    }


                   // 3. New Account
                    if (ProcessNewUser(conn, u))
                    {
                        continue;
                    }
                    
                }                
            }


            Console.SetCursorPosition(0, 1);
            Console.WriteLine("");
            Console.WriteLine("Import Summary");
            Console.WriteLine("----------------------------------------");

            var query = from n in users
                        group n by n.Status into g
                        select new { Status = g.Key, Total = g.Count() };
                        

            foreach (var num in query.OrderByDescending(p => p.Total))
            {
                Console.WriteLine(num.Status + ": " + num.Total.ToString("###,##"));
            }
                       
            Console.ReadKey();
        }

        private static void CreateLegacyRecord(SqlConnection conn, User u, SqlTransaction transaction)
        {
            SqlCommand cmd = conn.CreateCommand();
            if (transaction != null)
            {
                cmd.Transaction = transaction;
            }
            cmd.CommandText = "insert into userlegacy(_uid, _site, _password, _salt, _legacyid, _status, _renamed) values(@uid, @site, @password, @salt, @legacyid, @status, @renamed)";
            cmd.Parameters.AddWithValue("@uid", u.NewUserID);
            cmd.Parameters.AddWithValue("@site", Settings.Default.SiteID);
            cmd.Parameters.AddWithValue("@password", u.Password);
            cmd.Parameters.AddWithValue("@legacyid", u.LegacyUserID);
            cmd.Parameters.AddWithValue("@status", u.Status);

            if (u.Salt.IsNullOrEmpty())
            {
                cmd.Parameters.AddWithValue("@salt", System.DBNull.Value);
            }
            else
            {
                cmd.Parameters.AddWithValue("@salt", u.Salt);
            }
            
            cmd.Parameters.AddWithValue("@renamed", (u.Status == EImportStatus.Merge_UsernameChanged || u.Status == EImportStatus.NewAccount_UsernameChanged));
            cmd.ExecuteNonQuery();

            if (u.Status == EImportStatus.Merge_UsernameChanged || u.Status == EImportStatus.NewAccount_UsernameChanged)
            {
                cmd.CommandText = "insert into userlegacy(_uid) values(@uid)";
                cmd.Parameters.AddWithValue("@uid", u.NewUserID);
                try
                {
                    cmd.ExecuteNonQuery();
                }
                catch { }
            }
        }

        private static String EscapeAndQuote(String pValue)
        {
            return "'" + pValue.Replace("'", "''") + "'";
        }
    }
}
