﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using System.Windows;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Globalization;

using Curse.Extensions;
using Curse.ClientService.Models;

namespace Curse.CurseClient.Common
{
    public static class CHardwareProfileExtension
    {

        private static Regex sCpuInfoRegex = new Regex(@"^(?<CpuString>.*?)(\((?<CpuCount>\d) CPUs\))?(, ~(?<CpuFrequency>.*?)GHz)?$", RegexOptions.Compiled);
        private static Regex sCpuFrequencyRegex = new Regex(@"@\s+(?<CpuFrequency>.*?)GHz", RegexOptions.Compiled);

        private static void setMiscData(CHardwareProfile profile, XmlNode systemInfo)
        {
            // Misc
            profile.ComputerName = Environment.MachineName.ToUpper();

            profile.LanguageName = CultureInfo.InstalledUICulture.Parent.EnglishName;

            profile.IsLaptop = (SystemInformation.PowerStatus.BatteryChargeStatus != BatteryChargeStatus.NoSystemBattery
                && SystemInformation.PowerStatus.BatteryChargeStatus != BatteryChargeStatus.Unknown);


            string dxVersion = systemInfo.SelectSingleNode("DirectXVersion").InnerText;
            if (dxVersion.Contains("("))
            {
                dxVersion = dxVersion.Substring(0, dxVersion.IndexOf("(") - 1);
            }

            // DX Version
            profile.DirectXVersionName = dxVersion;
        }

        private static void setMonitorData(CHardwareProfile profile, XmlNode primaryDisplayDevice)
        {
            profile.MonitorResolutionWidth = (int)SystemParameters.PrimaryScreenWidth;
            profile.MonitorResolutionHeight = (int)SystemParameters.PrimaryScreenHeight;

            XmlNode monitorModel = primaryDisplayDevice.SelectSingleNode("MonitorModel");
            if (monitorModel == null)
            {
                monitorModel = primaryDisplayDevice.SelectSingleNode("Monitor");
            }

            if (monitorModel != null)
            {
                profile.MonitorDescription = monitorModel.InnerText;
            }

            profile.MonitorCount = (byte)SystemInformation.MonitorCount;

        }

        private static void setOSData(CHardwareProfile profile, XmlNode systemInfo)
        {
            profile.OSVersionName = getOsName(systemInfo.SelectSingleNode("OperatingSystem").InnerXml);
            profile.OSIs64Bit = Is64Bit();
            profile.OSServicePackName = Environment.OSVersion.ServicePack;
            profile.OSVersionMajor = Environment.OSVersion.Version.Major;
            profile.OSVersionMajorRevision = Environment.OSVersion.Version.MajorRevision;
            profile.OSVersionMinor = Environment.OSVersion.Version.Minor;
            profile.OSVersionMinorRevision = Environment.OSVersion.Version.MinorRevision;
        }

        private static void setGpuData(CHardwareProfile profile, XmlNode primaryDisplayDevice, byte displayDeviceCount)
        {
            // GPU
            profile.GpuCount = displayDeviceCount;
            profile.GpuDescription = primaryDisplayDevice.SelectSingleNode("CardName").InnerText;
            profile.GpuManufacturerName = primaryDisplayDevice.SelectSingleNode("Manufacturer").InnerText;
            string memory = string.Empty;
            if (primaryDisplayDevice.SelectSingleNode("DedicatedMemory") != null)
            {
                memory = primaryDisplayDevice.SelectSingleNode("DedicatedMemory").InnerText;
            }
            else
            {
                memory = primaryDisplayDevice.SelectSingleNode("DisplayMemory").InnerText;
            }
            memory = memory.Replace("MB", "").Replace("RAM", "").Trim();
            profile.GpuMemorySize = (int)memory.ParseDouble(System.Globalization.CultureInfo.InvariantCulture);
            profile.GpuDriverName = getGpuDriverName(primaryDisplayDevice.SelectSingleNode("DriverName").InnerText);
            profile.GpuDriverVersionName = primaryDisplayDevice.SelectSingleNode("DriverVersion").InnerText;
            string driverDate = primaryDisplayDevice.SelectSingleNode("DriverDate").InnerText;

            if (string.IsNullOrEmpty(driverDate))
            {
                profile.GpuDriverDate = DateTime.MinValue;
            }
            else
            {
                profile.GpuDriverDate = DateTime.Parse(driverDate, System.Globalization.CultureInfo.InvariantCulture);
            }

        }

        private static void setRamData(CHardwareProfile profile, XmlNode systemInfo)
        {
            profile.RamFrequency = 0;
            profile.RamManufacturerName = "";
            profile.RamTypeName = "";
            profile.RamSize = int.Parse(systemInfo.SelectSingleNode("Memory").InnerText.Replace("MB", "").Replace("RAM", "").Trim(), System.Globalization.CultureInfo.InvariantCulture);
        }

        private static void setCpuData(CHardwareProfile profile, XmlNode systemInfo)
        {

            string cpuInfoString = systemInfo.SelectSingleNode("Processor").InnerText;
            Match cpuInfo = sCpuInfoRegex.Match(cpuInfoString);

            if (cpuInfo.Success)
            {
                string cpuString = cpuInfo.Groups["CpuString"].Value;

                // Manufacturer
                profile.CpuManufacturerName = GetCpuManufacturer(cpuString.ToLowerInvariant());

                // Description
                profile.CpuDescription = GetCpuDescription(cpuString);

                // Core Count
                if (cpuInfo.Groups["CpuCount"] != null && !string.IsNullOrEmpty(cpuInfo.Groups["CpuCount"].Value))
                {
                    profile.CpuCount = byte.Parse(cpuInfo.Groups["CpuCount"].Value, System.Globalization.CultureInfo.InvariantCulture);
                }
                else
                {
                    profile.CpuCount = 1;
                }

                // Frequency
                double rawCpuFrequency = 0;
                if (!string.IsNullOrEmpty(cpuInfo.Groups["CpuFrequency"].Value))
                {
                    rawCpuFrequency = double.Parse(cpuInfo.Groups["CpuFrequency"].Value, System.Globalization.CultureInfo.InvariantCulture);
                }
                else
                {
                    Match frequencyMatch = sCpuFrequencyRegex.Match(cpuString);
                    if (frequencyMatch.Success)
                    {
                        rawCpuFrequency = double.Parse(frequencyMatch.Groups["CpuFrequency"].Value, System.Globalization.CultureInfo.InvariantCulture);
                    }
                }

                if (rawCpuFrequency < 1000)
                {
                    rawCpuFrequency = rawCpuFrequency * 1000;
                }
                else if (rawCpuFrequency > 10000)
                {
                    rawCpuFrequency = rawCpuFrequency / 1000;
                }

                profile.CpuFrequency = (int)rawCpuFrequency;


            }
            else
            {
                // Manufacturer
                profile.CpuManufacturerName = GetCpuManufacturer(cpuInfoString.ToLowerInvariant());

                // Description
                profile.CpuDescription = GetCpuDescription(cpuInfoString);
            }

        }

        private static string GetCpuDescription(string cpuString)
        {
            if (cpuString.Contains(","))
            {
                cpuString = cpuString.Substring(0, cpuString.IndexOf(","));
            }
            if (cpuString.Contains("@"))
            {
                cpuString = cpuString.Substring(0, cpuString.IndexOf("@"));
            }
            return cpuString.Trim();
        }

        private static string GetCpuManufacturer(string cpuString)
        {
            if (IsIntelCpu(cpuString.ToLowerInvariant()))
            {
                return "Intel";
            }
            else if (IsAmdCpu(cpuString.ToLowerInvariant()))
            {
                return "AMD";
            }
            else
            {
                return "Unknown";
            }
        }

        private static bool IsAmdCpu(string cpuString)
        {
            return cpuString.Contains("amd") || cpuString.Contains("athlon");
        }

        private static bool IsIntelCpu(string cpuString)
        {
            return cpuString.Contains("intel") || cpuString.Contains("pentium") || cpuString.Contains("celeron");
        }

        private static void setSoundDate(CHardwareProfile profile, XmlNodeList soundDevices)
        {
            if (soundDevices != null && soundDevices.Count > 0)
            {
                XmlNode primarySoundDevice = soundDevices.Item(0);
                profile.SoundIsPresent = true;
                profile.SoundDescription = primarySoundDevice.SelectSingleNode("Description").InnerXml;
                profile.SoundIsHwAccel = primarySoundDevice.SelectSingleNode("HwAccelLevel").InnerXml != "Emulation Only";
                profile.SoundIsPci = primarySoundDevice.SelectSingleNode("HardwareID").InnerXml.StartsWith("PCI");
            }
            else
            {
                profile.SoundIsPresent = false;
                profile.SoundDescription = string.Empty;
                profile.SoundIsHwAccel = false;
                profile.SoundIsPci = false;
            }
        }

        private static void setHardDriveData(CHardwareProfile profile, XmlNodeList logicalDisks)
        {
            if (logicalDisks == null)
            {
                return;
            }

            foreach (XmlNode node in logicalDisks)
            {
                string fileSystem = node.SelectSingleNode("FileSystem").InnerXml;
                string freeSpace = node.SelectSingleNode("FreeSpace").InnerXml;
                string totalSpace = node.SelectSingleNode("MaxSpace").InnerXml;
                if (string.IsNullOrEmpty(fileSystem) || string.IsNullOrEmpty(freeSpace) || string.IsNullOrEmpty(totalSpace))
                {
                    continue;
                }
                ++profile.HDDCount;
                long freeSpaceBytes = long.Parse(freeSpace, System.Globalization.CultureInfo.InvariantCulture);
                long totalSpaceBytes = long.Parse(totalSpace, System.Globalization.CultureInfo.InvariantCulture);
                profile.HDDFreeSpace += (int)(freeSpaceBytes / (1000 * 1000 * 1000)); // Convert to GB
                profile.HDDTotalSpace += (int)(totalSpaceBytes / (1000 * 1000 * 1000)); // Convert to GB
            }
        }

        public static bool SetFromSystemData(this CHardwareProfile profile, string dxDiagXml)
        {


            XmlDocument doc = new XmlDocument();
            doc.InnerXml = dxDiagXml;

            XmlNodeList displayDevices = doc.SelectNodes("/DxDiag/DisplayDevices/DisplayDevice");
            if (displayDevices.Count == 0)
            {
                Logger.Log("CHardwareProfileManager - No DisplayDevices Found!", ELogLevel.Error);
                return false;
            }

            XmlNode systemInfo = doc.SelectSingleNode("/DxDiag/SystemInformation");
            if (systemInfo == null)
            {
                Logger.Log("CHardwareProfileManager - No SystemInformation Found!", ELogLevel.Error);
                return false;
            }

            XmlNodeList logicalDisks = doc.SelectNodes("/DxDiag/LogicalDisks/LogicalDisk");
            if (logicalDisks == null)
            {
                Logger.Log("CHardwareProfileManager - No LogicalDisks Found!", ELogLevel.Error);
                return false;
            }

            // Misc data
            try
            {
                setMiscData(profile, systemInfo);
            }
            catch (Exception ex)
            {
                Logger.Log("CHardwareProfileManager - Misc Data Exception: {0}", ELogLevel.Error, ex.CompleteDetails());
                return false;
            }


            // Monitor
            try
            {
                setMonitorData(profile, displayDevices.Item(0));
            }
            catch (Exception ex)
            {
                Logger.Log("CHardwareProfileManager - Monitor Data Exception: {0}", ELogLevel.Error, ex.CompleteDetails());
                return false;
            }


            // GPU
            try
            {
                setGpuData(profile, displayDevices.Item(0), (byte)displayDevices.Count);
            }
            catch (Exception ex)
            {
                Logger.Log("CHardwareProfileManager - GPU Data Exception: {0}", ELogLevel.Error, ex.CompleteDetails());
                return false;
            }

            // OS
            try
            {
                setOSData(profile, systemInfo);
            }
            catch (Exception ex)
            {
                Logger.Log("CHardwareProfileManager - OS Data Exception: {0}", ELogLevel.Error, ex.CompleteDetails());
                return false;
            }

            // RAM
            try
            {
                setRamData(profile, systemInfo);
            }
            catch (Exception ex)
            {
                Logger.Log("CHardwareProfileManager - RAM Data Exception: {0}", ELogLevel.Error, ex.CompleteDetails());
                return false;
            }


            // CPU
            try
            {
                setCpuData(profile, systemInfo);
            }
            catch (Exception ex)
            {
                Logger.Log("CHardwareProfileManager - CPU Data Exception: {0}", ELogLevel.Error, ex.CompleteDetails());
                return false;
            }


            // Hard Drive
            try
            {
                setHardDriveData(profile, logicalDisks);
            }
            catch (Exception ex)
            {
                Logger.Log("CHardwareProfileManager - Hard Drive Data Exception: {0}", ELogLevel.Error, ex.CompleteDetails());
                return false;
            }


            // Sound
            try
            {
                XmlNodeList soundDevices = doc.SelectNodes("/DxDiag/DirectSound/SoundDevices/SoundDevice");
                setSoundDate(profile, soundDevices);
            }
            catch (Exception ex)
            {
                Logger.Log("CHardwareProfileManager - Sound Data Exception: {0}", ELogLevel.Error, ex.CompleteDetails());
                return false;
            }

            return true;

        }

        /*
        private static void getNetworkInterfaceCard()
        {
            ManagementObjectSearcher search = new ManagementObjectSearcher("SELECT * FROM Win32_NetworkAdapter where AdapterType like 'Ethernet%'");
            search.Options.Timeout = new TimeSpan(0, 0, 0, 5);
            foreach (ManagementObject info in search.Get())
            {
                string manufacturer = info["Manufacturer"].ToString();
                string description = info["Description"].ToString();
                string adapterType = info["AdapterType"].ToString();
                Debug.WriteLine(manufacturer);
            }
        }
        */


        [DllImport("kernel32.dll", SetLastError = true, CallingConvention = CallingConvention.Winapi)]
        [return: MarshalAs(UnmanagedType.Bool)]
        private static extern bool IsWow64Process([In] IntPtr hProcess, [Out] out bool lpSystemInfo);

        private static bool Is64Bit()
        {
            try
            {
                if (IntPtr.Size == 8 || (IntPtr.Size == 4 && Is32BitProcessOn64BitProcessor()))
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
            catch { }
            return false;
        }

        private static bool Is32BitProcessOn64BitProcessor()
        {
            bool retVal;

            IsWow64Process(Process.GetCurrentProcess().Handle, out retVal);

            return retVal;
        }


        private static string getGpuDriverName(string fullDriverName)
        {

            if (fullDriverName.Contains(","))
            {
                return fullDriverName.Substring(0, fullDriverName.IndexOf(","));
            }
            else if (fullDriverName.Contains(" "))
            {
                return fullDriverName.Substring(0, fullDriverName.IndexOf(" "));
            }
            else
            {
                return fullDriverName;
            }

        }


        private static string getOsName(string fullVersion)
        {
            return fullVersion.Substring(0, fullVersion.IndexOf("(") - 1).Trim();
        }

    }
}
