﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace VoiceSubscriptionMiner.Extensions
{
    public static class StringExtensions
    {
        /// <summary>
        /// Converts a string to an enum value or null if it can't convert.
        /// </summary>
        /// <typeparam name="T">The enum type</typeparam>
        /// <param name="target"></param>
        /// <returns>Converted enum value or null</returns>
        public static T? ToEnum<T>(this string target) where T : struct, IComparable, IFormattable
        {
            T? convertedValue = null;

            if (!string.IsNullOrEmpty(target))
            {
                int intValue;

                if (int.TryParse(target, out intValue))
                {
                    T possibleValue = (T)Enum.ToObject(typeof(T), intValue);
                    if (possibleValue.IsDefined())
                    {
                        return possibleValue;
                    }
                    else
                    {
                        return null;
                    }
                }

                try
                {
                    // clean up string
                    target = target.Replace(" ", "").Replace("\'", "").Replace("-", "");

                    convertedValue = (T)Enum.Parse(typeof(T), target.Trim(), true);
                }
                catch (ArgumentException) { }
                catch (OverflowException) { }
            }

            return convertedValue;
        }
    }

    public static class EnumExtensions
    {
        /// <summary>
        /// Return whether the given enumeration is defined by its type.
        /// </summary>
        /// <typeparam name="T">The enum type</typeparam>
        /// <param name="enumeration">The enum value</param>
        /// <returns>Whether <paramref name="enumeration"/> is defined.</returns>
        public static bool IsDefined<T>(this T enumeration) where T : struct, IComparable, IFormattable
        {
            return EnumIsDefinedHelper<T>.IsDefined(enumeration);
        }

        /// <summary>
        /// Helper class for the IsDefined method.
        /// </summary>
        /// <typeparam name="T">The enum type</typeparam>
        private class EnumIsDefinedHelper<T> where T : struct, IComparable, IFormattable
        {
            /// <summary>
            /// The pre-calculated array of defined values of the enum.
            /// 
            /// This will be calculated only once per <typeparam name="T"/> and on-demand.
            /// </summary>
            private static readonly HashSet<T> DefinedValues =
                new HashSet<T>(
                    Enum.GetNames(typeof(T))
                        .Select(n => (T)Enum.Parse(typeof(T), n)));

            public static bool IsDefined(T value)
            {
                return DefinedValues.Contains(value);
            }
        }
    }
}
