﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.SqlClient;
using System.Configuration;
using Curse;
using Curse.Extensions;

namespace VoiceSubscriptionMiner.Models
{
	public enum SubscriptionStatus
	{
		Active = 0,
		Suspended = 1,
		Cancelled = 2,
		Upgraded = 3,
		PendingCustomerAction = 4,
	}

	public enum AccountVoiceServiceStatus
	{
		Active = 0,
		Inactive = 1,
		Removed = 2
	}

	class SubscriptionData
	{
		public uint ServiceID { get; set; }
		public string ProductID { get; set; }
		public string BIllingPlanID { get; set; }
		public int Channels { get; set; }
		public int StartDay { get; set; }
		public int EndDay { get; set; }
		public SubscriptionStatus Status { get; set; }
		public DateTime DateStarted { get; set; }
		public DateTime DateExpires { get; set; }

		public SubscriptionData() { }
		public SubscriptionData(SqlDataReader reader)
		{
			ServiceID = uint.Parse(reader.GetInt64(reader.GetOrdinal("ServiceID")).ToString());
			ProductID = reader.GetString(reader.GetOrdinal("ProductID"));
			BIllingPlanID = reader.GetString(reader.GetOrdinal("BillingPlanID"));
			Channels = reader.GetInt32(reader.GetOrdinal("Channels"));
			StartDay = reader.GetInt32(reader.GetOrdinal("StartDay"));
			EndDay = reader.GetInt32(reader.GetOrdinal("EndDay"));
			Status = (SubscriptionStatus)reader.GetByte(reader.GetOrdinal("Status"));
			DateStarted = reader.GetDateTime(reader.GetOrdinal("DateStarted"));
			DateExpires = reader.GetDateTime(reader.GetOrdinal("DateExpires"));
		}

		#region	Static	Methods
		public static List<SubscriptionData> GetAllSubscriptionData()
		{
			var subs = new List<SubscriptionData>();

			#region	Query
			var query = @"
				SELECT
					VoiceService.ServiceID, 
					Subscription.ProductID, 
					Subscription.BillingPlanID, 
						CASE 
							WHEN Subscription.BillingPlanID LIKE 'GHV%' 
								THEN SUBSTRING(Subscription.BillingPlanID, 4, PATINDEX('%-%', Subscription.BillingPlanID) - 4) 
							WHEN Subscription.BillingPlanID LIKE 'VH%' 
								THEN SUBSTRING(Subscription.BillingPlanID, 3, PATINDEX('%-%', Subscription.BillingPlanID) - 3) 
							ELSE 0 
						END AS Channels, 
					DATEPART(DD, Subscription.DateStarted) AS StartDay, 
					DATEPART(DD, Subscription.DateExpires) AS EndDay, 
					Subscription.DateStarted, 
					Subscription.DateExpires, 
					Subscription.[Status] 
				FROM BillingService.dbo.AccountVoiceService AS VoiceService
					INNER JOIN BillingService.dbo.AccountVoiceClient AS VoiceClient ON VoiceClient.ID = VoiceService.AccountVoiceClientID
					INNER JOIN BillingService.dbo.Subscription AS Subscription ON Subscription.AccountID = VoiceClient.AccountID
				WHERE Subscription.ProductID IN ('GHV', 'VH')";
			#endregion

			using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["BillingService"].ConnectionString))
			{
				var cmd = new SqlCommand(query, conn);

				conn.Open();
				using (var reader = cmd.ExecuteReader())
				{
					while (reader.Read())
					{
						var sub = new SubscriptionData(reader);
						subs.Add(sub);
					}
				}
				conn.Close();
			}

			return subs;
		}
		public static List<SubscriptionData> GetUpdatedSubscriptions(DateTime lastRunTime)
		{
			var subs = new List<SubscriptionData>();

			#region	Query
			var query = @"
				SELECT 
					VoiceService.ServiceID, 
					Subscription.ProductID, 
					Subscription.BillingPlanID, 
						CASE 
							WHEN Subscription.BillingPlanID LIKE 'GHV%' 
								THEN SUBSTRING(Subscription.BillingPlanID, 4, PATINDEX('%-%', Subscription.BillingPlanID) - 4) 
							WHEN Subscription.BillingPlanID LIKE 'VH%' 
								THEN SUBSTRING(Subscription.BillingPlanID, 3, PATINDEX('%-%', Subscription.BillingPlanID) - 3) 
							ELSE 0 
						END AS Channels, 
					DATEPART(DD, Subscription.DateStarted) AS StartDay, 
					DATEPART(DD, Subscription.DateExpires) AS EndDay, 
					Subscription.DateStarted, 
					Subscription.DateExpires, 
					Subscription.[Status] 
				FROM BillingService.dbo.AccountVoiceService AS VoiceService 
					INNER JOIN BillingService.dbo.AccountVoiceClient AS VoiceClient ON VoiceClient.ID = VoiceService.AccountVoiceClientID 
					INNER JOIN BillingService.dbo.Subscription AS Subscription ON Subscription.AccountID = VoiceClient.AccountID 
				WHERE Subscription.ProductID IN ('GHV', 'VH') 
					AND Subscription.DateUpdated >= @lastRunTime";
			#endregion

			using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["BillingService"].ConnectionString))
			{
				var cmd = new SqlCommand(query, conn);
				cmd.Parameters.Add("lastRunTime", System.Data.SqlDbType.DateTime).Value = lastRunTime;

				conn.Open();
				using (var reader = cmd.ExecuteReader())
				{
					while (reader.Read())
					{
						var sub = new SubscriptionData(reader);
						subs.Add(sub);
					}
				}
				conn.Close();
			}

			return subs;
		}
		public static bool UpdateServiceStatus(uint serviceId, AccountVoiceServiceStatus status)
		{
			using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["BillingService"].ConnectionString))
			{
				var cmd = new SqlCommand("UPDATE AccountVoiceService SET [Status] =	@Status	WHERE ServiceID	= @ServiceID", conn);
				cmd.Parameters.Add("Status", System.Data.SqlDbType.TinyInt).Value = status;
				cmd.Parameters.Add("ServiceID", System.Data.SqlDbType.Int).Value = serviceId;

				try
				{
					conn.Open();
					cmd.ExecuteNonQuery();
					conn.Close();
					return true;
				}
				catch (SqlException exc)
				{
					Logger.Log("Failed	to	update	service	status	for	service	ID	{0}. Details: {1}", ELogLevel.Error, serviceId, exc.GetExceptionDetails());
					conn.Close();
					return false;
				}
			}
		}

		public static void InsertLightSpeedHistory(uint serviceId, string actionSentToLightSpeed, DateTime dateSentToLightSpeed, string responseFromLightSpeed)
		{
			using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["BillingService"].ConnectionString))
			{
				var cmd = new SqlCommand("INSERT INTO AccountVoiceServiceLightSpeedHistory (ServiceID, ActionSentToLightSpeed, DateSentToLightSpeed, ResponseFromLightSpeed) VALUES(@ServiceId, @ActionSentToLightSpeed, @DateSentToLightSpeed, @ResponseFromLightSpeed)");
				cmd.Parameters.Add("ServiceId", System.Data.SqlDbType.BigInt).Value = serviceId;
				cmd.Parameters.Add("ActionSentToLightSpeed", System.Data.SqlDbType.VarChar).Value = actionSentToLightSpeed;
				cmd.Parameters.Add("DateSentToLightSpeed", System.Data.SqlDbType.DateTime).Value = dateSentToLightSpeed;
				string temp = responseFromLightSpeed;
				if (temp.Length > 60)
				{
					temp = responseFromLightSpeed.Substring(0, 59);
				}
				cmd.Parameters.Add("ResponseFromLightSpeed", System.Data.SqlDbType.VarChar).Value = temp;

				try
				{
					conn.Open();
					cmd.Connection = conn;
					cmd.ExecuteNonQuery();
					conn.Close();
				}
				catch (SqlException exc)
				{
					conn.Close();
					Logger.Log("Failed to insert history for service id: {0} details: {1}", ELogLevel.Error, serviceId, exc.GetExceptionDetails());
				}
			}
		}

		#endregion
	}
}
