﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Curse;
using Curse.Extensions;
using System.IO;
using VoiceSubscriptionMiner.Models;
using VoiceSubscriptionMiner.Services;
using System.ServiceModel;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;

namespace VoiceSubscriptionMiner
{
	enum ThreadStatus
	{
		ACTIVE,
		SLEEPING
	}

	class VoiceSubscriptionManager
	{
		bool _isAlive = true;

		TimeSpan _reportingTimeout;
		Thread _reportingThread;
		DateTime _lastReportingRunTime = DateTime.Parse("01/01/1975");
		ThreadStatus _reportingStatus = ThreadStatus.SLEEPING;
		string _reportingProgress = string.Empty;

		TimeSpan _updateServiceTimeout;
		Thread _updateServiceThread;
		DateTime _lastUpdateThreadRuntime = DateTime.Parse("01/01/1975");
		ThreadStatus _updateServiceThreadStatus = ThreadStatus.SLEEPING;
		string _updateServiceProgress = string.Empty;
		Queue<ServiceUpdateAction> _servicesToUpdate;

		static VoiceSubscriptionManager _instance = new VoiceSubscriptionManager();
		public static VoiceSubscriptionManager Instance { get { return _instance; } }

		public VoiceSubscriptionManager()
		{
			var reportingTimeout = Double.Parse(ConfigurationManager.AppSettings["ReportingTimeout"]);
			var cancelTimeout = Double.Parse(ConfigurationManager.AppSettings["UpdateServiceTimeout"]);

			var timeoutGrain = ConfigurationManager.AppSettings["TimeoutGrain"];
			switch (timeoutGrain)
			{
				case "milliseconds":
					_reportingTimeout = TimeSpan.FromMilliseconds(reportingTimeout);
					_updateServiceTimeout = TimeSpan.FromMilliseconds(cancelTimeout);
					break;
				case "seconds":
					_reportingTimeout = TimeSpan.FromSeconds(reportingTimeout);
					_updateServiceTimeout = TimeSpan.FromSeconds(cancelTimeout);
					break;
				case "minutes":
					_reportingTimeout = TimeSpan.FromMinutes(reportingTimeout);
					_updateServiceTimeout = TimeSpan.FromMinutes(cancelTimeout);
					break;
				case "hours":
					_reportingTimeout = TimeSpan.FromHours(reportingTimeout);
					_updateServiceTimeout = TimeSpan.FromHours(cancelTimeout);
					break;
				case "days":
					_reportingTimeout = TimeSpan.FromDays(reportingTimeout);
					_updateServiceTimeout = TimeSpan.FromDays(cancelTimeout);
					break;
			}
			_servicesToUpdate = new Queue<ServiceUpdateAction>();

			_reportingThread = new Thread(ReportingThread) { IsBackground = true };
			_updateServiceThread = new Thread(UpdateServiceThread) { IsBackground = true };
		}

		public void Start(bool immediate)
		{
			new Thread(UpdateProgress) { IsBackground = true }.Start();

			if (immediate)
			{
				CheckForExpiredEntitlements();
				UpdateServices();
			}

			if (_reportingThread != null)
			{
				_reportingThread.Start();
			}

			if (_updateServiceThread != null)
			{
				_updateServiceThread.Start();
			}
		}
		public void Stop()
		{
			_isAlive = false;
		}

		#region Thread Methods
		//void ReportingThread()
		//{
		//    while (_isAlive)
		//    {
		//        Thread.Sleep(_reportingTimeout);
		//        Logger.Log("Checking for updated subscriptions...", ELogLevel.Info);

		//        _reportingStatus = ThreadStatus.ACTIVE;
		//        try
		//        {
		//            var runTime = DateTime.UtcNow;
		//            List<SubscriptionData> subscriptions = SubscriptionData.GetUpdatedSubscriptions(_lastReportingRunTime);
		//            if (subscriptions.Count > 0)
		//            {
		//                Logger.Log("Processing {0} Subscriptions...", ELogLevel.Info, subscriptions.Count);
		//            }

		//            var serviceIds = new List<long>();
		//            for(int i = 0; i < subscriptions.Count; i++)
		//            {
		//                SubscriptionData subscription = subscriptions[i];
		//                _reportingProgress = string.Format("Processing Subscription {0} of {1}", i + 1, subscriptions.Count);

		//                if (serviceIds.Contains(subscription.ServiceID))
		//                {
		//                    continue; // skip this one we have already done it
		//                }

		//                // Get the service status from the API
		//                VentriloServiceInformation serviceInfo;
		//                try
		//                {
		//                    serviceInfo = VentrilloService.Instance.GetVoiceSubStatus(subscription.ServiceID);
		//                }
		//                catch (FaultException exc)
		//                {
		//                    //Logger.Log("Failed to pull the status for Service ID {0}! Details: {1}", ELogLevel.Error, subscription.ServiceID, exc.GetExceptionDetails());
		//                    serviceIds.Add(subscription.ServiceID);
		//                    continue;
		//                }

		//                // Process the results
		//                if (subscription.Status != SubscriptionStatus.Active)
		//                {
		//                    //check for an active subscription
		//                    var activeSub = subscriptions.FirstOrDefault(p => p.Status == SubscriptionStatus.Active && p.ServiceID == subscription.ServiceID);
		//                    if (serviceInfo.ServerStatus == VentriloServiceServerStatus.Online)
		//                    {
		//                        if (activeSub == null)
		//                        {
		//                            //get the most recent subscription end date
		//                            //var mostRecentSubs = subscriptions.Where(p => p.ServiceID == subscription.ServiceID).OrderByDescending(p => p.DateExpires);
		//                            if (_servicesToUpdate.Count(p => p.Key == subscription.ServiceID) <= 0)
		//                            {
		//                                Logger.Log("Queuing Service for update: {0}", ELogLevel.Info, subscription.ServiceID);
		//                                _servicesToUpdate.Enqueue(new KeyValuePair<uint, VentriloServiceAction>(subscription.ServiceID, VentriloServiceAction.Uninstall));
		//                            }
		//                        }
		//                        else
		//                        {
		//                            //check the port counts
		//                            if (activeSub.Channels != serviceInfo.MaxClients)
		//                            {
		//                                //UpdateSlotCount(sub.ServiceID, sub.Channels);
		//                            }
		//                        }
		//                    }
		//                }
		//                else
		//                {
		//                    //check the port counts
		//                    if (subscription.Channels != serviceInfo.MaxClients) 
		//                    {
		//                        //UpdateSlotCount(sub.ServiceID, sub.Channels);
		//                    }
		//                }

		//                serviceIds.Add(subscription.ServiceID);
		//            }
		//            //Logger.Log("Queuing services for update: {0}", ELogLevel.Info, string.Join(", ", _servicesToUpdate.Select(p => p.Key.ToString()).ToArray()));

		//            _lastReportingRunTime = runTime;
		//            _reportingProgress = string.Empty;
		//        }
		//        catch (ThreadAbortException exc)
		//        {
		//            Logger.Log("Reporting Thread Aborted: {0}", ELogLevel.Error, exc.GetExceptionDetails());
		//            _isAlive = false;
		//        }

		//        _reportingStatus = ThreadStatus.SLEEPING;
		//    }
		//}
		void ReportingThread()
		{
			while (_isAlive)
			{
				Thread.Sleep(_reportingTimeout);
				Logger.Log("Checking for updated subscriptions...", ELogLevel.Info);

				_reportingStatus = ThreadStatus.ACTIVE;
				try
				{
					CheckForExpiredEntitlements();
				}
				catch (ThreadAbortException exc)
				{
					Logger.Log("Reporting Thread Aborted: {0}", ELogLevel.Error, exc.GetExceptionDetails());
					_isAlive = false;
				}

				_reportingStatus = ThreadStatus.SLEEPING;
			}
		}
		void CheckForExpiredEntitlements()
		{
			var runTime = DateTime.UtcNow;
			List<AccountEntitlementData> entitlements = AccountEntitlementData.GetUpdatedEntitlements(_lastReportingRunTime);
			List<SubscriptionData> subscriptions = SubscriptionData.GetAllSubscriptionData();
			if (entitlements.Any())
			{
				Logger.Log("Processing {0} Subscriptions...", ELogLevel.Info, entitlements.Count);
			}

			//entitlements = entitlements.Where(p => p.ServiceID == uint.Parse("19629")).ToList();

			var serviceIds = new List<long>();
			for (int i = 0; i < entitlements.Count; i++)
			{
				AccountEntitlementData entitlement = entitlements[i];
				_reportingProgress = string.Format("Processing Subscription {0} of {1}", i + 1, entitlements.Count);

				if (serviceIds.Contains(entitlement.ServiceID))
				{
					Logger.Log("skipping {0} cause it is in the serviceIds List", ELogLevel.Info, entitlement.ServiceID);
					continue; // skip this one we have already done it
				}

				// Get the service status from the API
				VentriloServiceInformation serviceInfo;
				try
				{
					serviceInfo = VentrilloService.Instance.GetVoiceSubStatus(entitlement.ServiceID);
				}
				catch (FaultException exc)
				{
					//Logger.Log("Failed to pull the status for Service ID {0}! Details: {1}", ELogLevel.Error, subscription.ServiceID, exc.GetExceptionDetails());
					serviceIds.Add(entitlement.ServiceID);
					continue;
				}

				// Process the results
				if (!entitlement.IsActive)
				{
					//check for an active subscription
					var activeEntitlement = entitlements.FirstOrDefault(p => p.IsActive == true && p.ServiceID == entitlement.ServiceID);
					if (serviceInfo.ServerStatus != VentriloServiceServerStatus.Uninstalled)
					{
						if (activeEntitlement == null)
						{
							//get the most recent subscription end date
							DateTime mostRecentSubExpiration = subscriptions.Where(p => p.ServiceID == entitlement.ServiceID).Max(p => p.DateExpires);
							if (mostRecentSubExpiration > DateTime.Now)
							{
								Logger.Log("Entitlement for ServiceID: {0} Expired on {1} with a Future Subscription Expiry Date: {2}", ELogLevel.Warning, entitlement.ServiceID, entitlement.DateExpires, mostRecentSubExpiration);
							}
							else if (!_servicesToUpdate.Any(p => p.ServiceID == entitlement.ServiceID))
							{
								Logger.Log("Queuing Service for uninstallation: {0}", ELogLevel.Info, entitlement.ServiceID);
								_servicesToUpdate.Enqueue(new ServiceUpdateAction { Action = VentriloServiceAction.Uninstall, ServiceID = entitlement.ServiceID });
							}
							else
							{
								// this shouldnt hit but I am putting a break point here just in case
								Logger.Log("If you're reading this something did not go quite right...", ELogLevel.Info, entitlement.ServiceID);
							}
						}
						else
						{
							//check the port counts
							if (activeEntitlement.MaxClients != serviceInfo.MaxClients)
							{
								Logger.Log("Queuing Service to update max client count: {0}", ELogLevel.Info, entitlement.ServiceID);
								_servicesToUpdate.Enqueue(new ServiceUpdateAction { Action = VentriloServiceAction.UpdateMaxClientCount, ServiceID = entitlement.ServiceID, MaxClientCount = activeEntitlement.MaxClients });
							}
						}
					}
				}
				else
				{
					//check the port counts
					if (entitlement.MaxClients != serviceInfo.MaxClients)
					{
						Logger.Log("Queuing Service to update max client count: {0}", ELogLevel.Info, entitlement.ServiceID);
						_servicesToUpdate.Enqueue(new ServiceUpdateAction { Action = VentriloServiceAction.UpdateMaxClientCount, ServiceID = entitlement.ServiceID, MaxClientCount = entitlement.MaxClients });
					}
				}

				serviceIds.Add(entitlement.ServiceID);
			}

			_lastReportingRunTime = runTime;
			_reportingProgress = string.Empty;
			serviceIds.Clear();
		}

		void UpdateServiceThread()
		{
			while (_isAlive)
			{
				Thread.Sleep(_updateServiceTimeout);
				Logger.Log("Checking for services to update...", ELogLevel.Info);

				_updateServiceThreadStatus = ThreadStatus.ACTIVE;
				try
				{
					UpdateServices();
				}
				catch (ThreadAbortException exc)
				{
					Logger.Log("Cancel Service Thread Aborted: {0}", ELogLevel.Error, exc.GetExceptionDetails());
					_isAlive = false;
				}
				_updateServiceThreadStatus = ThreadStatus.SLEEPING;
			}
		}
		void UpdateServices()
		{
			var runtime = DateTime.UtcNow;

			int queueCount = _servicesToUpdate.Count;
			for (int i = 0; i < queueCount; i++)
			{
				// this is to hold the result from light speed to be later inserted into history table
				string historyResult = "non gotten";

				_updateServiceProgress = string.Format("Updating Service {0} of {1}", i + 1, queueCount);

				var updateAction = _servicesToUpdate.Dequeue();
				Logger.Log("Updating Service; ID={0} Action={1}", ELogLevel.Info, updateAction.ServiceID, updateAction.Action.ToString());
				try
				{
					bool result = false;
					switch (updateAction.Action)
					{
						case VentriloServiceAction.Uninstall:
							{
								result = VentrilloService.Instance.CancelService(updateAction.ServiceID, updateAction.Action);
								SubscriptionData.UpdateServiceStatus(updateAction.ServiceID, AccountVoiceServiceStatus.Inactive);
								Logger.Log("{0} was set to {1} with result of {2}", ELogLevel.Info, updateAction.ServiceID, updateAction.Action, result);
							} break;
						case VentriloServiceAction.UpdateMaxClientCount:
							{
								result = VentrilloService.Instance.UpdateSlotCount(updateAction.ServiceID, updateAction.MaxClientCount);
								Logger.Log("{0} was set to {1} with result of {2}", ELogLevel.Info, updateAction.ServiceID, updateAction.MaxClientCount, result);
							} break;
					}

					if (result == false)
					{
						historyResult = result.ToString();
						Logger.Log("Failed to update ServiceID: {0}", ELogLevel.Warning, updateAction.ServiceID);
						_servicesToUpdate.Enqueue(updateAction);
					}
					else // result true light speed got the action and perfomed it.
					{
						historyResult = result.ToString();
					}
				}
				catch (Exception exc)
				{
					Logger.Log("Failed to update ServiceID {0}. Details: {1}", ELogLevel.Warning, updateAction.ServiceID, exc.GetExceptionDetails());
					historyResult = exc.Message;
					if (historyResult != null && historyResult.Length > 254)
					{
						historyResult.Remove(0);
						historyResult = exc.Message.Substring(0, 254);
					}
					if (historyResult == null) { historyResult = "no message from light speed"; }
					_servicesToUpdate.Enqueue(updateAction);
				}
				// Insert action on service id that was sent to light speed and result
				SubscriptionData.InsertLightSpeedHistory(updateAction.ServiceID, updateAction.Action.ToString(), runtime, historyResult);
			} //ENDOF for (int i = 0; i < queueCount; i++)

			_lastUpdateThreadRuntime = runtime;
			_updateServiceProgress = string.Empty;
		}

		void UpdateProgress()
		{
			while (_isAlive)
			{
				Thread.Sleep(240);

				string console = "----------------------------------------------------------------------" + Environment.NewLine;
				console += string.Format("Reporting Status: {0} - {1}", _reportingStatus.ToString(), DateTime.UtcNow) + Environment.NewLine;
				console += "----------------------------------------------------------------------" + Environment.NewLine;
				console += string.Format("Last Runtime: {0}", _lastReportingRunTime) + Environment.NewLine;
				console += string.Format("Next Runtime: {0}", _lastReportingRunTime.AddTicks(_reportingTimeout.Ticks)) + Environment.NewLine;
				console += "----------------------------------------------------------------------" + Environment.NewLine;
				console += _reportingProgress + Environment.NewLine;
				console += Environment.NewLine + Environment.NewLine;
				console += "----------------------------------------------------------------------" + Environment.NewLine;
				console += string.Format("Update Status: {0} - Services Queued: {1}", _updateServiceThreadStatus.ToString(), _servicesToUpdate.Count) + Environment.NewLine;
				console += "----------------------------------------------------------------------" + Environment.NewLine;
				console += string.Format("Last Runtime: {0}", _lastUpdateThreadRuntime) + Environment.NewLine;
				console += string.Format("Next Runtime: {0}", _lastUpdateThreadRuntime.AddTicks(_updateServiceTimeout.Ticks)) + Environment.NewLine;
				console += "----------------------------------------------------------------------" + Environment.NewLine;
				console += _updateServiceProgress + Environment.NewLine;

				Console.Clear();
				Console.WriteLine(console);
			}
		}
		#endregion

		void ValidateInvoice(string filename)
		{
			var voiceSubs = VoiceSubInvoiceData.ParseInvoiceFile(filename);
			VoiceSubInvoiceData.GenerateValidationReport(voiceSubs, @"C:\temp\SubscriptionErrors.txt");
			VoiceSubInvoiceData.GenerateChannelsByDayReport(voiceSubs, @"C:\temp\ChannelReportByDay.txt");
			VoiceSubInvoiceData.GenerateMissingSubscriptionReport(voiceSubs, @"C:\temp\missingVoiceSubs.txt");
			VoiceSubInvoiceData.GenerateServiceWithoutSubscriptionReport(voiceSubs, @"C:\temp\ServicesWithoutSubscriptions.txt");
		}
		void ReconcileVoiceSubs()
		{
			var subs = SubscriptionData.GetAllSubscriptionData();
			var activeSubs = subs.Count(p => p.Status == SubscriptionStatus.Active);

			var brokenSubs = subs.Where(p => p.ServiceID == 13507).ToList();

			var tw = new TabWriter(@"C:\temp\ventrolloReconciliation.txt");
			var serviceIds = new List<long>();
			foreach (var sub in subs)
			{
				if (serviceIds.Contains(sub.ServiceID))
				{
					continue; // Skip this one if we have done it before
				}

				Console.Write(sub.ServiceID);

				VentriloServiceInformation serviceInfo;
				try
				{
					serviceInfo = VentrilloService.Instance.GetVoiceSubStatus(sub.ServiceID);
				}
				catch (FaultException)
				{
					serviceIds.Add(sub.ServiceID);

					tw.Write(sub.ServiceID);
					tw.Write(sub.Status.ToString());
					tw.Write("Service ID does not exist.");
					tw.EndLine();

					Console.Write("\t" + sub.Status.ToString());
					Console.Write("\tService ID does not exist.\n");

					continue;
				}

				if (sub.Status != SubscriptionStatus.Active)
				{
					//check for an active subscription
					var activeSub = subs.FirstOrDefault(p => p.Status == SubscriptionStatus.Active && p.ServiceID == sub.ServiceID);
					if (serviceInfo.ServerStatus == VentriloServiceServerStatus.Online)
					{
						if (activeSub == null)
						{
							//get the most recent subscription end date
							var mostRecentSubs = subs.Where(p => p.ServiceID == sub.ServiceID).OrderByDescending(p => p.DateExpires);

							//This should not be online
							tw.Write(sub.ServiceID);
							tw.Write(sub.Status.ToString());
							tw.Write(serviceInfo.ServerStatus.ToString());
							tw.Write(mostRecentSubs.First().DateExpires);
							tw.EndLine();

							Console.Write("\t" + sub.Status.ToString());
							Console.Write("\t" + serviceInfo.ServerStatus.ToString());
							Console.Write("\t" + mostRecentSubs.First().DateExpires.ToShortDateString() + "\n");

							//CancelService(sub.ServiceID, VentriloServiceAction.Uninstall);
						}
						else
						{
							//check the port counts
							if (activeSub.Channels != serviceInfo.MaxClients)
							{
								//The port count is off
								tw.Write(sub.ServiceID);
								tw.Write(activeSub.Channels);
								tw.Write(serviceInfo.MaxClients);
								tw.EndLine();

								Console.Write("\t" + activeSub.Channels);
								Console.Write("\t" + serviceInfo.MaxClients + "\n");

								//UpdateSlotCount(sub.ServiceID, sub.Channels);
							}
							else Console.Write("\tOK\n");
						}
					}
					else Console.Write("\tOK\n");
				}
				else
				{
					//check the port counts
					if (sub.Channels != serviceInfo.MaxClients)
					{
						//The port count is off
						tw.Write(sub.ServiceID);
						tw.Write(sub.Channels);
						tw.Write(serviceInfo.MaxClients);
						tw.EndLine();

						Console.Write("\t" + sub.Channels);
						Console.Write("\t" + serviceInfo.MaxClients + "\n");

						//UpdateSlotCount(sub.ServiceID, sub.Channels);
					}
					else Console.Write("\tOK\n");
				}

				serviceIds.Add(sub.ServiceID);
			}
			tw.Close();
		}

		#region Static Members
		public static bool CancelVoiceSub(uint serviceId)
		{
			return VentrilloService.Instance.CancelService(serviceId, VentriloServiceAction.Uninstall);
		}
		#endregion
	}
}
