﻿using Curse;
using Curse.WAR;

using System;
using System.Text;
using System.Data;
using System.Data.SqlClient;
using System.Collections.Generic;
using System.Configuration;


namespace WARDataService
{
    public static partial class DB
    {
        public static String[] sLocalizations = null;                

        public static Boolean Load()
        {
            CPUMonitor monitorTotal = new CPUMonitor("DB Loading");
            Boolean ok = true;
            sLocalizations = System.Enum.GetNames(typeof(ELocale));
            
            try
            {                                
                DBServers.Initialize();
                DBGuilds.Initialize();
                DBPlayers.Initialize();
                DBItems.Initialize();
                DBItemSets.Initialize();
                DBNPCs.Initialize();
                DBGameObjects.Initialize();
                DBQuests.Initialize();
                DBInfluence.Initialize();
                DBAdvances.Initialize();
                DBAbilities.Initialize();
                DBPublicQuests.Initialize();
                DBBattlefieldStatus.Initialize();
            }
            catch (Exception exc)
            {
                ok = false;
                Logger.Log(ELogLevel.Debug,
                           null,
                           "DB Load Exception: {0}",
                           exc.Message);
            }

            Logger.Log(ELogLevel.Debug,
                       null,
                       monitorTotal.ToString());

            return ok;
        }

        public static Boolean IsEntityConfirmed(Dictionary<CustomKey, Confirmation> pConfirmations,
                                                CustomKey pKey,
                                                Int32 pRequiredConfirmations,
                                                Int32 pUserId,
                                                Int32 pVersion,
                                                bool pIsTrustedUser)
        {
            if (pIsTrustedUser)
            {
                return true;
            }

            if (pRequiredConfirmations < 2 || pVersion == Config.Instance.SkipConfirmationForVersion)
            {
                return true;
            }
                        
            Confirmation confirmation = null;
            if (!pConfirmations.TryGetValue(pKey, out confirmation))
            {
                confirmation = new Confirmation(pUserId);

                lock (pConfirmations)
                {
                    pConfirmations[pKey] = confirmation;
                }                
            }
            else
            {
                ++confirmation.Count;
            }
            
            return confirmation.Count >= pRequiredConfirmations;            
        }

        public static Boolean IsEntityExpired(Dictionary<CustomKey, DateTime> pHistory,
                                                CustomKey pKey,
                                                Int32 pExpirationSeconds,                                                
                                                bool pIsTrustedUser)
        {
            if (pIsTrustedUser)
            {
                return true;
            }

            DateTime found;
            DateTime now = DateTime.UtcNow;

            if (pHistory.TryGetValue(pKey, out found))
            {
                if (now.Subtract(found).TotalSeconds < pExpirationSeconds)
                {
                    return false;
                }
            }
            lock (pHistory)
            {
                pHistory[pKey] = now;
            }
            return true;

        }

        public static Boolean ReadyForDB(Dictionary<CustomKey, DateTime> pExpirations,
                                          Dictionary<CustomKey, Confirmation> pConfirmations,
                                          Int32 pKey,
                                          Int32 pExpiration,
                                          Int32 pConfirmationCount,
                                          Int32 pUserId,
                                          Int32 pVersion,                    
                                          ELocale pLocale,
                                          bool pIsTrustedUser)
        {
            if (pIsTrustedUser)
            {
                return true;
            }

            CustomKey key = new CustomKey(pVersion,
                                          pLocale,
                                          pKey);

            if (!IsEntityConfirmed(pConfirmations,
                            key,
                            pConfirmationCount,
                            pUserId,
                            pVersion, pIsTrustedUser))
            {
                return false;
            }
            return IsEntityExpired(pExpirations, key, pExpiration, pIsTrustedUser);
        }

       
        public static void LoadSchema(ref DataTable pSchema,
                                       String pTable)
        {
            pSchema = new DataTable();

            using (SqlConnection conn = new SqlConnection(Config.Instance.ConnectionString))
            {
                conn.Open();
                
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "SELECT TOP 1 * FROM " + pTable + " with(nolock)";
                using (SqlDataReader dr = cmd.ExecuteReader(CommandBehavior.SchemaOnly | CommandBehavior.KeyInfo))
                {
                    pSchema.Load(dr);
                }                                        
            }
        }

        public static void LoadSchema(ref DataTable pSchema, String pTable, String pPreSql)
        {
            
            pSchema = new DataTable();

            using (SqlConnection conn = new SqlConnection(Config.Instance.ConnectionString))
            {
                conn.Open();

                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = pPreSql;
                cmd.ExecuteNonQuery();
                cmd.CommandText = "SELECT TOP 1 * FROM " + pTable + " with(nolock)";
                using (SqlDataReader dr = cmd.ExecuteReader(CommandBehavior.SchemaOnly | CommandBehavior.KeyInfo))
                {
                    pSchema.Load(dr);
                }
            }
        }
        public static bool IsLocalizedColumn(String pColumName)
        {
            foreach (String localization in sLocalizations)
            {
                if(pColumName.EndsWith("_"+localization))
                {
                    return true;
                }
            }
            return false;
        }
        
        public static void AddNameFromId(Dictionary<CustomKey, Int32> pHash, CustomKey pKey, Int32 pId)
        {
            if (!pHash.ContainsKey(pKey))
            {
                pHash.Add(pKey, pId);
            }
        }
        public static Int32 GetIdFromName(Dictionary<CustomKey, Int32> pHash, CustomKey pKey)
        {
            Int32 id = 0;
            pHash.TryGetValue(pKey, out id);
            return id;
        }

        public static void AddNameFromId(Dictionary<CustomKey, Int32> pHash, String pLocale, String pName, Int32 pId)
        {
            CustomKey key;
            if (pLocale.Length > 0)
            {
                key = new CustomKey(pName, pLocale);
            }
            else
            {
                key = new CustomKey(pName);
            }
            if (!pHash.ContainsKey(key))
            {
                pHash.Add(key, pId);
            }
        }

        public static Int32 GetIdFromName(Dictionary<CustomKey,Int32> pHash, String pName, String pLocale)
        {
            if (pName.Length == 0)
            {
                return 0;
            }
            CustomKey key;
            if (pLocale.Length > 0)
            {
                key = new CustomKey(pName, pLocale);
            }
            else
            {
                key = new CustomKey(pName);
            }
            if (!pHash.ContainsKey(key))
            {
                return 0;
            }
            return pHash[key];
        }

        public static string GetUpdateSQL(DataTable pSchema, String[] pIgnoredColumns, Dictionary<String, String> pCustomColumns)
        {
            List<String> handledColumns = new List<String>();
            String columnName = null;
            String updateSQL = String.Empty;
            bool isLocalizedColumn = false;
            foreach (DataColumn dc in pSchema.Columns)
            {

                isLocalizedColumn = IsLocalizedColumn(dc.ColumnName);

                if (isLocalizedColumn)
                {
                    columnName = dc.ColumnName.Substring(0, dc.ColumnName.LastIndexOf("_"));
                }
                else
                {
                    columnName = dc.ColumnName;
                }

                if (Array.IndexOf(pIgnoredColumns, dc.ColumnName) >= 0 || handledColumns.Contains(columnName))
                {
                    continue;
                }
                
                if (pCustomColumns.ContainsKey(columnName))
                {
                    updateSQL = updateSQL + "," + pCustomColumns[columnName];
                    handledColumns.Add(columnName);
                }
                else if (isLocalizedColumn)
                {                    
                    if (handledColumns.Contains(columnName))
                    {
                        continue;
                    }
                    updateSQL = updateSQL + "," + columnName + "_{0}=tmp." + columnName + "_{0}";
                    handledColumns.Add(columnName);
                }
                else
                {
                    updateSQL = updateSQL + "," + dc.ColumnName + "=tmp." + dc.ColumnName;
                    handledColumns.Add(dc.ColumnName);
                }
            }
            return updateSQL.Substring(1);
        }

        public static string GetFilledInsertSQL(String insertQuery, DataTable pSchema, DataRow pDataRow, String[] pIgnoredColumns, String pLocale)
        {
            
            //foreach (DataColumn dc in pSchema.Columns)
            //{
                //insertQuery = String.Format(insertQuery, pDataRow.ItemArray)
            //}
            List<String> values = new List<String>();
            foreach (DataColumn dc in pSchema.Columns)
            {
                if (Array.IndexOf(pIgnoredColumns, dc.ColumnName) >= 0)
                {
                    continue;
                }
                String value = pDataRow[dc.ColumnName].ToString();
                if (dc.DataType.Name == "String")
                {
                    value = value.Replace("'", "''");
                }

                if (IsLocalizedColumn(dc.ColumnName))
                {
                    if(!dc.ColumnName.EndsWith("_"+pLocale))
                    {
                        continue;
                    }

                    values.Add(value);
                }
                else
                {
                    values.Add(value);
                }
            }

            return String.Format(insertQuery, values.ToArray());
            
            
        }
        public static string GetInsertSQL(DataTable pSchema, String[] pIgnoredColumns)
        {
            return GetInsertSQL(pSchema, pIgnoredColumns, true);
        }
        public static string GetInsertSQL(DataTable pSchema, String[] pIgnoredColumns, Boolean pReturnId)
        {
            List<String> handledColumns = new List<String>();
            String columnName = null;
            String columns = String.Empty;
            String values = String.Empty;
            Int32 columnIndex = 0;
            foreach (DataColumn dc in pSchema.Columns)
            {               
                columnName = dc.ColumnName;
                if (Array.IndexOf(pIgnoredColumns, dc.ColumnName) >= 0)
                {
                    continue;
                }

                if (IsLocalizedColumn(columnName))
                {
                    columnName = dc.ColumnName.Substring(0, dc.ColumnName.LastIndexOf("_"));
                    if (handledColumns.Contains(columnName))
                    {
                        continue;
                    }
                    columns += "," + dc.ColumnName;
                    values  += ",N'{" + columnIndex + "}'";
                    handledColumns.Add(columnName);
                }
                else
                {
                    columns += "," + dc.ColumnName;

                    if(dc.DataType  == typeof(String))
                    {
                        values += ",N'{" + columnIndex + "}'";   
                    }
                    else if(dc.DataType  == typeof(DateTime))
                    {
                        values += ",'{" + columnIndex + "}'";   
                    }
                    else{
                        values += ",{" + columnIndex + "}";   
                    }
                                       
                                     
                    handledColumns.Add(dc.ColumnName);
                }
                columnIndex += 1;
            }
            if (pReturnId)
            {
                return "insert into " + pSchema.TableName + "(" + columns.Substring(1) + ") output inserted.id values(" + values.Substring(1) + ")";
            }
            else
            {
                return "insert into " + pSchema.TableName + "(" + columns.Substring(1) + ") values(" + values.Substring(1) + ")";
            }
            
        }

        public static void PopulateId(HashSet<CustomKey> pHash,
                                       String pEntityTable,
                                       Int32 pVersion)
        {
            String query = null;
            query = "select id from " + pEntityTable + " with(nolock) where version_id=" + pVersion;

            CustomKey key;
            Int32 id;
            Int32 count = 0;
            
            using (SqlConnection conn = new SqlConnection(Config.Instance.ConnectionString))
            {
                conn.Open();

                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = query;
                using (SqlDataReader dr = cmd.ExecuteReader())
                {
                    while (dr.Read())
                    {
                        id = (Int32)dr[0];
                        
                        key = new CustomKey(pVersion,
                                            id);                        
                        pHash.Add(key);
                        ++count;
                    }
                }             
            }
            Logger.Log(ELogLevel.Debug,
                       null,
                       "Populated {0} ids from table {1} for version {2}",
                       count,
                       pEntityTable,
                       pVersion);
        }
        
        public static void PopulateNameToID(Dictionary<CustomKey,Int32> pHash,
                                      String pEntityTable,
                                      String pLocale)
        {
            String query = null;
            if (pLocale.Length == 0)
            {
                query = "select distinct id, name from " + pEntityTable + " with(nolock) where name is not null";
            }
            else
            {
                query = "select distinct id, name_" + pLocale + " from " + pEntityTable + " with(nolock) where name_" + pLocale + " is not null";
            }
                        
            Int32 id;
            Int32 count = 0;
            using (SqlConnection conn = new SqlConnection(Config.Instance.ConnectionString))
            {
                conn.Open();

                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = query;
                using (SqlDataReader dr = cmd.ExecuteReader())
                {
                    while (dr.Read())
                    {
                        id = (Int32)dr[0];
                        AddNameFromId(pHash, pLocale, (String)dr[1].ToString(), id);
                        ++count;
                    }
                }
            }
            Logger.Log(ELogLevel.Debug,
                       null,
                       "Populated {0} names from table {1} for locale {2}",
                       count,
                       pEntityTable,
                       pLocale);
        }

    }
}
