﻿using Curse;
using Curse.WAR;
using System;
using System.Data.SqlClient;
using System.Collections.Generic;
using System.Data;
using System.Text;
namespace WARDataService
{
    public static partial class DBAbilities
    {

        private const Byte NUM_REQUIREMENTS = 3;

        private static HashSet<CustomKey> sKnownAbilities =
            new HashSet<CustomKey>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, DateTime> sAbilityExpirations =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, DateTime> sAbilityDescriptionExpirations =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, Confirmation> sConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, String> sAbilityDescriptionCache =
            new Dictionary<CustomKey, String>(new CustomKey.CustomKeyComparer());

        private static String sUpdateQuery = null;

        private static String[] sUpdateIgnoredColumns = { "id", "version_id", "posted", "posted_id" };


        private static String sAbilityDescriptionQueryFormat = "UPDATE ability_description SET " +
                                                  "description_{0}=N'{1}'" +                                                  
                                                  " WHERE ability_id={2} AND version_id={3} and level={4}" +
                                                  " IF @@ROWCOUNT = 0 " +
                                                  "INSERT INTO ability_description(ability_id," +
                                                  "version_id,level,description_{0})" +
                                                  " VALUES(" +
                                                  "{2},{3},{4},N'{1}');";

        private static Dictionary<string, string> sUpdateCustomColumns = new Dictionary<string, string>();

        private static DataTable sTableSchema = null;

        public static void Initialize()
        {
            sKnownAbilities.Clear();
            sAbilityExpirations.Clear();
            sUpdateCustomColumns.Clear();

            
            // Known Ability Cache
            foreach (Int32 version in Config.Instance.Versions)
            {                
                DB.PopulateId(sKnownAbilities, "ability", version);

                 // Ability Description Cache
                using (SqlConnection conn = new SqlConnection(Config.Instance.ConnectionString))
                {
                    conn.Open();
                    SqlCommand cmd = conn.CreateCommand();
                    foreach (String locale in Config.Instance.SupportedLocales)
                    {
                        cmd.CommandText = "select ability_id, description_" + locale + " as description from ability_description where level = 1 and version_id = " + version.ToString();                        
                        using (SqlDataReader reader = cmd.ExecuteReader())
                        {
                            while (reader.Read())
                            {
                                AddAbilityDescription((int)reader["ability_id"], version, locale, (string)reader["description"]);
                            }
                        }
                    }
                }
            }

            DB.LoadSchema(ref sTableSchema, "ability");

            sUpdateCustomColumns.Add("careers", "ability.careers=ability.careers|tmp.careers");

            sUpdateQuery = "update ability set ";
            sUpdateQuery = sUpdateQuery + DB.GetUpdateSQL(sTableSchema, sUpdateIgnoredColumns, sUpdateCustomColumns);
            sUpdateQuery = sUpdateQuery + " FROM ability,#ability tmp" +
                " WHERE ability.id=tmp.id AND ability.version_id=tmp.version_id;";

        }

        public static void AddAbilityDescription(Int32 pId, Int32 pVersion, String pLocale, String pDescription)
        {
            CustomKey key = new CustomKey(pId, pVersion, pLocale);
            sAbilityDescriptionCache.Add(key, pDescription);                            
        }

        public static Boolean IsNewDescription(Int32 pId, Int32 pVersion, String pLocale, Int32 pLevel, String pDescription)
        {
            CustomKey key = new CustomKey(pId, pVersion, pLocale);
            if (!sAbilityDescriptionCache.ContainsKey(key))
            {
                // This ensures that we have a basis for comparison:
                if (pLevel == 1)
                {
                    AddAbilityDescription(pId, pVersion, pLocale, pDescription);
                    return true;
                }
                else
                {
                    return false;
                }
            }

            if (pDescription != sAbilityDescriptionCache[key])
            {
                return true;
            }

            return false;
        }

        public static void Save(Update pUpdate, SqlConnection pConn)
        {

            if (pUpdate.Abilities.Count == 0)
            {
                return;
            }

            DataRow dr = null;
            DataTable dtInsert = sTableSchema.Clone();
            DataTable dtUpdate = sTableSchema.Clone();

            dtInsert.BeginLoadData();
            dtUpdate.BeginLoadData();

            CustomKey key;
            String locale = pUpdate.Language.ToString();
            

            bool isNewRow = false;

            // Main Ability Info:
            foreach (Ability ability in pUpdate.Abilities)
            {
                if (!DB.ReadyForDB(sAbilityExpirations,
                                sConfirmations,
                                ability.Id,
                                Config.Instance.SpellExpiration,
                                Config.Instance.MinConfirmations,
                                pUpdate.UserId,
                                pUpdate.ClientVersion.Value,
                                (ELocale)pUpdate.Language,
                                pUpdate.IsTrustedUser))
                {
                    continue;
                }

                key = new CustomKey(pUpdate.ClientVersion.Value,
                                    ability.Id);
                isNewRow = !sKnownAbilities.Contains(key);

                if (isNewRow)
                {
                    dr = dtInsert.NewRow();
                }
                else
                {
                    dr = dtUpdate.NewRow();
                }

                dr["category"] = ability.Category;
                dr["id"] = ability.Id;
                dr["version_id"] = pUpdate.ClientVersion.Value;
                dr["name_" + locale] = ability.Name.Value;
                
                dr["icon_id"] = ability.Id;
                dr["type_id"] = ability.Type;

                dr["required_rank"] = ability.RequiredRank;
                dr["required_renown"] = ability.RequiredRenown;

                dr["package_id"] = ability.PackageId;
                dr["reuse_timer"] = ability.ReuseTimer;
                dr["stance_order"] = ability.StanceOrder;
                dr["target_type_id"] = ability.TacticType;
                dr["is_healing"] = ability.IsHealing;                
                dr["is_stats_buff"] = ability.IsStatsBuff;
                dr["is_offensive"] = ability.IsOffensive;
                dr["is_granted"] = ability.IsGranted;                
                dr["is_passive"] = ability.IsPassive;
                dr["is_defensive"] = ability.IsDefensive;
                dr["is_damaging"] = ability.IsDamaging;

                dr["is_debuff"] = ability.IsDebuff;
                dr["is_buff"] = ability.IsBuff;
                dr["is_hex"] = ability.IsHex;
                dr["is_curse"] = ability.IsCurse;
                dr["is_cripple"] = ability.IsCripple;
                dr["is_ailment"] = ability.IsAilment;
                dr["is_bolster"] = ability.IsBolster;
                dr["is_augmentation"] = ability.IsAugmentation;
                dr["is_blessing"] = ability.IsBlessing;
                dr["is_enchantment"] = ability.IsEnchantment;


                dr["requires_pet"] = ability.RequiresPet;

                dr["has_cost_per_second"] = ability.HasCostPerSecond;
                dr["specialization_id"] = ability.Specialization;
                dr["tactic_slots"] = ability.TacticSlots;
                dr["type_id"] = ability.Type;
                dr["icon_id"] = ability.Icon;
                dr["improvement_threshold"] = ability.ImprovementThreshold;
                dr["cost_morale"] = ability.CostMorale;
                dr["cost_action"] = ability.CostAction;
                dr["cost_renown"] = ability.CostRenown;
                dr["cost_coin"] = ability.CostCoin;
                dr["improvement_cap"] = ability.ImprovementCap;
                dr["morale_level"] = ability.MoraleLevel;
                dr["tactic_type_id"] = ability.TacticType;
                dr["cast_time"] = ability.CastTime;
                dr["cooldown"] = ability.Cooldown;
                dr["tier"] = ability.Tier;
                dr["max_purchase_count"] = ability.MaxPurchaseCount;

                dr["advance_id"] = ability.AdvanceId;
                dr["required_category_points"] = ability.RequiredCategoryPoints;
                dr["career_id"] = ability.CareerId;
                dr["careers"] = Utility.GetBitValue(ability.CareerId);
                
                dr["range_min"] = ability.RangeMin;
                dr["range_max"] = ability.RangeMax;

                if (ability.Dependencies.Count > 0)
                {
                    dr["dependency_points"] = ability.Dependencies[0].RequiredPoints;
                    dr["dependency_specialization_id"] = ability.Dependencies[0].SpecializationId;
                }
                else
                {
                    dr["dependency_points"] = 0;
                    dr["dependency_specialization_id"] = 0;
                }

                for (Byte i = 0; i < NUM_REQUIREMENTS; i++)
                {
                    if (i < ability.Requirements.Count)
                    {
                        dr["requirement" + i + "_" + locale] = ability.Requirements[i].Description;
                    }
                    else
                    {
                        dr["requirement" + i + "_" + locale] = String.Empty;
                    }
                }
                
                // Meta Data
                dr["updated_id"] = pUpdate.UserId;
                dr["posted_id"] = pUpdate.UserId;
                dr["posted"] = DateTime.UtcNow;
                dr["updated"] = DateTime.UtcNow;

                if (isNewRow)
                {
                    dtInsert.Rows.Add(dr);
                    sKnownAbilities.Add(key);
                }
                else
                {
                    dtUpdate.Rows.Add(dr);
                }

            }

            StringBuilder sql = new StringBuilder(1000);
            CustomKey descriptionKey;
            List<string> updateAbilityDescriptions = new List<string>();

            // Ability Descriptions
            foreach (Ability ability in pUpdate.Abilities)
            {
                if (ability.Descriptions.Count == 0)
                {
                    continue;
                }
                
                ability.Descriptions.Sort(delegate (AbilityDescription x, AbilityDescription y)
                {
                    return x.Level.CompareTo(y.Level);
                });

                foreach (AbilityDescription description in ability.Descriptions)
                {
                    if (!IsNewDescription(ability.Id, pUpdate.ClientVersion.Value, pUpdate.Language.ToString(), description.Level, description.Description))
                    {
                        continue;
                    }

                    descriptionKey = new CustomKey(ability.Id, description.Level, pUpdate.ClientVersion.Value);
                    if (DB.IsEntityExpired(sAbilityDescriptionExpirations, descriptionKey, Config.Instance.SpellExpiration, pUpdate.IsTrustedUser))
                    {
                        
                        sql.AppendFormat(sAbilityDescriptionQueryFormat,
                            locale,
                            description.Description.Replace("'", "''"),
                            ability.Id,
                            pUpdate.ClientVersion.Value,
                            description.Level);

                        if(!updateAbilityDescriptions.Contains(ability.Id.ToString()))
                        {
                            updateAbilityDescriptions.Add(ability.Id.ToString());
                        }
                    }
                }

            }

            if (updateAbilityDescriptions.Count > 0)
            {
                sql.Append("update ability set updated = GetUTCDate() where id in(" + String.Join(",", updateAbilityDescriptions.ToArray()) + ") and version_id = " + pUpdate.ClientVersion.Value + ";");
            }

            SqlCommand cmd = pConn.CreateCommand();

            if (sql.Length > 0)
            {
                cmd.CommandText = sql.ToString();
                cmd.ExecuteNonQuery();
            }

            // New Items
            if (dtInsert.Rows.Count > 0)
            {

                using (SqlBulkCopy bulk = new SqlBulkCopy(pConn,
                                                        SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                        null))
                {
                    bulk.BatchSize = 5000;
                    bulk.DestinationTableName = "ability";
                    bulk.WriteToServer(dtInsert);
                    bulk.Close();
                }
                
            }

            // Ability Updates
            if (dtUpdate.Rows.Count > 0)
            {                
                cmd.CommandText = "SELECT TOP 0 * INTO #ability FROM ability;";
                cmd.ExecuteNonQuery();

                using (SqlBulkCopy bulk = new SqlBulkCopy(pConn,
                                                     SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                     null))
                {
                    bulk.BatchSize = 5000;
                    bulk.DestinationTableName = "#ability";
                    bulk.WriteToServer(dtUpdate);
                    bulk.Close();
                }
                
                cmd.CommandText = String.Format(sUpdateQuery, locale);
                cmd.ExecuteNonQuery();
                cmd.CommandText = "DROP TABLE #ability;";
                cmd.ExecuteNonQuery();
            }
        }

    }

}