﻿using Curse;
using Curse.WAR;
using System;
using System.Data.SqlClient;
using System.Collections.Generic;
using System.Data;
using System.Linq;
namespace WARDataService
{
    public static partial class DBGuilds
    {

        private static Dictionary<CustomKey, DateTime> sExpiration =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());

        private static String sUpdateQuery = null;
        private static String sInsertQuery = null;

        private static String[] sUpdateIgnoredColumns = { "id", "name", "server_id", "posted", "posted_id" };
        private static String[] sIgnoreInsertColumns = {"id"};

        private static Dictionary<String, String> sUpdateCustomColumns = new Dictionary<string, string>();

        private static String sTempTableCreation = "SELECT TOP 0 * INTO #guild FROM guild;ALTER TABLE #guild DROP COLUMN id;ALTER TABLE #guild ADD id int;";
        private static DataTable sInsertGuildSchema = null;
        private static DataTable sGuildSchema = null;
        private static DataTable sGuildRosterSchema = null;
        private static DataTable sGuildNewsSchema = null;

        private static Dictionary<CustomKey, Int32> sGuildNameToId = new Dictionary<CustomKey, Int32>(new CustomKey.CustomKeyComparer());

        public static void Initialize()
        {
            sUpdateCustomColumns.Clear();
            sGuildNameToId.Clear();

            DB.LoadSchema(ref sInsertGuildSchema, "guild");
            DB.LoadSchema(ref sGuildSchema, "#guild", sTempTableCreation);
            DB.LoadSchema(ref sGuildRosterSchema, "guild_roster_entry");
            DB.LoadSchema(ref sGuildNewsSchema, "guild_news_entry");

            PopulateGuildNameToID();
            sInsertQuery = DB.GetInsertSQL(sInsertGuildSchema, sIgnoreInsertColumns);

            sUpdateQuery = "update guild set ";
            sUpdateQuery = sUpdateQuery + DB.GetUpdateSQL(sGuildSchema, sUpdateIgnoredColumns, sUpdateCustomColumns);
            sUpdateQuery = sUpdateQuery + " FROM guild,#guild tmp" +
                " WHERE guild.id=tmp.id;";

        }

        public static void AddGuild(String pGuildName, Int32 pServerId, Int32 pId)
        {
            CustomKey key = new CustomKey(pGuildName, pServerId);
            sGuildNameToId.Add(key, pId);
        }

        public static CustomKey GetGuildKey(String pGuildName, Int32 pServerId)
        {
            return new CustomKey(pGuildName.ToLower(), pServerId);
        }

        public static Int32 GetGuildId(String pGuildName, Int32 pServerId)
        {
            CustomKey key = GetGuildKey(pGuildName, pServerId);
            Int32 id;
            sGuildNameToId.TryGetValue(key, out id);
            return id;
        }

        private static void PopulateGuildNameToID()
        {
            String query = "select id, server_id, name from guild with(nolock)";

            Int32 guildId;
            Int32 serverId;
            Int32 count = 0;
            String guildName;            
            using (SqlConnection conn = new SqlConnection(Config.Instance.ConnectionString))
            {
                conn.Open();

                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = query;
                using (SqlDataReader dr = cmd.ExecuteReader())
                {
                    while (dr.Read())
                    {
                        guildId = dr.GetInt32(0);
                        serverId = dr.GetInt32(1);
                        guildName = dr.GetString(2);
                        sGuildNameToId.Add( GetGuildKey(guildName, serverId), guildId);
                        ++count;
                    }
                }
            }
            Logger.Log(ELogLevel.Debug,
                       null,
                       "Populated {0} names from table guild.",
                       count);
        }

        public static void Save(Update pUpdate, SqlConnection pConn)
        {

            if (pUpdate.Guilds.Count == 0)
            {
                return;
            }

            DataRow dr = null;            
            DataTable dtUpdate = sGuildSchema.Clone();
            DataTable dtInsert = sGuildSchema.Clone();
            DataTable dtRosterInsert = sGuildRosterSchema.Clone();
            DataTable dtNewsInsert = sGuildNewsSchema.Clone();            

            dtUpdate.BeginLoadData();
            dtRosterInsert.BeginLoadData();
            dtNewsInsert.BeginLoadData();

            CustomKey key;
            String locale = pUpdate.Language.ToString();         
            SqlCommand cmd = pConn.CreateCommand();
                                
            bool isNewRow = false;
            Int32 guildId = 0;
            Int32 serverId = DBServers.GetServerId(pUpdate.ServerName);
            if (serverId == 0)
            {
                Logger.Log(ELogLevel.Debug, pUpdate.UserHost,
                       "Unable to retrieve server ID: {0}", pUpdate.ServerName);
                return;
            }

            foreach (Guild guild in pUpdate.Guilds)
            {
                if (guild.Name.Value.Length == 0)
                {
                    continue;
                }
                
                guildId = DBGuilds.GetGuildId(guild.Name.Value, serverId);
                
                isNewRow = guildId <= 0;
                if (isNewRow)
                {
                    dr = dtInsert.NewRow();
                }
                else
                {
                    CustomKey exp = new CustomKey(guildId);
                    if (!DB.IsEntityExpired(sExpiration, exp, Config.Instance.GuildExpiration, pUpdate.IsTrustedUser))
                    {
                        continue;
                    }

                    dr = dtUpdate.NewRow();
                    dr["id"] = guildId;
                }

                dr["name"] = guild.Name.Value;
                dr["server_id"] = serverId;
                // Add 6 hours, from Eastern Time Zone for UTC:
                dr["created"] = DateTime.Parse(guild.CreationDate).AddHours(6);

                dr["details"] = guild.Details;
                dr["summary"] = guild.Summary;
                dr["website"] = guild.WebSite;
                dr["email"] = guild.Email;

                dr["rank"] = guild.Rank;
                dr["renown"] = guild.Renown;
                dr["xp_current"] = guild.XpCurrent;
                dr["xp_needed"] = guild.XpNeeded;
                dr["playstyle"] = guild.PlayStyle;
                dr["banners_captured"] = guild.BannersCaptured;
                dr["banners_lost"] = guild.BannersLost;
                dr["member_count"] = guild.Roster.Count;

                // Meta Data
                dr["updated_id"] = pUpdate.UserId;
                dr["posted_id"] = pUpdate.UserId;
                dr["posted"] = DateTime.UtcNow;
                dr["updated"] = DateTime.UtcNow;

                if (isNewRow)
                {
                    cmd.CommandText = DB.GetFilledInsertSQL(sInsertQuery, sInsertGuildSchema, dr, sIgnoreInsertColumns, locale);
                    try
                    {
                        guildId = (Int32)cmd.ExecuteScalar();
                    }
                    catch
                    {
                        Logger.Log(ELogLevel.Debug, pUpdate.UserHost, "Unable to insert new guild: {0} on server: {1}", guild.Name.Value, pUpdate.ServerName);                        
                        return;
                    }
                    
                    AddGuild(guild.Name.Value, serverId, guildId);
                }
                else
                {
                    dtUpdate.Rows.Add(dr);
                }


                List<string> processedPlayers = new List<string>();
                foreach (GuildRosterEntry entry in guild.Roster) 
                {
                    if (processedPlayers.Contains(entry.Name.Value))
                    {
                        continue;
                    }
                    DataRow rosterRow = dtRosterInsert.NewRow();
                    Int32 playerId = DBPlayers.GetPlayerId(entry.Name.Value, serverId);
                    rosterRow["guild_id"] = guildId;
                    rosterRow["player_name"] = entry.Name;
                    rosterRow["player_id"] = playerId;
                    rosterRow["status_title"] = entry.StatusTitle;
                    rosterRow["status_number"] = entry.StatusNumber;
                    rosterRow["rank"] = entry.Rank;
                    rosterRow["career_id"] = entry.Career;
                    rosterRow["last_login"] = entry.LastLogin;
                    processedPlayers.Add(entry.Name.Value);
                    dtRosterInsert.Rows.Add(rosterRow);
                }

                foreach (GuildNewsEntry entry in guild.News)
                {

                    DataRow newsRow = dtNewsInsert.NewRow();
                    newsRow["guild_id"] = guildId;
                    newsRow["display_order"] = entry.Order;
                    newsRow["type"] = entry.Type;
                    newsRow["text"] = entry.Text;
                    dtNewsInsert.Rows.Add(newsRow);
                }

                if (dtNewsInsert.Rows.Count > 0)
                {
                    cmd.CommandText = "delete from guild_news_entry where guild_id = " + guildId;
                    cmd.ExecuteNonQuery();
                }
                
                if (dtRosterInsert.Rows.Count > 0)
                {
                    cmd.CommandText = "delete from guild_roster_entry where guild_id = " + guildId;
                    cmd.ExecuteNonQuery();
                }
                
            }

            if (dtRosterInsert.Rows.Count > 0)
            {                                             
                using (SqlBulkCopy bulk = new SqlBulkCopy(pConn,
                                                        SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                        null))
                {
                    bulk.BatchSize = 5000;
                    bulk.DestinationTableName = "guild_roster_entry";
                    bulk.WriteToServer(dtRosterInsert);
                    bulk.Close();
                }
            }
                

            if (dtNewsInsert.Rows.Count > 0)
            {

                using (SqlBulkCopy bulk = new SqlBulkCopy(pConn,
                                                         SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                         null))
                {
                    bulk.BatchSize = 5000;
                    bulk.DestinationTableName = "guild_news_entry";
                    bulk.WriteToServer(dtNewsInsert);
                    bulk.Close();
                }

            }

            if (dtUpdate.Rows.Count > 0)
            {
                cmd.CommandText = sTempTableCreation;
                cmd.ExecuteNonQuery();
                
                using (SqlBulkCopy bulk = new SqlBulkCopy(pConn,
                                         SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                         null))
                {
                    bulk.BatchSize = 5000;
                    bulk.DestinationTableName = "#guild";

                    bulk.WriteToServer(dtUpdate);
                    bulk.Close();
                }
                
                cmd.CommandText = sUpdateQuery;
                cmd.ExecuteNonQuery();
                cmd.CommandText = "DROP TABLE #guild;";
                cmd.ExecuteNonQuery();
            }      
    }

    }
}
